package server.controller;

import java.io.File;
import java.io.IOException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;

import nl.tudelft.labracore.lib.security.user.Person;

import org.gitlab4j.api.GitLabApi;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.FileSystemResource;
import org.springframework.http.ContentDisposition;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.CrossOrigin;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import server.entity.Access;
import server.entity.CourseEdition;
import server.entity.StudentGroup;
import server.model.ApiKeyDTO;
import server.model.GeitStatisticsRequestDTO;
import server.model.UserCourseInfoDTO;
import server.security.GitBullSecurity;
import server.security.HmacValidator;
import server.service.*;
import server.util.FancyPrint;

/**
 * Controller that manages the connection with the frontend, allows users to access their data in the system
 * and manage their API keys.
 */
@RestController
@RequestMapping("/api/frontend")
@CrossOrigin("*")
public class FrontendController {

	@Value("${gitbull.frontend.api-key}")
	private String apiKey;

	@Value("${gitbull.frontend.api-secret}")
	private String apiSecret;

	@Autowired
	private HttpServletRequest request;

	@Autowired
	private AccessService accessService;

	@Autowired
	private CourseEditionService courseEditionService;

	@Autowired
	private GroupService groupService;

	@Autowired
	private GitBullSecurity gitBullSecurity;

	@Autowired
	private InstanceMappingService instanceMappingService;

	@Autowired
	private AccountService accountService;

	/**
	 * POST endpoint to get current authenticated user's username
	 *
	 * @param  secret User secret
	 * @return        Respective username if a mapping from the secret to the username exists, otherwise
	 *                UNAUTHORIZED
	 */
	@PostMapping("/whoami")
	public ResponseEntity<String> whoami(@RequestBody String secret) {
		String actualSecret = gitBullSecurity.decodeSecretString(secret);
		Person p = gitBullSecurity
				.getAuthenticatedUser(actualSecret);

		if (p == null) {
			return ResponseEntity.status(401).build();
		} else {
			String res = "From GitBull security via secret (proper): ";
			return ResponseEntity.ok(res + p.getDisplayName());
		}
	}

	/**
	 * GET endpoint to check the connection to the backend server
	 *
	 * @return Confirmation message
	 */
	@GetMapping("/checkConnection")
	public ResponseEntity<String> checkConnection() {
		return ResponseEntity.ok("Hello from GitBull!");
	}

	/**
	 * POST endpoint to download a statistics file.
	 *
	 * @param  dto GeitStatisticsRequestDTO containing user and file information
	 * @return     FileSystemResource object representing statistics file
	 */
	@PostMapping("/download")
	public ResponseEntity<FileSystemResource> getFile(@RequestBody GeitStatisticsRequestDTO dto) {
		String currentUser = gitBullSecurity.getCurrentUsername(dto.getSecret());
		GitLabApi gitLabApi = instanceMappingService.getInstance(currentUser);
		if (gitLabApi == null)
			return ResponseEntity.badRequest().build();

		String repoPath = dto.getRepoPath();
		Integer week = dto.getWeek();
		if (week < 1 || week > 10) {
			return ResponseEntity.badRequest().build();
		}

		String filename = repoPath.replaceAll("/", "_") + "_" + week + ".html";

		//TODO: check if user is allowed to download stats

		FancyPrint.println("statistics", "Frontend requested download of " + filename);

		HttpHeaders headers = new HttpHeaders();
		//headers.setContentType(MediaType.TEXT_HTML);
		headers.setContentDisposition(ContentDisposition.attachment().filename("file.html").build());

		return ResponseEntity.status(200).headers(headers)
				.body(new FileSystemResource(new File("statistics/results/" + filename)));
	}

	/**
	 * POST endpoint to upload a file.
	 *
	 * @param  file file to be uploaded
	 * @return      String representing upload success.
	 */
	@PostMapping("/uploadFile")
	public ResponseEntity<String> handleFileUpload(@RequestParam("file") MultipartFile file) {
		String filePath = request.getServletContext().getRealPath("/uploads/");
		try {
			file.transferTo(new File(filePath));
		} catch (IOException e) {
			return ResponseEntity.internalServerError().body(e.getMessage());
		}
		return ResponseEntity.ok("File uploaded");
	}

	/**
	 * POST endpoint to perform the handshake with the frontend server. Response OK if key is validated,
	 * otherwise 401 or 500.
	 *
	 * @param  hash                     hash representing secret
	 * @throws NoSuchAlgorithmException if the decryption algorithm is invalid
	 * @throws InvalidKeyException      if the key is invalid
	 */
	@PostMapping("/handshake")
	public ResponseEntity<Void> handshakeFrontend(@RequestParam String hash)
			throws NoSuchAlgorithmException, InvalidKeyException {
		if (HmacValidator.HashIsValid(apiSecret, apiKey.getBytes(), hash)) {
			System.out.println("Frontend server validated.");
			return ResponseEntity.ok().build();
		} else {
			System.out.println("Could not validate frontend server.");
			return ResponseEntity.status(401).build();
		}
	}

	/**
	 * POST endpoint to get whether a user is logged in or not through his secret. Checks whether the secret
	 * is in the system at the time.
	 *
	 * @param  secret user identification secret
	 * @return        whether the user is logged in or not
	 */
	@PostMapping("/isLoggedIn")
	public ResponseEntity<Boolean> isLoggedIn(@RequestBody String secret) {
		return ResponseEntity
				.ok(gitBullSecurity.getAuthenticatedUser(gitBullSecurity.decodeSecretString(secret)) != null);
	}

	/**
	 * POST endpoint to get the role of a user through his secret.
	 *
	 * @param  secret user identification secret
	 * @return        the role of the user
	 */

	@PostMapping("/getRole")
	public ResponseEntity<String> getRole(@RequestBody String secret) {
		return ResponseEntity
				.ok(gitBullSecurity.getAuthenticatedUser(gitBullSecurity.decodeSecretString(secret))
						.getDefaultRole().toString());
	}

	/**
	 * Deletes a specific course edition from the database, and the user's access to it.
	 *
	 * @param  courseName The name of the course for which the edition should be deleted.
	 * @param  edition    The name of the specific edition to be deleted.
	 * @return            ResponseEntity containing the deleted CourseEdition if found, or ResponseEntity with
	 *                    NOT_FOUND status if the edition does not exist.
	 */
	@PostMapping("/deleteEdition/{courseName}/{edition}")
	public ResponseEntity deleteEdition(@PathVariable String courseName, @PathVariable String edition,
			@RequestBody String secret) {
		CourseEdition deletedEdition = courseEditionService.deleteCourseEdition(courseName, edition);

		if (deletedEdition == null) {
			return ResponseEntity.status(HttpStatus.OK).body(null);
		}

		String actualSecret = gitBullSecurity.decodeSecretString(secret);
		String netId = gitBullSecurity.getCurrentUsername(actualSecret);

		List<Access> userAccesses = accessService.fetchAllAccessesOfAccount(netId);

		Access accessDeleted = userAccesses.stream()
				.filter(x -> x.getCourseEditionID() == deletedEdition.getCourseEditionID()).toList().get(0);

		accessService.deleteAccess(accessDeleted.getAccessID());

		return ResponseEntity.status(HttpStatus.OK).body(deletedEdition);
	}

	/**
	 * POST endpoint to get the list of courses a user has access to.
	 *
	 * @param  secret user identification secret
	 * @return        the list of courses the user has access to
	 */
	@PostMapping("/coursesList")
	public ResponseEntity<List<UserCourseInfoDTO>> getCourses(@RequestBody String secret) {
		String actualSecret = gitBullSecurity.decodeSecretString(secret);
		String netId = gitBullSecurity.getCurrentUsername(actualSecret);

		if (netId == null)
			return ResponseEntity.status(HttpStatus.UNAUTHORIZED).build();

		//Fetching all accesses
		List<Access> userAccesses = accessService.fetchAllAccessesOfAccount(netId);

		//Fetching all courses where we have an access
		List<Long> courseIds = userAccesses.stream().map(x -> x.getCourseEditionID())
				.collect(Collectors.toList());
		List<CourseEdition> courses = courseEditionService.fetchAllCoursesOfListOfIds(courseIds);

		//Fetching all groups for user
		List<Long> groupIds = userAccesses.stream().map(x -> x.getGroupID()).collect(Collectors.toList());
		List<StudentGroup> groups = groupService.fetchAllGroupsFromId(groupIds);
		List<String> groupPaths = groups.stream().map(x -> x.getGroupPath()).collect(Collectors.toList());

		List<UserCourseInfoDTO> res = new ArrayList<>();

		if (userAccesses.size() == courses.size()) {
			for (int i = 0; i < userAccesses.size(); i++) {
				UserCourseInfoDTO uci = new UserCourseInfoDTO();
				uci.setCourseName(courses.get(i).getCourseName());
				uci.setCourseEdition(courses.get(i).getEdition());
				uci.setCoursePath(courses.get(i).getCoursePath());
				int role = userAccesses.get(i).getRole();
				switch (role) {
					case 1 -> {
						uci.setRole("STUDENT");
						uci.setGroupPath(groupPaths.get(i));
					}
					case 2 -> {
						uci.setRole("TA");
					}
					case 3 -> {
						uci.setRole("TEACHER");
					}
				}

				res.add(uci);
			}
		} else {
			return ResponseEntity.status(HttpStatus.NOT_FOUND).build();
		}

		return ResponseEntity.ok(res);
	}

	/**
	 * POST endpoint to add a GitLab API key to an existing user.
	 *
	 * @param apiKeyDTO ApiKeyDTO containing user secret and key to be added.
	 */
	@PostMapping("/addKey")
	public ResponseEntity<Void> addKey(@RequestBody ApiKeyDTO apiKeyDTO) {
		String currentUser = gitBullSecurity.getCurrentUsername(apiKeyDTO.getSecret());

		if (apiKeyDTO.getApiKey().isEmpty())
			return new ResponseEntity<>(HttpStatus.BAD_REQUEST);

		if (currentUser == null || currentUser.isEmpty())
			return new ResponseEntity<>(HttpStatus.UNAUTHORIZED);

		accountService.addAPIKey(currentUser, apiKeyDTO.getApiKey());
		instanceMappingService.addInstance(currentUser, apiKeyDTO.getApiKey());

		return ResponseEntity.ok().build();
	}

	/**
	 * POST endpoint to remove the GitLab API key of an existing user.
	 *
	 * @param apiKeyDTO ApiKeyDTO containing user secret
	 */
	@PostMapping("/removeKey")
	public ResponseEntity<String> removeKey(@RequestBody ApiKeyDTO apiKeyDTO) {
		String currentUser = gitBullSecurity.getCurrentUsername(apiKeyDTO.getSecret());

		if (currentUser == null || currentUser.isEmpty())
			return ResponseEntity.status(HttpStatus.UNAUTHORIZED).build();

		GitLabApi gitLabApi = instanceMappingService.getInstance(currentUser);

		if (gitLabApi == null)
			return ResponseEntity.ok("API key not found.");

		accountService.removeAPIKey(currentUser);
		instanceMappingService.removeInstance(currentUser);

		return ResponseEntity.ok("API key successfully removed.");
	}

	/**
	 * POST endpoint to verify whether there is a valid mapping from the user to a GitLab API key
	 *
	 * @param  apiKeyDTO ApiKeyDTO containing user secret
	 * @return           true if the user has a key in the system, false otherwise
	 */
	@PostMapping("/keyExists")
	public ResponseEntity<Boolean> keyExists(@RequestBody ApiKeyDTO apiKeyDTO) {
		String currentUser = gitBullSecurity.getCurrentUsername(apiKeyDTO.getSecret());

		System.out.println(currentUser);

		if (currentUser == null || currentUser.isEmpty())
			return ResponseEntity.status(HttpStatus.UNAUTHORIZED).build();

		// an instance mapping may exist with an invalid key. that means the returned object is not null
		GitLabApi gitLabApi = instanceMappingService.getInstance(currentUser);
		if (gitLabApi == null) {
			return ResponseEntity.ok(false);
		}

		try {
			// check if the current key is valid
			String currentUsername = gitLabApi.getUserApi().getCurrentUser().getUsername();
			System.out.println(currentUsername);

			return ResponseEntity.ok(true);

		} catch (Exception e) {
			return ResponseEntity.ok(false);
		}
	}

}
