package server.controller;

import static java.util.Collections.emptyList;
import static server.logging.LogType.*;

import java.time.LocalDate;
import java.time.Month;
import java.time.Year;
import java.util.*;
import java.util.concurrent.CompletableFuture;

import nl.tudelft.labracore.api.dto.PersonDetailsDTO;
import nl.tudelft.labracore.api.dto.PersonSummaryDTO;

import org.gitlab4j.api.GitLabApi;
import org.gitlab4j.api.GitLabApiException;
import org.gitlab4j.api.models.*;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.data.util.Pair;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import server.csvconverter.CSVService;
import server.entity.Access;
import server.entity.Account;
import server.entity.CourseEdition;
import server.entity.CourseName;
import server.entity.StudentGroup;
import server.labraService.LabraConnectionService;
import server.logging.LogWriter;
import server.model.*;
import server.security.GitBullSecurity;
import server.service.*;
import server.service.GroupService;
import server.statistics.StatisticsRunner;

/**
 * Controller class that handles operations involving the GitLab API required for setting up courses,
 * editions, importing students and staff.
 */
@Controller
@RequestMapping("/api/gitlab")
@CrossOrigin("*")
@ComponentScan
public class GitLabController {

	private final GroupAPIService groupAPIService;

	private final ProjectAPIService projectAPIService;

	private final UserAPIService userAPIService;

	private final LabraConnectionService labraConnectionService;

	private final AccountService accountDBService;

	private final GroupService groupDBService;

	private final InstanceMappingService instanceMappingService;

	private final CourseEditionService courseEditionDBService;

	private final CourseNameService courseNameDBService;

	private final AccessService accessDBService;

	private final CSVService csvService;

	private final LogWriter writer;

	private StatisticsRunner statisticsRunner;

	private final GitBullSecurity gitBullSecurity;

	public GitLabController(GroupAPIService groupAPIService,
			GroupService groupDBService, LabraConnectionService labraConnectionService,
			UserAPIService userAPIService, ProjectAPIService projectAPIService,
			InstanceMappingService instanceMappingService, CourseEditionService courseEditionDBService,
			CourseNameService courseNameDBService, AccessService accessDBService, LogWriter writer,
			AccountService accountDBService, CSVService csvService, GitBullSecurity gitBullSecurity,
			StatisticsRunner statisticsRunner) {
		this.labraConnectionService = labraConnectionService;
		this.projectAPIService = projectAPIService;
		this.accountDBService = accountDBService;

		this.groupAPIService = groupAPIService;
		this.userAPIService = userAPIService;

		this.courseEditionDBService = courseEditionDBService;
		this.courseNameDBService = courseNameDBService;

		this.accessDBService = accessDBService;
		this.groupDBService = groupDBService;

		this.instanceMappingService = instanceMappingService;

		this.csvService = csvService;
		this.writer = writer;
		this.gitBullSecurity = gitBullSecurity;
		this.statisticsRunner = statisticsRunner;
	}

	/**
	 * POST Endpoint to get all course names from LabraCORE. Used for frontend.
	 *
	 * @return HttpStatus.OK and the list of all course names in LabraCORE.
	 */
	@PostMapping("/getCourseNames")
	public ResponseEntity<List<String>> getCourseNames() {
		List<String> courseNames = labraConnectionService.getAllCourseNames();
		return ResponseEntity.status(HttpStatus.OK).body(courseNames);
	}

	/**
	 * POST Endpoint to create a course The name of the group should be the course name For frontend use
	 *
	 * @param  dto Group Request Model to extract information about the group
	 * @return     HttpStatus.CREATED and the group that has been created if any exception occurs return
	 *             HttpStatus from GitLab API
	 */
	@PostMapping("/createCourse")
	public ResponseEntity<Group> createCourse(@RequestBody GroupRequestDTO dto) {
		String currentUser = gitBullSecurity.getCurrentUsername(dto.getSecret());
		GitLabApi gitLabApi = instanceMappingService.getInstance(currentUser);

		if (gitLabApi == null)
			return ResponseEntity.status(HttpStatus.FORBIDDEN).build();

		Group group;
		try {
			String name = dto.getName().replaceAll("#", "")
					.replaceAll("&", "and").replaceAll("/", "");
			String path = name
					.replaceAll("\\s", "").toLowerCase();

			group = groupAPIService.createGroup(gitLabApi, name, dto.getDescription(),
					path,
					dto.isMembershipLock(), dto.getVisibility());
			writer.writeActionLog(INFO, COURSE_CREATED, dto.getName(),
					currentUser);

			CourseName cn = new CourseName();
			cn.setCoursePath(group.getPath());
			cn.setCourseName(group.getName());
			courseNameDBService.saveCourseName(cn);

			return ResponseEntity.status(HttpStatus.CREATED).body(group);
		} catch (GitLabApiException e) {
			writer.writeActionLog(ERROR, COURSE_CREATED, dto.getName(),
					currentUser, e.getMessage());
			return ResponseEntity.status(e.getHttpStatus()).build();
		}
	}

	/**
	 * POST Endpoint to create a course edition based on the path of the edition The path of a course should
	 * be the name of the course
	 *
	 * @param  groupRequestModel Group Request Model to extract information about the group
	 * @return                   HttpStatus.CREATED and the group that has been created if any exception
	 *                           occurs return HttpStatus from GitLab API
	 */
	@PostMapping("/createCourseEdition")
	public ResponseEntity<Group> createEdition(@RequestHeader String coursePath,
			@RequestBody GroupRequestDTO groupRequestModel) {
		String currentUser = gitBullSecurity.getCurrentUsername(groupRequestModel.getSecret());
		GitLabApi gitLabApi = instanceMappingService.getInstance(currentUser);

		if (gitLabApi == null) {
			return ResponseEntity.status(HttpStatus.UNAUTHORIZED).body(null);
		}

		// Get the current date
		LocalDate currentDate = LocalDate.now();

		// Get the current month
		Month currentMonth = currentDate.getMonth();

		// the course edition name should be the course years edition
		Year thisYear = Year.of(currentDate.getYear());

		// Get the month value (1-12)
		int monthValue = currentMonth.getValue();
		String courseEditionName = "";
		if (monthValue >= 7) {
			courseEditionName = String.valueOf(thisYear.getValue()) + "-"
					+ String.valueOf(thisYear.getValue() + 1);
		} else {
			courseEditionName = String.valueOf(thisYear.getValue() - 1) + "-"
					+ String.valueOf(thisYear.getValue());
		}

		try {
			// create edition

			Long groupId = groupAPIService.getGroupId(gitLabApi, coursePath);

			Group subgroup = groupAPIService.createSubgroup(gitLabApi,
					courseEditionName,
					groupRequestModel.getDescription(),
					courseEditionName,
					groupRequestModel.isMembershipLock(),
					groupRequestModel.getVisibility(),
					groupId);

			// log success

			writer.writeActionLog(INFO, EDITION_CREATED, coursePath + " " + courseEditionName,
					currentUser);

			// commit to database

			String courseName = courseNameDBService.fetchNameForCoursePath(coursePath);

			CourseEdition courseEdition = new CourseEdition();
			courseEdition.setCourseName(courseName);
			courseEdition.setEdition(courseEditionName);
			courseEdition.setCoursePath(coursePath + "/" + courseEditionName);

			long idNewSave = courseEditionDBService.saveCourseEdition(courseEdition).getCourseEditionID();

			Access newAccess = new Access();
			newAccess.setRole(3);
			newAccess.setAccountID(currentUser);
			newAccess.setCourseEditionID(idNewSave);
			accessDBService.saveAccess(newAccess);

			return ResponseEntity.status(HttpStatus.CREATED).body(subgroup);
		} catch (GitLabApiException e) {
			// log failure
			writer.writeActionLog(ERROR, EDITION_CREATED, coursePath + " " + courseEditionName,
					currentUser);
			return ResponseEntity.status(e.getHttpStatus()).build();
		}
	}

	/**
	 * Set up GitLab accounts for the students if needed. Add them to groups then make projects for said
	 * groups. If forkPath is null then the repositories will be empty with just a basic initial commit made
	 * to them. If forkPath is not null then the projects will be forked from an already-existing project.
	 * Included in the entity of the response is an error log containing all the failed actions contained
	 * within the method. In the case of Response.OK, the errors contained within the log are not considered
	 * critical and are to be presented to the user in the frontend. In the case of critical errors, the
	 * return is Response.serverError(). The staff members are added to the course edition before students are
	 * added to their groups
	 *
	 * @param  dto SetUpRequestDTO containing information about the following: coursePath, courseName,
	 *             courseEdition, individualRepoForkPath, forkPath, groupVisibility, repoVisibility,
	 *             defaultBranchProtectionLevel, spIndex. spIndex represents the column in the import file
	 *             containing the group names.
	 *
	 * @return     response OK if the whole set up is finished
	 */
	@Async
	@PostMapping(value = "/setUp", produces = { MediaType.APPLICATION_JSON_VALUE })
	@ResponseBody
	public CompletableFuture<ResponseEntity<List<String>>> setUp(@RequestBody SetUpRequestDTO dto) {
		Boolean enableIndividualRepositories = dto.getEnableIndividualRepositories();
		if (enableIndividualRepositories == null)
			enableIndividualRepositories = false;

		String coursePath = dto.getCoursePath();
		String courseName = dto.getCourseName();
		String courseEdition = dto.getCourseEdition();
		String individualRepoForkPath = dto.getIndividualRepoForkPath();
		String forkPath = dto.getForkPath();
		String groupVisibility = dto.getGroupVisibility();
		Boolean repoVisibility = dto.getRepoVisibility();
		GroupParams.DefaultBranchProtectionLevel defaultBranchProtectionLevel = dto
				.getDefaultBranchProtectionLevel();
		int weeksOfStatistics = dto.getWeeksOfStatistics() == null ? 0 : dto.getWeeksOfStatistics();

		String currentUser = gitBullSecurity.getCurrentUsername(dto.getSecret());
		GitLabApi gitLabApi = instanceMappingService.getInstance(currentUser);

		if (gitLabApi == null) {
			return CompletableFuture.completedFuture(
					ResponseEntity.status(HttpStatus.UNAUTHORIZED).body(List.of("You are not authorized.")));
		}

		// Get course edition from GitLab.
		Long parentId;
		String editionPath = "";
		if (coursePath.isEmpty()) {
			// Generate the editionPath using the name of the course and the edition
			editionPath = courseName.replaceAll("#", "").replaceAll("&", "and").replaceAll("/", "")
					.replaceAll("\\s", "").toLowerCase() +
					"/" + courseEdition.replaceAll("\\s", "").toLowerCase();
		} else {
			editionPath = coursePath;
		}

		try {
			parentId = groupAPIService.getGroupId(gitLabApi, editionPath);
		} catch (GitLabApiException e) {
			// critical error
			return CompletableFuture
					.completedFuture(ResponseEntity
							.internalServerError()
							.body(List.of("Couldn't get parent group! Error message: " + e.getMessage())));
		}

		// Parse the set-up file given to extract students and groups.
		Map<StudentGroup, Set<Account>> studentGroupSetMap;

		try {
			//String formattedContent = csvService.format(dto.getFileContent());
			studentGroupSetMap = csvService.extractGroups(dto.getFileContent(), dto.getSpIndex());
		} catch (Exception e) {
			// critical error
			return CompletableFuture
					.completedFuture(ResponseEntity.internalServerError()
							.body(List.of("Couldn't parse file! Error message: " + e.getMessage())));
		}

		long courseEditionId = courseEditionDBService.getCourseEditionFromCoursePath(editionPath)
				.getCourseEditionID();

		// Create student accounts if needed.
		Pair<List<Account>, List<User>> pair = userAPIService.verifyAccounts(gitLabApi, studentGroupSetMap);

		//saveEntries(studentGroupSetMap, courseName, courseEdition);

		// Get list of all existing and created student accounts associated with the given import file.
		List<User> students = pair.getSecond();

		// Create list to log errors then log all user creation errors.
		List<String> errorLog = new ArrayList<>();
		List<Account> studentErrors = pair.getFirst();

		for (Account account : studentErrors) {
			errorLog.add("Account creation error for: " + account.getNetId());
		}

		// Create groups and assign students to them
		for (Map.Entry<StudentGroup, Set<Account>> entry : studentGroupSetMap.entrySet()) {
			StudentGroup group = entry.getKey();

			StudentGroup groupInDB;
			// Create group
			Group subgroup;
			try {
				subgroup = groupAPIService.createSubgroupUpdated(gitLabApi, group.getName(), "",
						group.getName().replaceAll("\\s", "").toLowerCase(), groupVisibility,
						parentId, defaultBranchProtectionLevel);
				writer.writeActionLog(INFO, GROUP_CREATED,
						group.getName() + " in " + courseName + " " + courseEdition,
						currentUser);

				StudentGroup newGroup = StudentGroup.builder().groupPath(subgroup.getFullPath())
						.courseID(courseEditionId).name(group.getName()).build();
				groupInDB = groupDBService.saveGroup(newGroup);
			} catch (Exception e) {
				// Cannot add the students to the group if it doesn't exist
				writer.writeActionLog(ERROR, GROUP_CREATED,
						group.getName() + " in " + courseName + " " + courseEdition, currentUser,
						e.getMessage());
				errorLog.add(writer.generateMessage(ERROR, GROUP_CREATED,
						group.getName() + " in " + courseName + " " + courseEdition, currentUser)
						+ e.getMessage());

				continue;
			}

			// Create project. If a project is provided to fork from then they will be forked.
			// Otherwise, it is created and an initial commit is made.

			Project project = initProject(gitLabApi, subgroup.getId(),
					group.getName(), forkPath, repoVisibility, currentUser);

			if (project == null) {
				// Project couldn't be created so skip over the rest
				errorLog.add("Cannot create repository for: [" + group.getName() + "]");

				continue;
			}

			// if statistics are enabled, add to stats tracker
			if (weeksOfStatistics > 0) {
				statisticsRunner.addRepository(project.getPathWithNamespace(), weeksOfStatistics);
			}

			Set<Account> accounts = entry.getValue();
			for (Account account : accounts) {

				// Commented below is an easier to read alternative which sadly results in additional API calls
				// User user = userAPIService.getUserByEmail(gitLabApi, account.getEmail().toLowerCase());

				Optional<User> optionalStudent = students.stream().filter(
						x -> x.getEmail().equalsIgnoreCase(account.getEmail()))
						.findFirst();

				if (optionalStudent.isPresent()) {
					User student = optionalStudent.get();

					if (enableIndividualRepositories) {
						String individualGroupName = student.getUsername() + " - individual";
						try {
							Group individualGroup = groupAPIService.createSubgroupUpdated(gitLabApi,
									individualGroupName, "", individualGroupName.replaceAll("\\s", ""),
									groupVisibility, parentId, defaultBranchProtectionLevel);

							Project individualRepo = generateIndividualRepo(gitLabApi,
									individualGroup.getId(), student,
									individualRepoForkPath, repoVisibility, currentUser);
							if (individualRepo == null) {
								errorLog.add(
										"Couldn't create individual repository for " + student.getUsername());
							}

							MemberRequestDTO memberRequestDTO = new MemberRequestDTO(student.getId(),
									individualGroup.getId(), 30,
									null, dto.getSecret());
							groupAPIService
									.addGroupMember(gitLabApi, memberRequestDTO);

						} catch (GitLabApiException e) {
							errorLog.add(
									"Couldn't create individual repository for " + student.getUsername() + " "
											+ e.getMessage());
						}

					}

					try {

						Access newAccess = Access.builder().groupID(groupInDB.getGroupID())
								.accountID(account.getNetId()).role(1).courseEditionID(courseEditionId)
								.build();
						accessDBService.saveAccess(newAccess);

						MemberRequestDTO memberRequestDTO = new MemberRequestDTO(student.getId(),
								subgroup.getId(), 30,
								null, dto.getSecret());
						groupAPIService
								.addGroupMember(gitLabApi, memberRequestDTO);
					} catch (GitLabApiException e) {
						errorLog.add("Couldn't add user [" + student.getUsername() + "] to group ["
								+ group.getName() + "]" + e.getMessage());
					}

				} else {
					errorLog.add("Couldn't retrieve user [" + account.getNetId() + "]");
				}
			}
		}

		return CompletableFuture.completedFuture(ResponseEntity.ok(errorLog));
	}

	/**
	 * Generates an individual repo for a student within a course edition.
	 *
	 * @param  gitLabApi              GitLabAPI instance
	 * @param  namespaceID            Namespace ID of the individual group designed to the student within the
	 *                                course edition
	 * @param  user                   Student account on GitLab instance
	 * @param  individualRepoForkPath Project path to fork for the student's repository (optional)
	 * @param  repoVisibility         Visibility of the student's individual repository
	 * @param  currentUser            Username of user initiating the action (teacher username)
	 * @return                        Created Project instance for the student
	 */
	public Project generateIndividualRepo(GitLabApi gitLabApi, Long namespaceID, User user,
			String individualRepoForkPath,
			Boolean repoVisibility, String currentUser) {
		String individualRepoName = user.getUsername() + " - individual";
		Project project = initProject(gitLabApi, namespaceID,
				individualRepoName, individualRepoForkPath, repoVisibility, currentUser);
		return project;
	}

	/**
	 * Creates a project in the given namespace and makes an initial commit to it. If a forkPath is given,
	 * then the project will be forked from an already-existing project.
	 *
	 * @param  namespaceID    Namespace where the project will be created.
	 * @param  name           Name of the project.
	 * @param  forkPath       Path of the project to fork from. (Optional)
	 * @param  repoVisibility The visibility of the repository
	 * @return                HttpsStatus.OK
	 */
	Project initProject(GitLabApi gitLabApi, Long namespaceID, String name, String forkPath,
			Boolean repoVisibility, String currentUser) {
		Project project;
		String projName = name.replaceAll("\\s", "").toLowerCase();
		ProjectRequestDTO projectRequestDTO = new ProjectRequestDTO(projName,
				name, "", namespaceID, true, true, true, true, repoVisibility);

		if (forkPath == null || forkPath.isEmpty()) {
			// try to create project
			try {
				project = projectAPIService.createProjectInNamespace(gitLabApi, projectRequestDTO);
			} catch (GitLabApiException e) {
				writer.writeActionLog(ERROR, PROJECT_CREATED, name, currentUser,
						e.getMessage());
				return null;
			}

			// try to make initial commit
			try {
				projectAPIService.createInitialCommit(gitLabApi, project.getId(),
						name);
			} catch (GitLabApiException e) {
				// This means that we couldn't create an initial commit but the project was still created.
				writer.writeActionLog(ERROR, INITIAL_COMMIT, name, currentUser,
						e.getMessage());
			}
		} else {
			// try to fork project
			try {
				Project forkOrigin = projectAPIService.getProject(gitLabApi, forkPath);
				project = projectAPIService.forkProject(gitLabApi, forkOrigin, namespaceID);
			} catch (GitLabApiException e) {
				writer.writeActionLog(ERROR, PROJECT_FORKED, name, currentUser,
						e.getMessage());
				return null;
			}
			writer.writeActionLog(INFO, PROJECT_FORKED, name, currentUser);
		}

		return project;
	}

	/**
	 * Returns all the staff associated with a given course edition.
	 *
	 * @param  dto StaffRequestDTO containing secret, course name and course edition
	 * @return     List containing information about the staff
	 */
	@PostMapping("/getCourseStaff")
	public ResponseEntity<List<PersonSummaryDTO>> getCourseStaff(@RequestBody StaffRequestDTO dto) {
		String currentUser = gitBullSecurity.getCurrentUsername(dto.getSecret());

		if (currentUser == null)
			return ResponseEntity.status(HttpStatus.UNAUTHORIZED).build();

		List<PersonSummaryDTO> summaries = labraConnectionService
				.getTeacherAssistantsForCourseEdition(dto.getCourseName(), dto.getCourseEdition());
		return ResponseEntity.ok(summaries);
	}

	/**
	 * Adds staff members to multiple groups.
	 *
	 * @param  dto StaffRequestDTO object containing a HashMap representing the mapping of staff email
	 *             addresses to a list of group names. Each staff member will be added to the corresponding
	 *             groups. Also includes edition path as a String, used to construct the group paths.
	 * @return     A ResponseEntity containing a list of error log messages as strings. Empty list indicates
	 *             successful execution.
	 */
	@PostMapping("/addStaffToGroups")
	public ResponseEntity<List<String>> addStaffToGroups(
			@RequestBody StaffRequestDTO dto) {
		String currentUsername = gitBullSecurity.getCurrentUsername(dto.getSecret());
		GitLabApi gitLabApi = instanceMappingService.getInstance(currentUsername);

		if (gitLabApi == null)
			return ResponseEntity.status(HttpStatus.UNAUTHORIZED).build();

		String editionPath = dto.getEditionPath();

		long courseEditionId = courseEditionDBService.getCourseEditionFromCoursePath(editionPath)
				.getCourseEditionID();

		List<String> errorLog = new ArrayList<>();

		Map<String, String> staffGroupsMap = dto.getStaffGroupsMap();

		for (Map.Entry<String, String> entry : staffGroupsMap.entrySet()) {
			String staffMail = entry.getKey();

			User user;
			try {
				user = userAPIService.getUserByUsername(gitLabApi, staffMail.toLowerCase().split("@")[0]);

			} catch (GitLabApiException e) {
				// couldn't get the user, so we can't add them to the groups
				errorLog.add("Couldn't get user associated with mail [" + staffMail + "]. Reason: "
						+ e.getMessage());

				continue;
			}

			if (user == null) {
				errorLog.add("Couldn't add staff with mail [" + staffMail + "] to group path [" + editionPath
						+ "]. Reason: User could not be found.");

				continue;
			}

			if (entry.getValue().isEmpty()) {
				continue;
			}

			List<String> groupsToAddTo = List.of(entry.getValue().split(","));

			for (String groupName : groupsToAddTo) {
				// TODO : make sure editionPath is right
				String groupPath = editionPath + "/" + groupName;
				try {
					groupAPIService.addGroupMember(gitLabApi, groupPath, user.getId(), 40, null);

					writer.writeMemberActionLog(INFO, USER_ADDED_TO_GROUP,
							staffMail.toLowerCase().split("@")[0], editionPath, currentUsername);

					long groupId = groupDBService.getGroupFromGroupPath(groupPath.toLowerCase()).getGroupID();
					Access newAccess = Access.builder().groupID(groupId).role(3).accountID(user.getUsername())
							.courseEditionID(courseEditionId).build();
					accessDBService.saveAccess(newAccess);
				} catch (GitLabApiException e) {
					writer.writeMemberActionLog(ERROR, USER_ADDED_TO_GROUP,
							staffMail.toLowerCase().split("@")[0], editionPath, currentUsername,
							e.getMessage());

					errorLog.add(
							"Couldn't add staff with mail [" + staffMail + "] to group path [" + groupPath
									+ "]. Reason: " + e.getMessage());
				}
			}

		}
		return ResponseEntity.ok(errorLog);
	}

	/**
	 * Adds the staff of a course (TAs, head TAs and lecturers) to the course edition The emails of the staff
	 * members are extracted from the LabraConnectionService TAs access level is set to Maintainer (40) Head
	 * TAs access level is set to Maintainer (40) Lecturers access level is set to Owner (50).
	 *
	 * @param  staffRequestDTO DTO containing information for the request
	 * @return                 A list of strings detailing the errors encountered when importing the staff.
	 */
	@PostMapping("/importStaff")
	public ResponseEntity<List<String>> importStaff(@RequestBody StaffRequestDTO staffRequestDTO) {
		String currentUsername = gitBullSecurity.getCurrentUsername(staffRequestDTO.getSecret());
		GitLabApi gitLabApi = instanceMappingService.getInstance(currentUsername);

		if (gitLabApi == null)
			return ResponseEntity.status(HttpStatus.UNAUTHORIZED).build();

		String courseName = staffRequestDTO.getCourseName();
		String courseEdition = staffRequestDTO.getCourseEdition();
		String editionPath = staffRequestDTO.getEditionPath();

		long courseEditionId = courseEditionDBService.getCourseEditionFromCoursePath(editionPath)
				.getCourseEditionID();

		List<List<String>> staffList = labraConnectionService.getStaffEmailsForCourseEdition(courseName,
				courseEdition);

		if (staffList.isEmpty()) {
			System.out.println("TA list for " + staffRequestDTO.getEditionPath() + "is empty");
			return ResponseEntity.internalServerError().body(
					List.of("Retrieving staff emails failed or there is no staff associated with the course."));
		}

		// check if staff email lists are null or empty

		List<String> tas = !staffList.get(0).isEmpty() ? staffList.get(0) : emptyList();
		List<String> headTas = !staffList.get(1).isEmpty() ? staffList.get(1) : emptyList();
		List<String> lecturers = !staffList.get(2).isEmpty() ? staffList.get(2) : emptyList();

		List<String> errorLog = new ArrayList<>();

		for (String taEmail : tas) {
			User user;
			try {
				user = userAPIService.getUserByUsername(gitLabApi, taEmail.toLowerCase().split("@")[0]);
				// Access level 40 -> Maintainer

				if (user == null) {
					errorLog.add(
							"Couldn't add staff with mail [" + taEmail.toLowerCase() + "] to group path ["
									+ editionPath
									+ "]. Reason: User could not be found.");
					continue;
				}

				groupAPIService.addGroupMember(gitLabApi, editionPath, user.getId(), 40, null);

				writer.writeMemberActionLog(INFO, USER_ADDED_TO_GROUP, taEmail.toLowerCase().split("@")[0],
						editionPath, currentUsername);

				Access newAccess = Access.builder().role(2).courseEditionID(courseEditionId)
						.accountID(user.getUsername()).build();
				accessDBService.saveAccess(newAccess);

			} catch (GitLabApiException e) {
				writer.writeMemberActionLog(ERROR, USER_ADDED_TO_GROUP, taEmail.toLowerCase().split("@")[0],
						editionPath, currentUsername, e.getMessage());

				errorLog.add(
						"Couldn't add TA with email [" + taEmail + "], status code: " + e.getHttpStatus()
								+ ", error-message: " + e.getMessage());
			}
		}

		for (String headTaEmail : headTas) {
			User user;
			try {
				user = userAPIService.getUserByUsername(gitLabApi, headTaEmail.toLowerCase().split("@")[0]);
				if (user == null) {
					errorLog.add(
							"Couldn't add staff with mail [" + headTaEmail.toLowerCase() + "] to group path ["
									+ editionPath
									+ "]. Reason: User could not be found.");
					continue;
				}
				// Access level 40 -> Maintainer
				MemberRequestDTO dto = new MemberRequestDTO(user.getId(), editionPath, 50, null,
						staffRequestDTO.getSecret());
				groupAPIService.addGroupMember(gitLabApi, dto);

				writer.writeMemberActionLog(INFO, USER_ADDED_TO_GROUP,
						headTaEmail.toLowerCase().split("@")[0], editionPath, currentUsername);

				Access newAccess = Access.builder().role(2).courseEditionID(courseEditionId)
						.accountID(user.getUsername()).build();
				accessDBService.saveAccess(newAccess);

			} catch (GitLabApiException e) {
				writer.writeMemberActionLog(ERROR, USER_ADDED_TO_GROUP,
						headTaEmail.toLowerCase().split("@")[0], editionPath, currentUsername,
						e.getMessage());

				errorLog.add("Couldn't add Head TA with email [" + headTaEmail + "], status code: "
						+ e.getHttpStatus() + ", error-message: " + e.getMessage());
			}
		}

		for (String lecturerEmail : lecturers) {
			User user;
			try {
				user = userAPIService.getUserByUsername(gitLabApi, lecturerEmail.toLowerCase().split("@")[0]);
				if (user == null) {
					errorLog.add(
							"Couldn't add staff with mail [" + lecturerEmail.toLowerCase()
									+ "] to group path [" + editionPath
									+ "]. Reason: User could not be found.");
					continue;
				}
				// Access level 50 -> Owner
				MemberRequestDTO dto = new MemberRequestDTO(user.getId(), editionPath, 50, null,
						staffRequestDTO.getSecret());
				groupAPIService.addGroupMember(gitLabApi, dto);

				writer.writeMemberActionLog(INFO, USER_ADDED_TO_GROUP,
						lecturerEmail.toLowerCase().split("@")[0], editionPath, currentUsername);

				Access newAccess = Access.builder().role(3).courseEditionID(courseEditionId)
						.accountID(user.getUsername()).build();
				accessDBService.saveAccess(newAccess);
			} catch (GitLabApiException e) {
				writer.writeMemberActionLog(ERROR, USER_ADDED_TO_GROUP,
						lecturerEmail.toLowerCase().split("@")[0], editionPath, currentUsername,
						e.getMessage());

				errorLog.add("Couldn't add Lecturer with email [" + lecturerEmail + "], status code: "
						+ e.getHttpStatus() + ", error-message: " + e.getMessage());
			}

		}

		return ResponseEntity.ok(errorLog);
	}

	/**
	 * Adds a teacher to a course.
	 *
	 * @param  teacherCourseRequestDTO TeacherCourseRequestDTO containing information about the course,
	 *                                 teacher and user secret
	 * @return                         Member object representing Teacher within course group if successful,
	 *                                 otherwise error message.
	 */
	@PostMapping("/addTeacherToCourse")
	public ResponseEntity<?> addTeacherToCourse(
			@RequestBody TeacherCourseRequestDTO teacherCourseRequestDTO) {
		// email + course path
		String currentUsername = gitBullSecurity.getCurrentUsername(teacherCourseRequestDTO.getSecret());
		GitLabApi gitLabApi = instanceMappingService.getInstance(currentUsername);
		if (gitLabApi == null)
			return ResponseEntity.status(HttpStatus.UNAUTHORIZED).build();

		try {
			User user = userAPIService.getUserByEmail(gitLabApi,
					teacherCourseRequestDTO.getUsername().toLowerCase());

			if (user == null) {
				PersonDetailsDTO personDetailsDTO = labraConnectionService
						.getPerson(teacherCourseRequestDTO.getUsername());
				user = userAPIService.createUser(gitLabApi, teacherCourseRequestDTO.getUsername(),
						personDetailsDTO.getUsername(), personDetailsDTO.getDisplayName());
			}

			Member member = groupAPIService.addGroupMember(gitLabApi, teacherCourseRequestDTO.getCoursePath(),
					user.getId(), 50, null);

			writer.writeMemberActionLog(INFO, USER_ADDED_TO_GROUP, teacherCourseRequestDTO.getUsername(),
					teacherCourseRequestDTO.getCoursePath(), currentUsername);

			return ResponseEntity.ok(member);

		} catch (GitLabApiException e) {
			writer.writeMemberActionLog(ERROR, USER_ADDED_TO_GROUP, teacherCourseRequestDTO.getUsername(),
					teacherCourseRequestDTO.getCoursePath(), currentUsername, e.getMessage());

			return ResponseEntity.status(e.getHttpStatus()).body(e.getMessage());
		}
	}

}
