package server.controller;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.gitlab4j.api.GitLabApi;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.util.Pair;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import server.entity.Statistics;
import server.model.GroupStatisticsRequestDTO;
import server.security.GitBullSecurity;
import server.service.InstanceMappingService;
import server.service.statisticsService.GroupStatisticsService;
import server.service.statisticsService.IndividualStatisticsService;

/**
 * Controller class that handles the GitLab statistics API endpoints.
 */
@Controller
@RequestMapping("/api/statistics")
@CrossOrigin("*")
public class GitLabStatisticsController {

	private final GroupStatisticsService groupStatisticsService;

	private final IndividualStatisticsService individualStatisticsService;

	private final InstanceMappingService instanceMappingService;

	private final GitBullSecurity gitBullSecurity;

	/**
	 * Constructs a new GitLabStatisticsController with the specified dependencies.
	 *
	 * @param groupStatisticsService      The service for retrieving group statistics.
	 * @param individualStatisticsService The service for retrieving individual statistics.
	 * @param instanceMappingService      The instance mapping service for retrieving GitLabAPI instances.
	 * @param gitBullSecurity             The security service for retrieving the current user from the given
	 *                                    secret.
	 *
	 */
	@Autowired
	public GitLabStatisticsController(GroupStatisticsService groupStatisticsService,
			IndividualStatisticsService individualStatisticsService,
			InstanceMappingService instanceMappingService,
			GitBullSecurity gitBullSecurity) {
		this.groupStatisticsService = groupStatisticsService;
		this.individualStatisticsService = individualStatisticsService;
		this.instanceMappingService = instanceMappingService;
		this.gitBullSecurity = gitBullSecurity;
	}

	/**
	 * Retrieves the statistics of all users in a given subgroup
	 *
	 * @param  dto GroupStatisticsRequestDTO containing groupPath, the path to the subgroup, and repoPath, the
	 *             path to the project of the subgroup
	 *
	 * @return     List containing the statistics per user.
	 */
	@PostMapping("/getAllStatsForGroup")
	public ResponseEntity<List<Statistics>> getStatsForGroup(@RequestBody GroupStatisticsRequestDTO dto) {
		String currentUser = gitBullSecurity.getCurrentUsername(dto.getSecret());
		GitLabApi gitLabApi = instanceMappingService.getInstance(currentUser);
		if (gitLabApi == null)
			return ResponseEntity.status(HttpStatus.UNAUTHORIZED).build();

		String groupPath = dto.getGroupPath();
		String repoPath = dto.getRepoPath();

		List<String> names = groupStatisticsService.getParticipantsNames(gitLabApi,
				groupPath);

		List<Statistics> statsForUsers = new ArrayList<>();

		Map<String, Pair<Integer, Integer>> userLinesInfo = groupStatisticsService.getUsersLinesOfCodeCount(
				gitLabApi,
				groupPath);

		for (String userName : names) {
			Integer comments = individualStatisticsService.getUserCommentsCountOnMerge(gitLabApi,
					repoPath, userName);
			Integer locAdded = userLinesInfo.getOrDefault(userName, Pair.of(0, 0)).getFirst();
			Integer locDeleted = userLinesInfo.getOrDefault(userName, Pair.of(0, 0)).getSecond();
			Integer issues = individualStatisticsService.getUserIssuesCount(gitLabApi,
					repoPath, userName);
			Integer mergeRequests = individualStatisticsService.getUserMergeRequestsCount(gitLabApi,
					repoPath, userName);
			Integer commits = individualStatisticsService.getUserCommitCount(gitLabApi,
					repoPath, userName);

			Statistics stats = new Statistics(userName, commits.toString(),
					issues.toString(), mergeRequests.toString(), locAdded.toString(), locDeleted.toString(),
					comments.toString());
			statsForUsers.add(stats);
		}
		return ResponseEntity.ok(statsForUsers);

	}

	/**
	 * Retrieves the statistics of all users in the given groups.
	 *
	 * @param  dto GroupStatisticsRequestDTO containing groupPaths, a list with paths to the subgroups, and
	 *             repoPaths, a list with paths to the projects of the subgroups
	 *
	 * @return     Matrix containing the statistics per user, per group.
	 */
	@PostMapping("/getAllStatsForMultipleGroups")
	public ResponseEntity<List<List<Statistics>>> getStatsForAllGroups(
			@RequestBody GroupStatisticsRequestDTO dto) {
		String currentUser = gitBullSecurity.getCurrentUsername(dto.getSecret());
		GitLabApi gitLabApi = instanceMappingService.getInstance(currentUser);

		if (gitLabApi == null)
			return ResponseEntity.status(HttpStatus.UNAUTHORIZED).build();
		List<List<Statistics>> groupStats = new ArrayList<>();

		List<String> groupPaths = dto.getGroupPaths();
		List<String> repoPaths = dto.getRepoPaths();

		for (int i = 0; i < groupPaths.size(); i++) {
			String groupPath = groupPaths.get(i);

			String repoPath = repoPaths.get(i);

			if (repoPath.endsWith("-INDIVIDUAL")) {
				continue;
			}

			List<String> names = groupStatisticsService.getParticipantsNames(gitLabApi,
					groupPath);

			if (names == null)
				continue;

			List<Statistics> statsForUsers = new ArrayList<>();

			Map<String, Pair<Integer, Integer>> userLinesInfo = groupStatisticsService
					.getUsersLinesOfCodeCount(
							gitLabApi,
							repoPath);

			if (userLinesInfo == null)
				continue;

			for (String userName : names) {
				Integer comments = individualStatisticsService.getUserCommentsCountOnMerge(gitLabApi,
						repoPath, userName);
				Integer locAdded = userLinesInfo.getOrDefault(userName, Pair.of(0, 0)).getFirst();
				Integer locDeleted = userLinesInfo.getOrDefault(userName, Pair.of(0, 0)).getSecond();
				Integer issues = individualStatisticsService.getUserIssuesCount(gitLabApi,
						repoPath, userName);
				Integer mergeRequests = individualStatisticsService.getUserMergeRequestsCount(gitLabApi,
						repoPath, userName);
				Integer commits = individualStatisticsService.getUserCommitCount(gitLabApi,
						repoPath, userName);

				Statistics stats = new Statistics(userName, commits.toString(),
						issues.toString(), mergeRequests.toString(), locAdded.toString(),
						locDeleted.toString(),
						comments.toString());

				statsForUsers.add(stats);
			}

			groupStats.add(statsForUsers);
		}
		return ResponseEntity.ok(groupStats);
	}

}
