package server.controller;

import static server.logging.LogType.*;
import static server.service.util.RoleDecoder.roleDecoder;

import java.util.ArrayList;
import java.util.List;

import org.gitlab4j.api.GitLabApi;
import org.gitlab4j.api.GitLabApiException;
import org.gitlab4j.api.models.Group;
import org.gitlab4j.api.models.GroupParams;
import org.gitlab4j.api.models.Member;
import org.gitlab4j.api.models.User;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import server.entity.Access;
import server.labraService.LabraConnectionService;
import server.logging.LogWriter;
import server.model.*;
import server.security.GitBullSecurity;
import server.service.*;

/**
 * Controller class that allows the modification and management of already-existing groups on GitLab.
 */
@RestController
@RequestMapping("/api/group")
@CrossOrigin("*")
@ComponentScan
public class GroupManagementController {
	private final GroupAPIService groupAPIService;

	private final UserAPIService userAPIService;
	private LogWriter writer;

	private final InstanceMappingService instanceMappingService;

	private final CourseEditionService courseEditionDBService;

	private final CourseNameService courseNameDBService;

	private final AccessService accessDBService;

	private final GroupService groupDBService;

	private final GitBullSecurity gitBullSecurity;

	private final LabraConnectionService labraConnectionService;

	@Autowired
	public GroupManagementController(GroupAPIService groupAPIService,
			UserAPIService userAPIService,
			InstanceMappingService instanceMappingService, CourseEditionService courseEditionDBService,
			CourseNameService courseNameDBService, AccessService accessDBService, GroupService groupDBService,
			GitBullSecurity gitBullSecurity,
			LabraConnectionService labraConnectionService) {
		this.groupAPIService = groupAPIService;
		this.userAPIService = userAPIService;
		this.instanceMappingService = instanceMappingService;
		this.courseEditionDBService = courseEditionDBService;
		this.courseNameDBService = courseNameDBService;
		this.accessDBService = accessDBService;
		this.groupDBService = groupDBService;
		this.gitBullSecurity = gitBullSecurity;
		this.labraConnectionService = labraConnectionService;
		this.writer = new LogWriter();
	}

	/**
	 * Get all subgroups within a group
	 *
	 * @param  groupRequestModel DTO containing secret and group path
	 * @return                   the list of subgroups within the groups
	 */
	@PostMapping("/getSubgroupPaths")
	public ResponseEntity<List<String>> getSubgroupPaths(@RequestBody GroupRequestDTO groupRequestModel) {
		String currentUsername = gitBullSecurity.getCurrentUsername(groupRequestModel.getSecret());
		GitLabApi gitLabApi = instanceMappingService.getInstance(currentUsername);
		if (gitLabApi == null)
			return ResponseEntity.status(HttpStatus.UNAUTHORIZED).build();

		try {
			List<String> groups = groupAPIService.getSubgroupPaths(gitLabApi, groupRequestModel.getPath());

			List<String> projectGroups = groups.stream().filter(x -> !x.endsWith("-individual")).toList();

			return ResponseEntity.status(HttpStatus.CREATED).body(projectGroups);
		} catch (GitLabApiException e) {
			return ResponseEntity.status(e.getHttpStatus()).build();
		}
	}

	/**
	 * Removes project with all resources (issues, merge requests, etc).
	 *
	 * @param  groupRequestDTO Group Request Model to extract information about the group
	 * @return                 a Response indicating the success or failure of the operation
	 */
	@DeleteMapping("/deleteGroup")
	public ResponseEntity<String> deleteGroup(@RequestBody GroupRequestDTO groupRequestDTO) {
		String currentUsername = gitBullSecurity.getCurrentUsername(groupRequestDTO.getSecret());
		GitLabApi gitLabApi = instanceMappingService.getInstance(currentUsername);
		String currentUser = "";
		if (gitLabApi == null)
			return ResponseEntity.status(HttpStatus.UNAUTHORIZED).build();

		try {
			groupAPIService.deleteGroup(gitLabApi, groupRequestDTO.getPath());
			writer.writeActionLog(INFO, GROUP_DELETED, groupRequestDTO.getPath(), currentUser);
			return ResponseEntity.ok("Group " + groupRequestDTO.getName() + " deleted successfully");

		} catch (GitLabApiException e) {
			writer.writeActionLog(ERROR, GROUP_DELETED, groupRequestDTO.getPath(), currentUser);
			return ResponseEntity.status(e.getHttpStatus()).body(e.getMessage());
		}
	}

	/**
	 * POST Endpoint to add a member to a group based on their email The method extracts the course name,
	 * course edition and group name from the UserRequestDTO and generates based on this information the group
	 * path in which the user will be added
	 *
	 * @param  userRequestDTO User Request Model to extract information about the member and group
	 * @return                HttpStatus.CREATED and the member that has been added to the group if any
	 *                        exception occurs return HttpStatus from GitLab API
	 */
	@PostMapping("/addGroupMemberByEmail")
	public ResponseEntity<Member> addGroupMemberByEmail(@RequestBody UserRequestDTO userRequestDTO) {
		String currentUsername = gitBullSecurity.getCurrentUsername(userRequestDTO.getSecret());
		GitLabApi gitLabApi = instanceMappingService.getInstance(currentUsername);
		if (gitLabApi == null)
			return ResponseEntity.status(HttpStatus.UNAUTHORIZED).build();

		try {
			String path = "";
			boolean hasGroup = false;
			if (stringExists(userRequestDTO.getGroupPath())) {
				hasGroup = true;
				path = userRequestDTO.getCourseName().replaceAll("\\s", "").toLowerCase() +
						"/" + userRequestDTO.getCourseEdition().replaceAll("\\s", "").toLowerCase() +
						"/" + userRequestDTO.getGroupPath().replaceAll("\\s", "").toLowerCase();
			} else {
				path = userRequestDTO.getCourseName().replaceAll("\\s", "").toLowerCase() +
						"/" + userRequestDTO.getCourseEdition().replaceAll("\\s", "").toLowerCase();
			}

			User user = userAPIService.getUserByEmail(gitLabApi, userRequestDTO.getEmail().toLowerCase());
			Member member = groupAPIService.addGroupMember(gitLabApi, path,
					user.getId(), userRequestDTO.getAccessLevel(), userRequestDTO.getExpiresAt());

			int role = roleDecoder(userRequestDTO.getAccessLevel());

			if (hasGroup) {
				long groupId = groupDBService.getGroupFromGroupPath(path).getGroupID();

				String editionPath = userRequestDTO.getCourseName().replaceAll("\\s", "").toLowerCase() +
						"/" + userRequestDTO.getCourseEdition().replaceAll("\\s", "").toLowerCase();
				long editionId = courseEditionDBService.getCourseEditionFromCoursePath(editionPath)
						.getCourseEditionID();

				Access newAccess = Access.builder().groupID(groupId).courseEditionID(editionId).role(role)
						.accountID(user.getUsername()).build();
				accessDBService.saveAccess(newAccess);
			} else {
				long editionId = courseEditionDBService.getCourseEditionFromCoursePath(path)
						.getCourseEditionID();
				Access newAccess = Access.builder().courseEditionID(editionId).role(role)
						.accountID(user.getUsername()).build();
				accessDBService.saveAccess(newAccess);
			}

			return ResponseEntity.status(HttpStatus.OK).body(member);
		} catch (GitLabApiException e) {
			return ResponseEntity.status(e.getHttpStatus()).build();
		}
	}

	/**
	 * DELETE Endpoint to remove a member to a group based on their email The method extracts the course name,
	 * course edition and group name from the UserRequestDTO and generates based on this information the group
	 * path in which the user will be added
	 *
	 * @param  userRequestDTO User Request Model to extract information about the member and group
	 * @return                HttpStatus.CREATED and the member that has been added to the group if any
	 *                        exception occurs return HttpStatus from GitLab API
	 */
	@PostMapping("/removeGroupMemberByEmail")
	public ResponseEntity<User> removeGroupMemberByEmail(@RequestBody UserRequestDTO userRequestDTO) {
		String currentUsername = gitBullSecurity.getCurrentUsername(userRequestDTO.getSecret());
		GitLabApi gitLabApi = instanceMappingService.getInstance(currentUsername);
		if (gitLabApi == null)
			return ResponseEntity.status(HttpStatus.UNAUTHORIZED).build();

		try {
			String path = "";
			boolean hasGroup = false;
			if (stringExists(userRequestDTO.getGroupPath())) {
				hasGroup = true;
				path = userRequestDTO.getCourseName().replaceAll("\\s", "").toLowerCase() +
						"/" + userRequestDTO.getCourseEdition().replaceAll("\\s", "").toLowerCase() +
						"/" + userRequestDTO.getGroupPath().replaceAll("\\s", "").toLowerCase();
			} else {
				path = userRequestDTO.getCourseName().replaceAll("\\s", "").toLowerCase() +
						"/" + userRequestDTO.getCourseEdition().replaceAll("\\s", "").toLowerCase();
			}

			User user = userAPIService.getUserByEmail(gitLabApi, userRequestDTO.getEmail().toLowerCase());
			groupAPIService.removeGroupMember(gitLabApi, path,
					user.getId());

			if (hasGroup) {
				long groupId = groupDBService.getGroupFromGroupPath(path).getGroupID();

				String editionPath = userRequestDTO.getCourseName().replaceAll("\\s", "").toLowerCase() +
						"/" + userRequestDTO.getCourseEdition().replaceAll("\\s", "").toLowerCase();
				long editionId = courseEditionDBService.getCourseEditionFromCoursePath(editionPath)
						.getCourseEditionID();

				Access accessToDelete = accessDBService.getAccessFromUserEditionGroup(user.getUsername(),
						editionId, groupId);
				accessDBService.deleteAccess(accessToDelete.getAccessID());
			} else {
				long editionId = courseEditionDBService.getCourseEditionFromCoursePath(path)
						.getCourseEditionID();
				Access accessToDelete = accessDBService.getAccessFromUserEdition(user.getUsername(),
						editionId);
				accessDBService.deleteAccess(accessToDelete.getAccessID());
			}

			return ResponseEntity.status(HttpStatus.OK).body(user);
		} catch (GitLabApiException e) {
			return ResponseEntity.status(e.getHttpStatus()).build();
		}
	}

	/**
	 * PUT Endpoint to update a member's information from a group The main functionality for this method is to
	 * change the access level of a user
	 *
	 * The method extracts the course name, course edition and group name from the UserRequestDTO and
	 * generates based on this information the group path in which the user will be added
	 *
	 * @param  userRequestDTO User Request Model to extract information about the member and group
	 * @return                HttpStatus.CREATED and the member that has been added to the group if any
	 *                        exception occurs return HttpStatus from GitLab API
	 */
	@PutMapping("/updateGroupMemberByEmail")
	public ResponseEntity<Member> updateGroupMemberByEmail(@RequestBody UserRequestDTO userRequestDTO) {
		String currentUsername = gitBullSecurity.getCurrentUsername(userRequestDTO.getSecret());
		GitLabApi gitLabApi = instanceMappingService.getInstance(currentUsername);
		if (gitLabApi == null)
			return ResponseEntity.status(HttpStatus.UNAUTHORIZED).build();

		try {
			String path = "";
			boolean hasGroup = false;
			if (stringExists(userRequestDTO.getGroupPath())) {
				hasGroup = true;
				path = userRequestDTO.getCourseName().replaceAll("\\s", "").toLowerCase() +
						"/" + userRequestDTO.getCourseEdition().replaceAll("\\s", "").toLowerCase() +
						"/" + userRequestDTO.getGroupPath().replaceAll("\\s", "").toLowerCase();
			} else {
				path = userRequestDTO.getCourseName().replaceAll("\\s", "").toLowerCase() +
						"/" + userRequestDTO.getCourseEdition().replaceAll("\\s", "").toLowerCase();
			}

			User user = userAPIService.getUserByEmail(gitLabApi, userRequestDTO.getEmail().toLowerCase());
			Member member = groupAPIService.updateGroupMember(gitLabApi, path, user.getId(),
					userRequestDTO.getAccessLevel(), userRequestDTO.getExpiresAt());

			int role = roleDecoder(userRequestDTO.getAccessLevel());

			if (hasGroup) {
				long groupId = groupDBService.getGroupFromGroupPath(path).getGroupID();

				String editionPath = userRequestDTO.getCourseName().replaceAll("\\s", "").toLowerCase() +
						"/" + userRequestDTO.getCourseEdition().replaceAll("\\s", "").toLowerCase();
				long editionId = courseEditionDBService.getCourseEditionFromCoursePath(editionPath)
						.getCourseEditionID();

				Access accessToUpdate = accessDBService.getAccessFromUserEditionGroup(user.getUsername(),
						editionId, groupId);

				accessToUpdate.setRole(role);

				accessDBService.saveAccess(accessToUpdate);
			} else {
				long editionId = courseEditionDBService.getCourseEditionFromCoursePath(path)
						.getCourseEditionID();

				Access accessToUpdate = accessDBService.getAccessFromUserEdition(user.getUsername(),
						editionId);

				accessToUpdate.setRole(role);

				accessDBService.saveAccess(accessToUpdate);
			}

			return ResponseEntity.status(HttpStatus.OK).body(member);
		} catch (GitLabApiException e) {
			return ResponseEntity.status(e.getHttpStatus()).build();
		}
	}

	/**
	 * Updates a project. Null values are not updated. Full path of the group must be provided.
	 *
	 * @param  groupRequestDTO the Group instance with the configuration for the updated group
	 *
	 * @return                 a Response containing the updated project info or an error message
	 */
	@PostMapping("/updateGroup")
	public ResponseEntity<?> updateGroup(@RequestBody GroupRequestDTO groupRequestDTO) {
		String currentUsername = gitBullSecurity.getCurrentUsername(groupRequestDTO.getSecret());
		GitLabApi gitLabApi = instanceMappingService.getInstance(currentUsername);
		if (gitLabApi == null)
			return ResponseEntity.status(HttpStatus.UNAUTHORIZED).build();

		try {
			GroupParams groupParams = new GroupParams()
					.withName(groupRequestDTO.getName())
					.withDescription(groupRequestDTO.getDescription())
					.withPath(groupRequestDTO.getPath())
					.withMembershipLock(groupRequestDTO.isMembershipLock())
					.withVisibility(groupRequestDTO.getVisibility())
					.withParentId(groupRequestDTO.getParentId());
			Group updatedGroup = groupAPIService.updateGroup(gitLabApi, groupRequestDTO.getPath(),
					groupParams);
			return ResponseEntity.ok(updatedGroup);
		} catch (GitLabApiException e) {
			return ResponseEntity.status(e.getHttpStatus()).body(e.getMessage());
		}
	}

	/**
	 * Updates a project and all of its immediate subprojects (so only direct descendants). Null values are
	 * not updated. Must provide full path.
	 *
	 * @param  groupRequestDTO the Group instance with the configuration for the updated group
	 *
	 * @return                 a Response containing the updated project info or an error message
	 */
	@PostMapping("/updateGroupAndSubgroups")
	public ResponseEntity<List<String>> updateGroupAndSubgroups(
			@RequestBody GroupRequestDTO groupRequestDTO) {
		String currentUsername = gitBullSecurity.getCurrentUsername(groupRequestDTO.getSecret());
		GitLabApi gitLabApi = instanceMappingService.getInstance(currentUsername);
		if (gitLabApi == null)
			return ResponseEntity.status(HttpStatus.UNAUTHORIZED).build();

		try {
			Group group = groupAPIService.getGroup(gitLabApi, groupRequestDTO.getPath());

			GroupParams groupParams = new GroupParams()
					.withName(groupRequestDTO.getName())
					.withDescription(groupRequestDTO.getDescription())
					.withMembershipLock(groupRequestDTO.isMembershipLock())
					.withVisibility(groupRequestDTO.getVisibility())
					.withDefaultBranchProtection(groupRequestDTO.getDefaultBranchProtectionLevel());

			groupAPIService.updateGroup(gitLabApi, group, groupParams);

			List<Group> subgroups = groupAPIService.getSubgroups(gitLabApi,
					group);

			List<String> errorLog = new ArrayList<>();

			for (Group subgroup : subgroups) {
				GroupParams updatedParams = new GroupParams().withName(subgroup.getName())
						.withDescription(groupRequestDTO.getDescription())
						.withPath(subgroup.getPath())
						.withParentId(subgroup.getParentId())
						.withDefaultBranchProtection(groupRequestDTO.getDefaultBranchProtectionLevel())
						.withVisibility(groupRequestDTO.getVisibility());

				try {
					groupAPIService.updateGroup(gitLabApi, subgroup, updatedParams);
				} catch (GitLabApiException e) {
					errorLog.add("Error updating group [" + subgroup.getName() + "]. Reason: "
							+ e.getHttpStatus() + " " + e.getMessage());
				}

			}

			return ResponseEntity.ok(errorLog);
		} catch (GitLabApiException e) {
			return ResponseEntity.status(e.getHttpStatus()).body(List.of(e.getMessage()));
		}
	}

	/**
	 * Add user in batches in a group
	 *
	 * @param batchRequestDTO DTO containing secret and the list of members to be removed from their current
	 *                        group
	 */
	@PostMapping("/addMembersInBatches")
	public ResponseEntity<List<Member>> addMembersInBatches(@RequestBody BatchRequestDTO batchRequestDTO) {
		String currentUsername = gitBullSecurity.getCurrentUsername(batchRequestDTO.getSecret());
		GitLabApi gitLabApi = instanceMappingService.getInstance(currentUsername);

		if (gitLabApi == null)
			return ResponseEntity.status(HttpStatus.UNAUTHORIZED).build();

		List<UserRequestDTO> users = batchRequestDTO.getUsers();

		List<Member> memberList = new ArrayList<>();
		for (UserRequestDTO userRequestDTO : users) {
			try {
				String path = "/" + userRequestDTO.getCourseName().replaceAll("\\s", "").toLowerCase() +
						"/" + userRequestDTO.getCourseEdition().replaceAll("\\s", "").toLowerCase() +
						"/" + userRequestDTO.getGroupPath().replaceAll("\\s", "").toLowerCase();
				User user = userAPIService.getUserByEmail(gitLabApi, userRequestDTO.getEmail().toLowerCase());
				Member member = groupAPIService.addGroupMember(gitLabApi, path,
						user.getId(), userRequestDTO.getAccessLevel(), userRequestDTO.getExpiresAt());
				memberList.add(member);
			} catch (GitLabApiException e) {
				return ResponseEntity.status(e.getHttpStatus()).build();
			}
		}
		return ResponseEntity.status(HttpStatus.OK).body(memberList);
	}

	/**
	 * Remove members in batches from their current group
	 *
	 * @param  batchRequestDTO DTO containing secret and the list of users to be removed from their current
	 *                         group
	 *
	 * @return                 ResponseEntity.OK if operations succeeds, otherwise ResponseEntity containing
	 *                         errorm essage
	 */
	@DeleteMapping("/removeMembersInBatches")
	public ResponseEntity<?> removeMembersInBatches(@RequestBody BatchRequestDTO batchRequestDTO) {
		String currentUsername = gitBullSecurity.getCurrentUsername(batchRequestDTO.getSecret());
		GitLabApi gitLabApi = instanceMappingService.getInstance(currentUsername);

		if (gitLabApi == null)
			return ResponseEntity.status(HttpStatus.UNAUTHORIZED).build();

		try {

			List<UserRequestDTO> users = batchRequestDTO.getUsers();

			for (UserRequestDTO userRequestDTO : users) {
				String path = "/" + userRequestDTO.getCourseName().replaceAll("\\s", "").toLowerCase() +
						"/" + userRequestDTO.getCourseEdition().replaceAll("\\s", "").toLowerCase() +
						"/" + userRequestDTO.getGroupPath().replaceAll("\\s", "").toLowerCase();
				User user = userAPIService.getUserByEmail(gitLabApi, userRequestDTO.getEmail().toLowerCase());
				groupAPIService.removeGroupMember(gitLabApi, path, user.getId());
			}

		} catch (GitLabApiException e) {
			return ResponseEntity.status(e.getHttpStatus()).body(e.getMessage());
		}
		return ResponseEntity.ok().build();
	}

	/**
	 * Helper method meant to help check if a string is null or empty.
	 *
	 * @param  string String to be checked
	 * @return        Whether the given string is null or empty
	 */
	public boolean stringExists(String string) {
		return string != null && !string.isBlank();
	}
}
