package server.csvconverter;

import java.io.IOException;
import java.io.StringReader;
import java.util.*;

import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVRecord;
import org.springframework.stereotype.Service;

import server.entity.Account;
import server.entity.StudentGroup;

/**
 * Service that handles and parses CSV files.
 */
@Service
public class CSVService {

	public CSVService() {
	}

	/**
	 * Takes a string that represents the contents of a string representing a CSV file, a number representing
	 * the index of the group name column and returns a map of groups and sets of associated student accounts.
	 *
	 * @param  fileContent        Contents of the CSV
	 * @param  projectHeaderIndex Index of the group name column
	 * @return                    Map of groups and sets of student accounts
	 * @throws IOException        If any parsing error occurs
	 */
	public Map<StudentGroup, Set<Account>> extractGroups(String fileContent, Integer projectHeaderIndex)
			throws IOException {
		List<List<String>> rows = parseContents(fileContent);

		Map<StudentGroup, Set<Account>> groupSetMap = convertToEntities(rows, projectHeaderIndex);
		return groupSetMap;
	}

	/**
	 * Formats a string to represent a valid CSV file.
	 *
	 * @param  fileContent
	 * @return
	 */
	public String format(String fileContent) {
		String formattedContents = fileContent.substring(fileContent.indexOf(":\"") + 2,
				fileContent.lastIndexOf("\""));
		formattedContents = formattedContents.replace("\\r", "");
		formattedContents = formattedContents.replace("\\n", System.lineSeparator());
		return formattedContents;
	}

	/**
	 * Parses the contents of a CSV file in a string format.
	 *
	 * @param  contents    Contents of the file in string format
	 * @return             A list of rows representing CSV entries.
	 * @throws IOException If the format is invalid or parsing encounters exceptions
	 */
	public List<List<String>> parseContents(String contents) throws IOException {
		// Validate input
		String validatedContents;
		if (!UTF8Validator.isValidUtf8(contents)) {
			validatedContents = UTF8Validator.replaceNonUtf8Characters(contents);
		} else {
			validatedContents = contents;
		}

		// Parse input as CSV records

		StringReader stringReader = new StringReader(validatedContents);
		Iterable<CSVRecord> csvRecords = CSVFormat.RFC4180.parse(stringReader);

		// Make CSV records into list of strings

		List<List<String>> rows = new ArrayList<>();

		for (CSVRecord record : csvRecords) {
			List<String> row = record.stream().toList();
			rows.add(row);
		}

		return rows;
	}

	/**
	 * Converts a list of rows to a map of groups and students. It parses each row, determines whether a
	 * student's name contains commas then makes a respective account, associates it with the group number
	 * determined by projectHeaderIndex then adds the user to the set pointed at by the group.
	 *
	 * @param  rows               Rows containing student and group information
	 * @param  projectHeaderIndex Number of the column containing the group name
	 * @return                    Map of groups containing sets of student accounts
	 * @throws IOException        If the input is invalid (no header)
	 */
	public Map<StudentGroup, Set<Account>> convertToEntities(List<List<String>> rows,
			Integer projectHeaderIndex) throws IOException {
		if (rows.get(0).isEmpty())
			throw new IOException("Header row is empty.");

		Map<StudentGroup, Set<Account>> studentGroupSetMap = new HashMap<>();

		int defaultRowSize = rows.get(0).size();

		for (int i = 1; i < rows.size(); i++) {
			List<String> row = rows.get(i);

			// Compute whether row is longer because of names containing commas

			int offset = row.size() != defaultRowSize ? row.size() - defaultRowSize : 0;

			// Get account from row

			Account account = getAccount(row, offset);
			if (account == null)
				continue;

			// Get group name from row

			String groupName = row.get(projectHeaderIndex + offset);
			if (groupName.isEmpty())
				continue;

			StudentGroup studentGroup = new StudentGroup(groupName);

			// Add account to already-existing group in map or create new one

			Set<Account> accounts;
			if (studentGroupSetMap.containsKey(studentGroup)) {
				accounts = studentGroupSetMap.get(studentGroup);

			} else {
				accounts = new HashSet<>();
			}
			accounts.add(account);
			studentGroupSetMap.put(studentGroup, accounts);
		}

		return studentGroupSetMap;
	}

	/**
	 * Takes an already-validated row and creates an Account from the information inside through an offset
	 * representing the number of commas in the name. The final name will not contain any commas. Instead,
	 * individual parts of it previously separated by commas will be separated by spaces. Returns null if the
	 * email address provided is not valid.
	 *
	 * @param  row    Row containing account information
	 * @param  offset Offset representing the number of commas in the name
	 * @return        Account reflecting the information inside the row
	 */
	public Account getAccount(List<String> row, int offset) {
		String studentNumber = normalize(row.get(0));
		String netId = normalizeNetId(row.get(1));
		String fullName = getFullName(row, offset);
		String email = normalize(row.get(4 + offset));

		if (!validateStudentEmail(email))
			return null;

		Account account = new Account(studentNumber, email, netId, fullName);

		return account;
	}

	/**
	 * Gets the full name of a student according to the information stored in a row. Uses an offset to find
	 * the column containing the email
	 *
	 * @param  row    Row containing student information
	 * @param  offset Offset resulted from commas in name
	 * @return        String representing the full name of the student
	 */
	public String getFullName(List<String> row, int offset) {
		StringBuilder fullName = new StringBuilder();

		for (int i = 2; i <= 2 + offset; i++) {
			fullName.append(row.get(i)).append(" ");
		}
		fullName.append(row.get(2 + 1 + offset));

		return fullName.toString();
	}

	/**
	 * Removes the "#" character from a String.
	 *
	 * @param  s the input String
	 *
	 * @return   the normalized String.
	 */
	public String normalize(String s) {
		return s.replace("#", "");
	}

	/**
	 * Normalizes the NetId.
	 *
	 * @param  s the input String.
	 *
	 * @return   the normalized NetId.
	 */
	private String normalizeNetId(String s) {
		return normalize(s).replace("@tudelft.nl", "");
	}

	/**
	 * Validates the student emails
	 *
	 * @param  email returns whether the input email is a valid student email.
	 *
	 * @return       whether the given mail is valid.
	 */
	public boolean validateStudentEmail(String email) {
		StudentEmailValidator validator = new StudentEmailValidator();
		return validator.validate(email);
	}
}
