package server.csvconverter;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;

/**
 * Service that validates content in the UTF-8 format.
 */
public class UTF8Validator {
	/**
	 * The path of the file.
	 */
	private String filePath;

	/**
	 * FileInputStream object.
	 */
	private FileInputStream fileStream;

	/**
	 * InputStreamReader object.
	 */
	private InputStreamReader streamReader;

	/**
	 * BufferedReader object.
	 */
	private BufferedReader bufferedReader;

	/**
	 * StringBuilder object.
	 */
	private StringBuilder stringBuilder;

	/**
	 * Constructor
	 *
	 * @param  filePath                     the path of the file that needs to be validated.
	 * @throws FileNotFoundException
	 * @throws UnsupportedEncodingException
	 */
	public UTF8Validator(String filePath) throws FileNotFoundException, UnsupportedEncodingException {
		this.filePath = filePath;
		this.fileStream = new FileInputStream(filePath);
		this.streamReader = new InputStreamReader(fileStream, "UTF-8");
		this.bufferedReader = new BufferedReader(streamReader);
		this.stringBuilder = new StringBuilder();
	}

	/**
	 * Reads the input of the file.
	 *
	 * @return             The text from the file.
	 * @throws IOException
	 */
	public String readFile() throws IOException {
		String line;
		while ((line = bufferedReader.readLine()) != null) {
			stringBuilder.append(line).append(System.lineSeparator());
		}
		return stringBuilder.toString();
	}

	/**
	 * Checks whether the given String contains non-utf-8 characters.
	 *
	 * @param  input Text.
	 * @return
	 */
	public static boolean isValidUtf8(String input) {
		try {
			byte[] bytes = input.getBytes("UTF-8");
			String decoded = new String(bytes, "UTF-8");
			return input.equals(decoded);
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
			return false;
		}
	}

	/**
	 * Replaces the non-utf-8 characters with Unicode replacement character.
	 *
	 * @param  input Text.
	 * @return
	 */
	public static String replaceNonUtf8Characters(String input) {
		try {
			byte[] bytes = input.getBytes("UTF-8");
			if (isValidUtf8(input)) {
				return input;
			} else {
				StringBuilder sb = new StringBuilder();
				for (byte b : bytes) {
					if ((b & 0xC0) == 0x80) {
						// Replace non-UTF-8 character with UTF-8 replacement character
						sb.append('\uFFFD');
					} else {
						sb.append((char) b);
					}
				}
				return sb.toString();
			}
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
			return input;
		}
	}

	/**
	 * Validates the file by calling the previous methods.
	 *
	 * @throws IOException
	 */
	public void validate() throws IOException {
		String text = readFile();
		if (!isValidUtf8(text)) {
			text = replaceNonUtf8Characters(text);
			FileOutputStream outFileStream = new FileOutputStream(this.filePath);
			OutputStreamWriter outputStreamWriter = new OutputStreamWriter(outFileStream, "UTF-8");
			BufferedWriter bufferedWriter = new BufferedWriter(outputStreamWriter);
			bufferedWriter.write(text);
			bufferedWriter.close();

		}
		this.bufferedReader.close();
	}

}
