package server.labraService;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import nl.tudelft.labracore.api.CourseControllerApi;
import nl.tudelft.labracore.api.EditionControllerApi;
import nl.tudelft.labracore.api.PersonControllerApi;
import nl.tudelft.labracore.api.RoleControllerApi;
import nl.tudelft.labracore.api.dto.EditionDetailsDTO;
import nl.tudelft.labracore.api.dto.PersonDetailsDTO;
import nl.tudelft.labracore.api.dto.PersonSummaryDTO;
import nl.tudelft.labracore.api.dto.RolePersonDetailsDTO;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * Service that allows fetching data from LabraCORE.
 */
@Service
public class LabraConnectionService {
	@Autowired
	private CourseControllerApi courseControllerApi;

	@Autowired
	private EditionControllerApi editionControllerApi;

	@Autowired
	private RoleControllerApi roleControllerApi;

	@Autowired
	private PersonControllerApi personControllerApi;

	public PersonDetailsDTO getPerson(String username) {
		if (username == null)
			return null;

		return personControllerApi.getPersonByUsername(username).block();
	}

	/**
	 * Method that extracts all course names from LabraCore.
	 *
	 * @return A List containing all course names from LabraCore
	 */
	public List<String> getAllCourseNames() {
		return courseControllerApi.getAllCourses()
				.toStream().map(x -> x.getName()).collect(Collectors.toList());
	}

	/**
	 * Method for extracting the Emails for participants in the course that are not Students. This will fetch
	 * from the Labracore system, emails for TAs, Head TAs and Teachers. This is done through two api
	 * services. The edition one is used to get the ID of the edition of that specific course. Afterwards,
	 * using that ID we got, we fetch the mails of the edition staff, and put them in a List of Lists of
	 * Strings. The first list contains the list of TAs mails, the second list has the head TAs mail and the
	 * third one has the teachers.
	 *
	 * @param  courseName    the name of the course we want to fetch the staff from
	 * @param  courseEdition the name of the edition of the course we want to fetch the people from
	 * @return               A list of 3 lists containing the emails of the TAs, head TAs and teachers of the
	 *                       course, in that exact order
	 */
	public List<List<String>> getStaffEmailsForCourseEdition(String courseName, String courseEdition) {

		List<List<String>> res = new ArrayList<>();

		Optional<EditionDetailsDTO> editionDetails = editionControllerApi.getAllEditions()
				.toStream()
				.filter(x -> x.getCourse().getName().equals(courseName))
				.filter(x -> x.getName().equals(courseEdition))
				.findFirst();

		if (editionDetails.isEmpty())
			return Collections.EMPTY_LIST;

		long editionId = editionDetails.get().getId();

		List<Long> editionIdList = new ArrayList<>();
		editionIdList.add(editionId);

		List<RolePersonDetailsDTO> allRolesOfEdition = roleControllerApi.getRolesByEditions(editionIdList)
				.toStream()
				.collect(Collectors.toList());

		List<String> taMails = allRolesOfEdition.stream()
				.filter(x -> x.getType().equals(RolePersonDetailsDTO.TypeEnum.TA))
				.map(x -> x.getPerson().getEmail()).collect(Collectors.toList());

		res.add(taMails);

		List<String> headTaMails = allRolesOfEdition.stream()
				.filter(x -> x.getType().equals(RolePersonDetailsDTO.TypeEnum.HEAD_TA))
				.map(x -> x.getPerson().getEmail()).collect(Collectors.toList());

		res.add(headTaMails);

		List<String> teacherMail = allRolesOfEdition.stream()
				.filter(x -> x.getType().equals(RolePersonDetailsDTO.TypeEnum.TEACHER))
				.map(x -> x.getPerson().getEmail()).collect(Collectors.toList());

		//TODO: Check TEACHER_RO if to add or not

		res.add(teacherMail);

		return res;
	}

	/**
	 * Method that retrieves all TAs for a specific course edition.
	 *
	 * @param  courseName    The name that the course has in LabraCore.
	 * @param  courseEdition The edition of that course.
	 * @return               A list of DTOs containing TAs (and head TAs) for a specific course edition.
	 */
	public List<PersonSummaryDTO> getTeacherAssistantsForCourseEdition(String courseName,
			String courseEdition) {
		Optional<EditionDetailsDTO> editionDetails = editionControllerApi.getAllEditions()
				.toStream()
				.filter(x -> x.getCourse().getName().equals(courseName))
				.filter(x -> x.getName().equals(courseEdition))
				.findFirst();

		if (editionDetails.isEmpty())
			return Collections.EMPTY_LIST;

		long editionId = editionDetails.get().getId();

		List<PersonSummaryDTO> result = new ArrayList<>();

		List<RolePersonDetailsDTO> allRolesOfEdition = roleControllerApi
				.getRolesByEditions(List.of(editionId))
				.toStream()
				.collect(Collectors.toList());

		List<PersonSummaryDTO> taSummaries = allRolesOfEdition.stream()
				.filter(x -> x.getType().equals(RolePersonDetailsDTO.TypeEnum.TA))
				.map(x -> x.getPerson())
				.toList();

		result.addAll(taSummaries);

		List<PersonSummaryDTO> headTaSummaries = allRolesOfEdition.stream()
				.filter(x -> x.getType().equals(RolePersonDetailsDTO.TypeEnum.HEAD_TA))
				.map(x -> x.getPerson())
				.toList();

		result.addAll(headTaSummaries);

		return result;
	}

}
