package server.logging;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

import lombok.Data;

import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.FileSystemResource;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

/**
 * Service that logs action in a text file.
 */
@Data
@Service
public class LogWriter {

	@Value("${LOG_ERRORS}")
	private boolean logFailures;

	/**
	 * Method that adds a log in a file based on the type of action, target information and triggering user.
	 * Only logs failures if logFailures is set to true.
	 *
	 * @param logType         Type of action done by the user
	 * @param targetInfo      Information about the target of the action in string format
	 * @param responsibleUser User responsible for the action
	 */
	@Async
	public void writeActionLog(LogType logType, LogType actionType, String targetInfo,
			String responsibleUser) {
		if (!logFailures && logType == LogType.ERROR)
			return;

		String message = generateMessage(logType, actionType, targetInfo, responsibleUser);

		writeMessage(message);
	}

	/**
	 * Method that adds a log in a file based on the type of action, target information and triggering user,
	 * as well as a custom message. Only logs failures if logFailures is set to true.
	 *
	 * @param logType           Type of action done by the user
	 * @param targetInfo        Information about the target of the action in string format
	 * @param responsibleUser   User responsible for the action
	 * @param additionalMessage Extra message to be added to the log
	 */
	@Async
	public void writeActionLog(LogType logType, LogType actionType, String targetInfo,
			String responsibleUser, String additionalMessage) {
		if (!shouldLogMessage(logFailures, logType))
			return;

		String message = generateMessage(logType, actionType, targetInfo, responsibleUser) + "; "
				+ additionalMessage;

		writeMessage(message);
	}

	/**
	 * ONLY to be used for membership-related operations: USER_ADDED/REMOVED/UPDATED_TO/FROM/IN_PROJECT/GROUP
	 * Method that adds a log in a file based on the type of action, target information and triggering user.
	 * Only logs failures if logFailures is set to true.
	 *
	 * @param logType         Type of action done by the user
	 * @param userInfo        Information about the user target of the action in string format
	 * @param destinationInfo Information about the group or project target of the action in string format
	 * @param responsibleUser User responsible for the action
	 */
	@Async
	public void writeMemberActionLog(LogType logType, LogType actionType, String userInfo,
			String destinationInfo,
			String responsibleUser) {

		if (!shouldLogMessage(logFailures, logType))
			return;

		String pre = switch (actionType) {
			case USER_ADDED_TO_GROUP -> "to group";
			case USER_ADDED_TO_PROJECT -> "to project";
			case USER_MODIFIED_IN_GROUP -> "in group";
			case USER_MODIFIED_IN_PROJECT -> "in project";
			case USER_REMOVED_FROM_GROUP -> "from group";
			case USER_REMOVED_FROM_PROJECT -> "from project";

			default -> throw new RuntimeException("Unexpected actionType: " + actionType);
		};

		String message = generateMessage(logType, actionType, userInfo, responsibleUser) + " " + pre + " "
				+ destinationInfo;

		writeMessage(message);
	}

	/**
	 * ONLY to be used for membership-related operations: USER_ADDED/REMOVED/UPDATED_TO/FROM/IN_PROJECT/GROUP
	 * Method that adds a log in a file based on the type of action, target information and triggering user,
	 * as well as a custom message. Only logs failures if logFailures is set to true.
	 *
	 * @param logType           Type of action done by the user
	 * @param userInfo          Information about the user target of the action in string format
	 * @param destinationInfo   Information about the group or proejct target of the action in string format
	 * @param responsibleUser   User responsible for the action
	 * @param additionalMessage Extra message to be added to the log
	 */
	@Async
	public void writeMemberActionLog(LogType logType, LogType actionType, String userInfo,
			String destinationInfo,
			String responsibleUser, String additionalMessage) {

		if (!shouldLogMessage(logFailures, logType))
			return;

		String pre = switch (actionType) {
			case USER_ADDED_TO_GROUP -> "to group";
			case USER_ADDED_TO_PROJECT -> "to project";
			case USER_MODIFIED_IN_GROUP -> "in group";
			case USER_MODIFIED_IN_PROJECT -> "in project";
			case USER_REMOVED_FROM_GROUP -> "from group";
			case USER_REMOVED_FROM_PROJECT -> "from project";

			default -> throw new RuntimeException("Unexpected actionType: " + actionType);
		};

		String message = generateMessage(logType, actionType, userInfo, responsibleUser) + " " + pre + " "
				+ destinationInfo + "; "
				+ additionalMessage;

		writeMessage(message);
	}

	/**
	 * Wrapper for retrieving the current time through LocalTime.
	 *
	 * @return the current time
	 */
	LocalDateTime currentTime() {
		return LocalDateTime.now();
	}

	/**
	 * Method that generates the message that the log is going to contain.
	 *
	 * @param  logType         Type of action done by the user
	 * @param  targetInfo      Information about the target of the action in string format
	 * @param  responsibleUser User responsible for the action
	 * @return                 generated message
	 */
	public String generateMessage(LogType logType, LogType actionType, String targetInfo,
			String responsibleUser) {
		String timestamp = currentTime().format(DateTimeFormatter.ofPattern("uuuu-MM-dd HH:mm:ss.SSS"));

		final String infoType = switch (logType) {
			case INFO -> "[INFO]";
			case ERROR -> "[ERROR]";
			default -> throw new RuntimeException("Unexpected logType: " + logType);
		};

		final String hasFailed = logType == LogType.ERROR ? "NOT " : "";

		String target;
		String action;

		switch (actionType) {
			case USER_CREATED -> {
				target = "User";
				action = "created";
			}
			case USER_DELETED -> {
				target = "User";
				action = "deleted";
			}
			case COURSE_CREATED -> {
				target = "Course";
				action = "created";
			}
			case COURSE_DELETED -> {
				target = "Course";
				action = "deleted";
			}
			case EDITION_CREATED -> {
				target = "Course edition";
				action = "created";
			}
			case EDITION_DELETED -> {
				target = "Course edition";
				action = "deleted";
			}
			case GROUP_CREATED -> {
				target = "Group";
				action = "created";
			}
			case GROUP_DELETED -> {
				target = "Group";
				action = "deleted";
			}
			case PROJECT_CREATED -> {
				target = "Project";
				action = "created";
			}
			case PROJECT_DELETED -> {
				target = "Project";
				action = "deleted";
			}
			case PROJECT_FORKED -> {
				target = "Project";
				action = "forked";
			}
			case INITIAL_COMMIT -> {
				target = "Initial commit";
				action = "made";
			}
			case USER_ADDED_TO_GROUP, USER_ADDED_TO_PROJECT -> {
				target = "User";
				action = "added";
			}
			case USER_MODIFIED_IN_GROUP, USER_MODIFIED_IN_PROJECT -> {
				target = "User";
				action = "modified";
			}
			case USER_REMOVED_FROM_GROUP, USER_REMOVED_FROM_PROJECT -> {
				target = "User";
				action = "removed";
			}

			default -> throw new RuntimeException("Unexpected actionType: " + actionType);
		}

		if (responsibleUser.isEmpty())
			responsibleUser = "UNKNOWN USER";

		final String responsibility = " by " + responsibleUser;

		final String message = infoType + " [" + timestamp + "] " + target + " " + targetInfo + " was "
				+ hasFailed + action + responsibility;

		return message;
	}

	/**
	 * Method that gets the file writer necessary for the log to be recorded.
	 *
	 * @return             respective FileWriter
	 * @throws IOException if an I/O exception of any sort occurs
	 */
	FileWriter getFileWriter() throws IOException {
		File file = new File(
				new FileSystemResource("").getFile().getAbsolutePath() + "/data/action_log.txt");
		FileWriter writer = new FileWriter(file, true);
		return writer;
	}

	/**
	 * Writer that logs in the file "server/data/action_log.txt" the input message. In case of an IOException,
	 * it catches and prints it to the console along with the input message.
	 *
	 * @param message a String representing an entry in the log.
	 */
	private void writeMessage(String message) {
		try {
			FileWriter writer = getFileWriter();
			writer.write(message + System.lineSeparator());
			writer.close();
		} catch (IOException e) {
			System.out.println(
					"-!!!-\n[IOException] Logging message : " + message + " failed.\nReason: "
							+ e.getMessage() + "\n-!!!-");
		}
	}

	/**
	 * Helper method that determines whether to log a message
	 */
	private boolean shouldLogMessage(boolean logFailures, LogType logType) {
		return logFailures || logType == LogType.INFO;
	}
}
