package server.security;

import nl.tudelft.labracore.lib.security.user.Person;

import org.springframework.stereotype.Service;

import server.service.SecretMatcher;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * Service that handles security within the application.
 */
@Service
public class GitBullSecurity {

	private final SecretMatcher security;

	public GitBullSecurity(SecretMatcher security) {
		this.security = security;
	}

	/**
	 * Matches the secret received from the frontend to a Person from Labracore. The match is made when the
	 * user first logs in.
	 *
	 * The secret is received in a JSON format, which looks like this: {@code {"secret":"hfQD9Y3NCYen"}} In
	 * this example, the JsonNode and ObjectMapper would then save {@code actualSecret = hfQD9Y3NCYen} without
	 * the surrounding JSON boilerplate.
	 *
	 * @param  secret compound user secret + client secret (received automatically from SvelteKit)
	 * @return        Person from Labracore that matches the given secret
	 */
	public Person getAuthenticatedUser(String secret) {
		return secret == null ? null : security.get(secret);
	}

	/**
	 * This method decodes json-ified strings: an example is {"secret":"this-is-the-actual-secret"}, which
	 * this method makes directly "this-is-the-actual-secret". To be used when an endpoint uses a String as
	 * its secret directly in the RequestBody instead of wrapping it in a DTO.
	 *
	 * @param  secret Input stringified secret
	 * @return        Decoded version of the string
	 */
	public String decodeSecretString(String secret) {
		ObjectMapper mapper = new ObjectMapper();
		JsonNode sec;
		try {
			sec = mapper.readTree(secret);
		} catch (JsonProcessingException e) {
			throw new RuntimeException(e);
		}
		String actualSecret = sec.get("secret").asText();
		return actualSecret;
	}

	/**
	 * Uses getAuthenticatedUser() to fetch the username of the currently authenticated user.
	 *
	 * @param  secret compound user secret + client secret (received automatically from SvelteKit)
	 * @return        username of person from Labracore that matches the given secret
	 */
	public String getCurrentUsername(String secret) {
		if (secret == null)
			return "";

		Person person = getAuthenticatedUser(secret);
		if (person == null)
			return "";
		return person.getUsername();
	}

}
