package server.security;

import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;

import nl.tudelft.labracore.lib.security.LabradorUserHandler;
import nl.tudelft.labracore.lib.security.user.Person;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.web.reactive.function.BodyInserters;
import org.springframework.web.reactive.function.client.WebClient;

import reactor.core.publisher.Mono;
import server.entity.Account;
import server.service.AccountService;
import server.service.InstanceMappingService;
import server.service.SecretMatcher;
import server.util.FancyPrint;

/**
 * Service that handles user login.
 */
@Service
public class GitBullUserHandler implements LabradorUserHandler {

	@Autowired
	private AccountService accountService;
	@Autowired
	private InstanceMappingService instanceMappingService;

	@Value("${gitbull.frontend.url}")
	private String frontendUrl;

	@Value("${gitbull.frontend.api-secret}")
	private String apiSecret;

	@Value("${gitbull.crypto.secret}")
	private String cryptoSecret;

	@Autowired
	private SecretMatcher sm;

	/**
	 * Computes the client secret and sends it to the frontend server to use in future requests for said user.
	 * Retrieves user details from the database and maps him to his GitLab API key.
	 *
	 * @param person The Person object representing the authenticated user.
	 */
	@Override
	public void handleUserLogin(Person person) {
		//calculate user secret (FE)
		String userDetails = person.getUsername() + person.getExternalId();
		String userSecret;

		try {
			userSecret = HmacValidator.ComputeHash(apiSecret, userDetails.getBytes());
		} catch (InvalidKeyException | NoSuchAlgorithmException e) {
			throw new RuntimeException(e);
		}

		//calc client secret for storage
		String clientDetails = person.getExternalId() + person.getUsername();
		String clientSecret;
		try {
			clientSecret = HmacValidator.ComputeHash(cryptoSecret, clientDetails.getBytes());
		} catch (InvalidKeyException | NoSuchAlgorithmException e) {
			throw new RuntimeException(e);
		}

		//send user secret to frontend
		String call = frontendUrl + "/api/secure/";

		FancyPrint.println("auth", "sending user secret " + userSecret + " to " + call);

		WebClient request = WebClient.create(frontendUrl);
		WebClient.UriSpec<WebClient.RequestBodySpec> uriSpec = request.put();
		WebClient.RequestBodySpec bodySpec = uriSpec.uri("/api/secure");

		LinkedMultiValueMap map = new LinkedMultiValueMap();
		map.add("usec", userSecret);
		map.add("csec", clientSecret);
		WebClient.RequestHeadersSpec<?> headersSpec = bodySpec.body(
				BodyInserters.fromMultipartData(map));

		Mono<String> response = headersSpec.retrieve()
				.bodyToMono(String.class);

		response.subscribe(
				System.out::println,
				Throwable::printStackTrace,
				() -> System.out.println("completed without a value"));

		//store secrets in map
		String identifier = userSecret + clientSecret;
		sm.add(identifier, person);

		FancyPrint.println("auth", "user succesfully logged in");
		String username = person.getUsername();

		try {
			accountService.getAccountFromNetId(person.getUsername());
		} catch (Exception e) {
			Account currentAccount = Account.builder().netId(person.getUsername()).email(person.getEmail())
					.name(person.getDisplayName()).build();
			accountService.saveAccount(currentAccount);
		}

		String apiKey = accountService.getAPIKey(username);
		instanceMappingService.addInstance(username, apiKey);
	}
}
