package server.security;

import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Base64;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

/**
 * Class that provides hashing functionalities.
 */
public class HmacValidator {

	/**
	 * Computes the HMAC-SHA256 hash of the given payload using the provided secret key.
	 *
	 * @param  secret                   The secret key used for HMAC-SHA256 hashing.
	 * @param  payload                  The byte array representing the payload to be hashed.
	 * @return                          The Base64-encoded HMAC-SHA256 hash of the payload.
	 * @throws InvalidKeyException      If the provided secret key is invalid.
	 * @throws NoSuchAlgorithmException If the "HmacSHA256" algorithm is not available.
	 */
	public static String ComputeHash(String secret, byte[] payload)
			throws InvalidKeyException, NoSuchAlgorithmException {
		String digest = "HmacSHA256";
		Mac mac = Mac.getInstance(digest);
		mac.init(new SecretKeySpec(secret.getBytes(), digest));
		return new String(Base64.getEncoder().encode(mac.doFinal(payload)));
	}

	/**
	 * Verifies if the provided hash matches the computed HMAC-SHA256 hash of the given payload using the
	 * provided secret key.
	 *
	 * @param  secret                   The secret key used for HMAC-SHA256 hashing.
	 * @param  payload                  The byte array representing the payload for which the hash was
	 *                                  computed.
	 * @param  verify                   The hash to be verified against the computed hash.
	 * @return                          true if the computed hash matches the provided hash, false otherwise.
	 * @throws InvalidKeyException      If the provided secret key is invalid.
	 * @throws NoSuchAlgorithmException If the "HmacSHA256" algorithm is not available.
	 */
	public static boolean HashIsValid(String secret, byte[] payload, String verify)
			throws InvalidKeyException, NoSuchAlgorithmException {
		String computedHash = ComputeHash(secret, payload);
		return MessageDigest.isEqual(computedHash.getBytes(StandardCharsets.UTF_8),
				verify.getBytes(StandardCharsets.UTF_8));
	}
}
