package server.security.memory;

import java.io.IOException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;

import javax.servlet.ServletException;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import nl.tudelft.labracore.api.PersonControllerApi;
import nl.tudelft.labracore.api.dto.PersonCreateDTO;
import nl.tudelft.labracore.api.dto.PersonPatchDTO;
import nl.tudelft.labracore.lib.security.LabradorUserDetails;
import nl.tudelft.labracore.lib.security.LabradorUserHandler;
import nl.tudelft.labracore.lib.security.user.DefaultRole;
import nl.tudelft.labracore.lib.security.user.Person;

import org.modelmapper.ModelMapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.security.core.Authentication;
import org.springframework.security.web.authentication.AuthenticationSuccessHandler;
import org.springframework.web.reactive.function.client.WebClientResponseException;

import server.security.HmacValidator;
import server.util.FancyPrint;

public class GitBullAuthenticationSuccessHandler implements AuthenticationSuccessHandler {
	private final ModelMapper mapper = new ModelMapper();

	@Value("${gitbull.crypto.secret}")
	private String cryptoSecret;

	@Autowired
	private PersonControllerApi pApi;

	@Autowired
	private LabradorUserHandler userHandler;

	/**
	 * Delegate {@link AuthenticationSuccessHandler} that is called after updating user information.
	 */
	private AuthenticationSuccessHandler delegate;

	public GitBullAuthenticationSuccessHandler(AuthenticationSuccessHandler delegate) {
		this.delegate = delegate;
	}

	/**
	 * Updates a person in the Labracore backend. This method ensures the Person is updated appropriately
	 * using a series of RXJava reactive events. It patches the user in CORE if one is found, or adds a new
	 * one if the user did not yet exist in CORE.
	 *
	 * @param person The Authenticated Person to update to.
	 */
	private Long updatePersonInLabracore(Person person) {
		return pApi.getPersonByExternalId(person.getExternalId())
				.flatMap(pvd -> {
					var patch = mapper.map(person, PersonPatchDTO.class);
					if (patch.getNumber() == -1L)
						patch.setNumber(null);
					return pApi.patchPerson(pvd.getId(), patch);
				})
				.onErrorResume(error -> error instanceof WebClientResponseException,
						error -> pApi.addPerson(mapper.map(person, PersonCreateDTO.class)))
				.block();
	}

	/**
	 * Overrides the default AuthenticationSuccessHandler. Updates the person ins labracore and computes the
	 * client secret.
	 *
	 * @param  request          the request which caused the successful authentication
	 * @param  response         the response
	 * @param  authentication   the Authentication object which was created during the authentication process.
	 * @throws IOException
	 * @throws ServletException
	 */
	@Override
	public void onAuthenticationSuccess(HttpServletRequest request, HttpServletResponse response,
			Authentication authentication) throws IOException, ServletException {
		LabradorUserDetails ud = (LabradorUserDetails) authentication.getPrincipal();
		Person person = ud.getUser();

		String clientDetails = person.getExternalId() + person.getUsername();
		String clientSecret;
		try {
			clientSecret = HmacValidator.ComputeHash(cryptoSecret, clientDetails.getBytes());
		} catch (InvalidKeyException | NoSuchAlgorithmException e) {
			throw new RuntimeException(e);
		}

		FancyPrint.println("auth", "sending client secret to browser: " + clientSecret);
		response.addCookie(new Cookie("csec", clientSecret));

		person.setId(updatePersonInLabracore(person));
		var updatedPerson = pApi.getPersonById(person.getId()).block();
		person.setDefaultRole(DefaultRole.valueOf(updatedPerson.getDefaultRole().getValue()));

		userHandler.handleUserLogin(person);

		delegate.onAuthenticationSuccess(request, response, authentication);
	}
}
