package server.security.memory;

import nl.tudelft.labracore.lib.properties.LabradoorProperties;
import nl.tudelft.labracore.lib.properties.MemProperties;
import nl.tudelft.labracore.lib.security.LabradorSSOConfigurer;
import nl.tudelft.labracore.lib.security.memory.InMemoryUserDetailsService;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnExpression;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.dao.DaoAuthenticationProvider;
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.web.authentication.AuthenticationSuccessHandler;
import org.springframework.security.web.authentication.SavedRequestAwareAuthenticationSuccessHandler;
import org.springframework.security.web.authentication.SimpleUrlAuthenticationSuccessHandler;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;

@ConditionalOnExpression("#{'${labrador.sso.type}' == 'in_memory'}")
@EnableConfigurationProperties(LabradoorProperties.class)
@Configuration
@Primary
public class GitBullInMemorySecurityConfig implements LabradorSSOConfigurer {

	@Autowired
	private LabradoorProperties properties;

	private final PasswordEncoder passwordEncoder = new BCryptPasswordEncoder();

	@Bean
	@Primary
	AuthenticationSuccessHandler successHandler() {
		MemProperties mem = properties.getSso().getMem();
		if (mem.isSavedRequestAware()) {
			var successHandler = new SavedRequestAwareAuthenticationSuccessHandler();
			successHandler.setDefaultTargetUrl(mem.getSuccessUrl());

			return new GitBullAuthenticationSuccessHandler(successHandler);
		} else {
			return new GitBullAuthenticationSuccessHandler(
					new SimpleUrlAuthenticationSuccessHandler(mem.getSuccessUrl()));
		}
	}

	@Bean
	@Primary
	AuthenticationProvider authenticationProvider() {
		DaoAuthenticationProvider authenticationProvider = new DaoAuthenticationProvider();
		authenticationProvider.setUserDetailsService(userDetailsService());
		authenticationProvider.setPasswordEncoder(passwordEncoder);
		return authenticationProvider;
	}

	@Override
	public void configure(HttpSecurity http, String loginPath, String logoutPath) throws Exception {
		//@formatter:off
            http
                    .authorizeRequests()
                    .antMatchers(loginPath, logoutPath).permitAll()
                    .requestMatchers(new AntPathRequestMatcher("/perform-login", "POST")).permitAll()
                    .and().logout()
                    .logoutUrl(logoutPath)
                    .logoutSuccessUrl(properties.getSso().getMem().getSuccessUrl())
                    .and().formLogin()
                    .loginPage(loginPath)
                    .loginProcessingUrl("/perform-login")
                    .successHandler(successHandler())
                    .failureUrl(loginPath + "?error");
            //@formatter:on
	}

	@Override
	public void configure(AuthenticationManagerBuilder auth) throws Exception {
		auth.authenticationProvider(authenticationProvider());
	}

	@Bean
	@Primary
	InMemoryUserDetailsService userDetailsService() {
		return new InMemoryUserDetailsService();
	}
}
