package server.service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import server.entity.Access;
import server.repository.AccessRepository;

/**
 * The AccessService class is responsible for providing an implementation for the AccessService interface.
 *
 * It allows users to save, delete and fetch Access entries from the database.
 */
@Service
public class AccessService {

	/**
	 * Repository storing all Access values.
	 */
	@Autowired
	private AccessRepository accessRepository;

	/**
	 * Method for saving an Access entry into the database and setting/updating its timestamps.
	 *
	 * @param  access the Access entry that is saved into the database.
	 * @return        an Access object that represents the object saved.
	 */
	public Access saveAccess(Access access) {

		Optional<Access> old = accessRepository.findById(access.getAccessID());
		LocalDateTime currentTime = LocalDateTime.now();
		if (old.isPresent()) {
			access.setTimeCreated(old.get().getTimeCreated());
			access.setLastModified(currentTime);
		} else {
			access.setTimeCreated(currentTime);
			access.setLastModified(currentTime);
		}
		return accessRepository.save(access);
	}

	/**
	 * Method for deleting Access entries from the database, based on their ID.
	 *
	 * @param accessID the ID of the entry to be deleted.
	 */
	public void deleteAccess(long accessID) {
		accessRepository.deleteById(accessID);
	}

	/**
	 * Method for fetching all Access entries from the database.
	 *
	 * @return a list of all Access objects.
	 */
	public List<Access> fetchAllAccesses() {
		return (List<Access>) accessRepository.findAll();
	}

	/**
	 * Method for fetching all Access entries for a specific account.
	 *
	 * @param  netId the net ID for which we want to fetch the Accesses.
	 * @return       a list of Access objects matching to a specific user.
	 */
	public List<Access> fetchAllAccessesOfAccount(String netId) {
		List<Access> res = new ArrayList<>();
		Iterable<Access> allAccesses = accessRepository.findAll();
		allAccesses.forEach(x -> {
			if (x.getAccountID().equals(netId)) {
				res.add(x);
			}
		});
		return res;
	}

	/**
	 * Method that retrieves an Access object from the database based on the provided user ID, edition ID, and
	 * group ID.
	 *
	 * @param  userId    the ID of the user.
	 * @param  editionId the ID of the course edition.
	 * @param  groupId   the ID of the group.
	 * @return           the Access object corresponding to the provided parameters.
	 */
	public Access getAccessFromUserEditionGroup(String userId, long editionId, long groupId) {
		List<Access> allAccesses = (List<Access>) accessRepository.findAll();

		List<Access> res = allAccesses.stream().filter(x -> x.getGroupID() == groupId &&
				x.getCourseEditionID() == editionId &&
				x.getAccountID().equals(userId)).collect(Collectors.toList());

		if (res.size() != 0)
			return res.get(0);
		else
			return null;
	}

	/**
	 * Method that retrieves an Access object from the database based on the provided user ID and edition ID.
	 *
	 * @param  userId    the ID of the user.
	 * @param  editionId the ID of the course edition.
	 * @return           the Access object corresponding to the provided parameters.
	 */
	public Access getAccessFromUserEdition(String userId, long editionId) {
		List<Access> allAccesses = (List<Access>) accessRepository.findAll();

		List<Access> res = allAccesses.stream().filter(x -> x.getCourseEditionID() == editionId &&
				x.getAccountID().equals(userId)).collect(Collectors.toList());

		if (res.size() != 0)
			return res.get(0);
		else
			return null;
	}

	/**
	 * Method used to test the repository and add it manually instead of autowiring it.
	 *
	 * @param mockRepository the mock repository that is used for testing.
	 */
	public void setRepository(AccessRepository mockRepository) {
		this.accessRepository = mockRepository;
	}
}
