package server.service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import server.entity.Account;
import server.repository.AccountRepository;

/**
 * The AccountService class is responsible for providing an implementation for the AccountService interface.
 * It allows users to save, delete and fetch Account entries from the database.
 */
@Service
public class AccountService {

	/**
	 * Repository storing all Account values.
	 */
	@Autowired
	private AccountRepository accountRepository;

	@Autowired
	private InstanceMappingService instanceMappingService;

	/**
	 * Method for saving an Account entry into the database and setting/updating its timestamps.
	 *
	 * @param  account the Account entry that is saved into the database.
	 * @return         an Account object that represents the object saved.
	 */
	public Account saveAccount(Account account) {

		Optional<Account> old = accountRepository.findById(account.getNetId());
		LocalDateTime currentTime = LocalDateTime.now();
		if (old.isPresent()) {
			account.setTimeCreated(old.get().getTimeCreated());
			account.setLastModified(currentTime);
		} else {
			account.setTimeCreated(currentTime);
			account.setLastModified(currentTime);
		}
		return accountRepository.save(account);
	}

	/**
	 * Method for deleting Account entries from the database, based on their ID.
	 *
	 * @param accountID the ID of the entry to be deleted.
	 */
	public void deleteAccount(String accountID) {
		accountRepository.deleteById(accountID);
	}

	/**
	 * Method for fetching all Account entries from the database.
	 *
	 * @return a list of all Account objects.
	 */
	public List<Account> fetchAllAccounts() {
		return (List<Account>) accountRepository.findAll();
	}

	/**
	 * Method used to test the repository and add it manually instead of autowiring it.
	 *
	 * @param mockRepository the mock repository that is used for testing.
	 */
	public void setRepository(AccountRepository mockRepository) {
		this.accountRepository = mockRepository;
	}

	/**
	 * Method for removing an API key from an already existing user in the database, based on their ID.
	 *
	 * @param netID the ID of the user to add the key to
	 */
	public void removeAPIKey(String netID) {
		Optional<Account> oldAccount = accountRepository.findById(netID);
		if (oldAccount.isPresent()) {
			Account account = oldAccount.get();
			account.setApiKey(null);
			accountRepository.save(account);
			instanceMappingService.removeInstance(account.getNetId());
		}
	}

	/**
	 * Method for adding an API key to an already existing user in the database, based on their ID.
	 *
	 * @param netID the ID of the user to add the key to
	 * @param key   the API key to add
	 */
	public void addAPIKey(String netID, String key) {
		Optional<Account> oldAccount = accountRepository.findById(netID);
		if (oldAccount.isPresent()) {
			instanceMappingService.addInstance(netID, key);
			Account account = oldAccount.get();
			account.setApiKey(key);
			accountRepository.save(account);
		}
	}

	/**
	 * Method for getting an API key from the database based on ID.
	 *
	 * @param  netID the ID of the user to get the key from
	 * @return       the user's API key
	 */
	public String getAPIKey(String netID) {
		Optional<Account> oldAccount = accountRepository.findById(netID);
		if (oldAccount.isPresent()) {
			Account account = oldAccount.get();
			return account.getApiKey();
		} else
			return null;
	}

	/**
	 * Method that retrieves an Account object from the database based on the provided email address.
	 *
	 * @param  mail      the email address to search for in the database.
	 * @return           the Account object corresponding to the provided email address.
	 * @throws Exception if the user is not found in the database.
	 */
	public Account getAccountFromMail(String mail) throws Exception {
		List<Account> res = new ArrayList<>();
		accountRepository.findAll().forEach(x -> {
			if (x.getEmail().equals(mail)) {
				res.add(x);
			}
		});

		if (res.size() != 0)
			return res.get(0);
		else
			throw new Exception("User not found in DB");
	}

	/**
	 * Method that retrieves an Account object from the database based on the provided net ID.
	 *
	 * @param  netId     the net ID to search for in the database.
	 * @return           the Account object corresponding to the provided net ID.
	 * @throws Exception if the user with the given net ID does not exist in the database.
	 */
	public Account getAccountFromNetId(String netId) throws Exception {
		Optional<Account> acc = accountRepository.findById(netId);
		if (acc.isPresent()) {
			return acc.get();
		} else
			throw new Exception("The user with this netId does not exist");
	}
}
