package server.service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import server.entity.CourseEdition;
import server.repository.CourseRepository;

/**
 * The CourseEditionService class is responsible for providing an implementation for the CourseEditionService
 * interface. It allows users to save, delete and fetch CourseEdition entries from the database.
 */
@Service
public class CourseEditionService {

	/**
	 * Repository storing all CourseEdition values.
	 */
	@Autowired
	private CourseRepository courseRepository;

	/**
	 * Method for saving a CourseEdition entry into the database and setting/updating its timestamps.
	 *
	 * @param  courseEdition the CourseEdition entry that is saved into the database.
	 * @return               a CourseEdition object that represents the object saved.
	 */
	public CourseEdition saveCourseEdition(CourseEdition courseEdition) {
		Optional<CourseEdition> old = courseRepository.findById(courseEdition.getCourseEditionID());
		LocalDateTime currentTime = LocalDateTime.now();
		if (old.isPresent()) {
			courseEdition.setTimeCreated(old.get().getTimeCreated());
			courseEdition.setLastModified(currentTime);
		} else {
			courseEdition.setTimeCreated(currentTime);
			courseEdition.setLastModified(currentTime);
		}
		return courseRepository.save(courseEdition);
	}

	/**
	 * Method for deleting CourseEdition entries from the database, based on their ID.
	 *
	 * @param courseID the ID of the entry to be deleted.
	 */
	public void deleteCourseEdition(long courseID) {
		courseRepository.deleteById(courseID);
	}

	/**
	 * Method for deleting CourseEdition entries from the database, based on the course name and edition name.
	 *
	 * @param  courseName the name of the course
	 * @param  edition    the name of the edition
	 *
	 * @return
	 */
	public CourseEdition deleteCourseEdition(String courseName, String edition) {
		Iterable<CourseEdition> allEditions = courseRepository.findAll();
		for (CourseEdition e : allEditions) {
			if (e.getCourseName().equals(courseName) && e.getEdition().equals(edition)) {
				courseRepository.delete(e);
				return e;
			}
		}
		return null;
	}

	/**
	 * Method for fetching all CourseEdition entries from the database.
	 *
	 * @return a list of all CourseEdition objects.
	 */
	public List<CourseEdition> fetchAllCourseEditions() {
		return (List<CourseEdition>) courseRepository.findAll();
	}

	/**
	 * Method for fetching all CourseEdition entries for a specific course name.
	 *
	 * @param  courseName the name of the course for which we want to fetch the Courses.
	 * @return            a list of CourseEdition objects matching to a specific course name.
	 */
	public List<CourseEdition> fetchAllCourseEditionsOfCourse(String courseName) {
		List<CourseEdition> res = new ArrayList<>();

		Iterable<CourseEdition> unfiltered = courseRepository.findAll();
		unfiltered.forEach(x -> {
			if (x.getCourseName().equals(courseName)) {
				res.add(x);
			}
		});

		return res;
	}

	/**
	 * Method that fetches all CourseEdition entries with the given course IDs.
	 *
	 * @param  courseIds the list of course IDs.
	 * @return           a list of CourseEdition objects matching the provided course IDs.
	 */
	public List<CourseEdition> fetchAllCoursesOfListOfIds(List<Long> courseIds) {
		return (List<CourseEdition>) courseRepository.findAllById(courseIds);
	}

	/**
	 * Method used to test the repository and add it manually instead of autowiring it.
	 *
	 * @param mockRepository the mock repository that is used for testing.
	 */
	public void setRepository(CourseRepository mockRepository) {
		this.courseRepository = mockRepository;
	}

	/**
	 * Method that retrieves a CourseEdition object from the database based on the provided course path.
	 *
	 * @param  coursePath the path of the course.
	 * @return            the CourseEdition object corresponding to the provided course path.
	 */
	public CourseEdition getCourseEditionFromCoursePath(String coursePath) {
		List<CourseEdition> allCourseEditions = (List<CourseEdition>) courseRepository.findAll();
		List<CourseEdition> res = allCourseEditions.stream().filter(x -> x.getCoursePath().equals(coursePath))
				.collect(Collectors.toList());

		if (res.size() != 0) {
			return res.get(0);
		} else
			return null;
	}
}
