package server.service;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Optional;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import server.entity.CourseName;
import server.repository.CourseNamesRepository;

/**
 * The CourseNameService class is responsible for providing services related to CourseName entities. It is
 * meant to represent a Course, and it's corresponding path from GitLab. It allows users to save, delete, and
 * fetch CourseName entries from the database.
 */
@Service
public class CourseNameService {

	@Autowired
	private CourseNamesRepository courseNamesRepository;

	/**
	 * Method that saves a CourseName entry into the database and sets or updates its timestamps.
	 *
	 * @param  courseName the CourseName entry to be saved.
	 * @return            the saved CourseName object.
	 */
	public CourseName saveCourseName(CourseName courseName) {
		Optional<CourseName> old = courseNamesRepository.findById(courseName.getCourseName());
		LocalDateTime currentTime = LocalDateTime.now();
		if (old.isPresent()) {
			courseName.setTimeCreated(old.get().getTimeCreated());
			courseName.setLastModified(currentTime);
		} else {
			courseName.setTimeCreated(currentTime);
			courseName.setLastModified(currentTime);
		}
		return courseNamesRepository.save(courseName);
	}

	/**
	 * Method that deletes CourseName entries from the database based on the coursePath.
	 *
	 * @param coursePath the coursePath of the entries to be deleted.
	 */
	public void deleteCourseNames(String coursePath) {
		courseNamesRepository.deleteById(coursePath);
	}

	/**
	 * Method that fetches all CourseName entries from the database.
	 *
	 * @return a list of all CourseName objects.
	 */
	public List<CourseName> fetchAllCoursesNames() {
		return (List<CourseName>) courseNamesRepository.findAll();
	}

	/**
	 * Method that fetches the name associated with the given coursePath from the database.
	 *
	 * @param  coursePath the coursePath for which to fetch the name.
	 * @return            the name associated with the coursePath, or null if not found.
	 */
	public String fetchNameForCoursePath(String coursePath) {
		Optional<CourseName> res = courseNamesRepository.findById(coursePath);
		return res.map(CourseName::getCourseName).orElse(null);
	}

	/**
	 * Method used to set the mock repository for testing.
	 *
	 * @param mockRepository the mock repository that is used for testing.
	 */
	public void setRepository(CourseNamesRepository mockRepository) {
		this.courseNamesRepository = mockRepository;
	}

}
