package server.service;

import java.util.Date;
import java.util.List;

import org.gitlab4j.api.GitLabApi;
import org.gitlab4j.api.GitLabApiException;
import org.gitlab4j.api.models.Group;
import org.gitlab4j.api.models.GroupParams;
import org.gitlab4j.api.models.Member;
import org.springframework.stereotype.Service;

import server.model.MemberRequestDTO;

/**
 * Service that allows interacting with GitLab's GroupAPI.
 */
@Service
public class GroupAPIService {

	/**
	 * Creates a new project group. Available only for users who can create groups.
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  groupName          the name of the project top create
	 * @param  description        the description of the project
	 * @param  path               the path of the group to be created
	 * @param  hasMemberShipLock  if the group has membership lock - Prevent adding new members to project
	 *                            membership within this group.
	 * @param  visibility         the visibility of the group
	 *
	 * @return                    the created Group instance
	 * @throws GitLabApiException if any exception occurs
	 */
	public Group createGroup(GitLabApi gitLabApi, String groupName, String description, String path,
			boolean hasMemberShipLock,
			String visibility) throws GitLabApiException {
		GroupParams groupParams = new GroupParams()
				.withName(groupName)
				.withDescription(description)
				.withPath(path)
				.withMembershipLock(hasMemberShipLock)
				.withVisibility(visibility);
		return gitLabApi.getGroupApi().createGroup(groupParams);
	}

	/**
	 * Creates a new project group. Available only for users who can create groups.
	 *
	 * @param  gitLabApi                    GitLab API instance
	 * @param  groupName                    the name of the project top create
	 * @param  description                  the description of the project
	 * @param  path                         the path of the group to be created
	 * @param  defaultBranchProtectionLevel the default branch protection level
	 *
	 *                                      NOT_PROTECTED - No protection. Users with the Developer or
	 *                                      Maintainer role can: - Push new commits - Force push changes -
	 *                                      Delete the branch PARTIALLY_PROTECTED - Partial protection. Users
	 *                                      with the Developer or Maintainer role can: - Push new commits
	 *                                      FULLY_PROTECTED - Full protection. Only users with the Maintainer
	 *                                      role can: - Push new commits
	 * @param  visibility                   the visibility of the group
	 *
	 * @return                              the created Group instance
	 * @throws GitLabApiException           if any exception occurs
	 */
	public Group createGroupUpdated(GitLabApi gitLabApi, String groupName, String description, String path,
			GroupParams.DefaultBranchProtectionLevel defaultBranchProtectionLevel,
			String visibility) throws GitLabApiException {
		GroupParams groupParams = new GroupParams()
				.withName(groupName)
				.withDescription(description)
				.withPath(path)
				.withDefaultBranchProtection(defaultBranchProtectionLevel)
				.withVisibility(visibility);
		return gitLabApi.getGroupApi().createGroup(groupParams);
	}

	/**
	 * Creates a new project subgroup. Available only for users who can create groups.
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  groupName          the name of the project top create
	 * @param  description        the description of the project
	 * @param  path               the path of the group to be created membership within this group.
	 * @param  visibility         the visibility of the group
	 * @param  parentId           the parentId where the subgroup is to be created
	 *
	 * @return                    the created Group instance
	 * @throws GitLabApiException if any exception occurs
	 */
	public Group createSubgroupUpdated(GitLabApi gitLabApi, String groupName, String description, String path,
			String visibility, Long parentId,
			GroupParams.DefaultBranchProtectionLevel defaultBranchProtectionLevel) throws GitLabApiException {
		GroupParams groupParams = new GroupParams()
				.withName(groupName)
				.withDescription(description)
				.withPath(path)
				.withParentId(parentId)
				.withMembershipLock(false)
				.withVisibility(visibility)
				.withDefaultBranchProtection(defaultBranchProtectionLevel);
		return gitLabApi.getGroupApi().createGroup(groupParams);
	}

	/**
	 * Creates a new project subgroup. Available only for users who can create groups.
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  groupName          the name of the project top create
	 * @param  description        the description of the project
	 * @param  path               the path of the group to be created
	 * @param  hasMemberShipLock  if the group has membership lock - Prevent adding new members to project
	 *                            membership within this group.
	 * @param  visibility         the visibility of the group
	 * @param  parentId           the parentId where the subgroup is to be created
	 *
	 * @return                    the created Group instance
	 * @throws GitLabApiException if any exception occurs
	 */
	public Group createSubgroup(GitLabApi gitLabApi, String groupName, String description, String path,
			boolean hasMemberShipLock,
			String visibility, Long parentId) throws GitLabApiException {
		GroupParams groupParams = new GroupParams()
				.withName(groupName)
				.withDescription(description)
				.withPath(path)
				.withParentId(parentId)
				.withMembershipLock(hasMemberShipLock)
				.withVisibility(visibility);
		return gitLabApi.getGroupApi().createGroup(groupParams);
	}

	/**
	 * Gets all the groups associated with GitLabApi namespace instance.
	 *
	 * @param  gitLabApi          GitLab API instance
	 *
	 * @return                    all groups
	 *
	 * @throws GitLabApiException if any exception occurs
	 */
	public List<Group> getGroups(GitLabApi gitLabApi) throws GitLabApiException {
		return gitLabApi.getGroupApi().getGroups();
	}

	/**
	 * Adds a new member to a group in the GitLabApi namespace instance.
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  dto                MemberRequestDTO containing information about the group and the member to be
	 *                            added
	 *
	 * @return                    member instance representing the new member
	 *
	 * @throws GitLabApiException if any exception occurs
	 */
	public Member addGroupMember(GitLabApi gitLabApi, MemberRequestDTO dto) throws GitLabApiException {
		return addGroupMember(gitLabApi, dto.getIdOrPath(), dto.getUserId(), dto.getAccessLevel(),
				dto.getExpiresAt());
	}

	/**
	 *
	 * Updates a member to a group in the GitLabApi namespace instance.
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  dto                MemberRequestDTO containing information about the group and the member to be
	 *                            added
	 *
	 * @return                    member instance representing the updated member
	 *
	 * @throws GitLabApiException if any exception occurs
	 */
	public Member updateGroupMember(GitLabApi gitLabApi, MemberRequestDTO dto) throws GitLabApiException {
		return gitLabApi.getGroupApi().updateMember(dto.getIdOrPath(), dto.getUserId(), dto.getAccessLevel(),
				dto.getExpiresAt());
	}

	/**
	 *
	 * Removes a member from a group in the GitLabApi namespace instance.
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  dto                MemberRequestDTO containing information about the group and the member to be
	 *                            added
	 *
	 * @throws GitLabApiException if any exception occurs
	 */
	public void removeGroupMember(GitLabApi gitLabApi, MemberRequestDTO dto)
			throws GitLabApiException {
		gitLabApi.getGroupApi().removeMember(dto.getIdOrPath(), dto.getUserId());
	}

	/**
	 * Adds a new member to a group in the GitLabApi namespace instance.
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  groupIdOrPath      group ID (Long), path of group (String) or Group instance
	 * @param  userId             user ID of member to add to group
	 * @param  accessLevel        access level of the new member
	 * @param  expiresAt          date membership will expire (optional)
	 *
	 * @return                    member instance representing the new member
	 *
	 * @throws GitLabApiException if any exception occurs
	 */
	public Member addGroupMember(GitLabApi gitLabApi, Object groupIdOrPath, Long userId, Integer accessLevel,
			Date expiresAt) throws GitLabApiException {
		return gitLabApi.getGroupApi().addMember(groupIdOrPath, userId, accessLevel, expiresAt);
	}

	/**
	 *
	 * Updates a member to a group in the GitLabApi namespace instance.
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  groupIdOrPath      group ID (Long), path of group (String) or Group instance
	 * @param  userId             user ID of member to add to group
	 * @param  accessLevel        access level of the new member
	 *
	 * @return                    member instance representing the updated member
	 *
	 * @throws GitLabApiException if any exception occurs
	 */
	public Member updateGroupMember(GitLabApi gitLabApi, Object groupIdOrPath, Long userId,
			Integer accessLevel, Date expiresAt) throws GitLabApiException {
		return gitLabApi.getGroupApi().updateMember(groupIdOrPath, userId, accessLevel, expiresAt);
	}

	/**
	 *
	 * Removes a member from a group in the GitLabApi namespace instance.
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  groupIdOrPath      group ID (Long), path of group (String) or Group instance
	 * @param  userId             user ID of member to add to group
	 *
	 * @throws GitLabApiException if any exception occurs
	 */
	public void removeGroupMember(GitLabApi gitLabApi, Object groupIdOrPath, Long userId)
			throws GitLabApiException {
		gitLabApi.getGroupApi().removeMember(groupIdOrPath, userId);
	}

	/**
	 * Removes group with all projects inside.
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  groupIdOrPath      group ID (Long), path of group (String) or Group instance
	 *
	 * @return                    list of members in the group
	 * @throws GitLabApiException if any exception occurs
	 */
	public List<Member> getMembers(GitLabApi gitLabApi, Object groupIdOrPath) throws GitLabApiException {
		return gitLabApi.getGroupApi().getMembers(groupIdOrPath);
	}

	/**
	 * Get a list of group members viewable by the authenticated user.
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  groupIdOrPath      group ID (Long), path of group (String) or Group instance
	 *
	 * @throws GitLabApiException if any exception occurs
	 */
	public void deleteGroup(GitLabApi gitLabApi, Object groupIdOrPath) throws GitLabApiException {
		gitLabApi.getGroupApi().deleteGroup(groupIdOrPath);
	}

	/**
	 * Updates a project group. Available only for users who can create groups.
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  groupIdOrPath      group ID (Long), path of group (String) or Group instance
	 * @param  params             the GroupParams instance holding the properties to update
	 *
	 * @return                    updated group instance
	 * @throws GitLabApiException if any exception occurs
	 */
	public Group updateGroup(GitLabApi gitLabApi, Object groupIdOrPath, GroupParams params)
			throws GitLabApiException {
		return gitLabApi.getGroupApi().updateGroup(groupIdOrPath, params);
	}

	/**
	 * Get the id of a group based on the path of the group
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  groupIdOrPath      group ID (Long), path of group (String) or Group instance
	 *
	 * @return                    the id of the group
	 * @throws GitLabApiException if any exception occurs
	 */
	public Long getGroupId(GitLabApi gitLabApi, Object groupIdOrPath) throws GitLabApiException {
		Group group = gitLabApi.getGroupApi().getGroup(groupIdOrPath);
		return group.getId();
	}

	/**
	 * Get all subgroups within a group
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  groupIdOrPath      group ID (Long), path of group (String) or Group instance
	 *
	 * @return                    the list of subgroups within the groups
	 * @throws GitLabApiException if any exception occurs
	 */
	public List<Group> getSubgroups(GitLabApi gitLabApi, Object groupIdOrPath) throws GitLabApiException {
		return gitLabApi.getGroupApi().getSubGroups(groupIdOrPath).stream().toList();
	}

	public Group getGroup(GitLabApi gitLabApi, Object groupIdOrPath) throws GitLabApiException {
		return gitLabApi.getGroupApi().getGroup(groupIdOrPath);
	}

	/**
	 * Get all subgroups within a group
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  groupIdOrPath      group ID (Long), path of group (String) or Group instance
	 *
	 * @return                    the list of subgroups within the groups
	 * @throws GitLabApiException if any exception occurs
	 */
	public List<String> getSubgroupPaths(GitLabApi gitLabApi, Object groupIdOrPath)
			throws GitLabApiException {
		List<Group> subgroups = gitLabApi.getGroupApi().getSubGroups(groupIdOrPath);
		List<String> subgroupPaths = subgroups.stream()
				.map(x -> x.getPath())
				.toList();
		return subgroupPaths;
	}

	// TODO : document API specific variables such as visibility and accesslevel
}
