package server.service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import server.entity.StudentGroup;
import server.repository.GroupRepository;

/**
 * The GroupService class is responsible for providing an implementation for the GroupService interface. It
 * allows users to save, delete and fetch Access entries from the database.
 */
@Service
public class GroupService {

	/**
	 * Repository storing all StudentGroup values.
	 */
	@Autowired
	private GroupRepository groupRepository;

	/**
	 * Method for saving a StudentGroup entry into the database and setting/updating its timestamps.
	 *
	 * @param  gr the StudentGroup entry that is saved into the database.
	 * @return    an StudentGroup object that represents the object saved.
	 */
	public StudentGroup saveGroup(StudentGroup gr) {
		Optional<StudentGroup> old = groupRepository.findById(gr.getGroupID());
		LocalDateTime currentTime = LocalDateTime.now();
		if (old.isPresent()) {
			gr.setTimeCreated(old.get().getTimeCreated());
			gr.setLastModified(currentTime);
		} else {
			gr.setTimeCreated(currentTime);
			gr.setLastModified(currentTime);
		}
		return groupRepository.save(gr);
	}

	/**
	 * Method for deleting StudentGroup entries from the database, based on their ID.
	 *
	 * @param groupId the ID of the entry to be deleted.
	 */
	public void deleteGroup(long groupId) {
		groupRepository.deleteById(groupId);
	}

	/**
	 * Method for fetching all StudentGroup entries from the database.
	 *
	 * @return a list of all StudentGroup objects.
	 */
	public List<StudentGroup> fetchAllGroups() {
		return (List<StudentGroup>) groupRepository.findAll();
	}

	/**
	 * Method for fetching all StudentGroup entries for a specific course.
	 *
	 * @param  courseId the course ID for which we want to fetch the StudentGroups.
	 * @return          a list of StudentGroup objects matching to a specific course.
	 */
	public List<StudentGroup> fetchAllGroupsFromCourse(long courseId) {
		List<StudentGroup> res = new ArrayList<>();

		Iterable<StudentGroup> notfilteredGroups = groupRepository.findAll();
		notfilteredGroups.forEach(x -> {
			if (x.getCourseID() == courseId) {
				res.add(x);
			}
		});

		return res;
	}

	/**
	 * Method that fetches all StudentGroup entries with the given groupIds.
	 *
	 * @param  groupIds the list of groupIds to fetch.
	 * @return          a list of StudentGroup objects matching the specified groupIds.
	 */
	public List<StudentGroup> fetchAllGroupsFromId(List<Long> groupIds) {
		return (List<StudentGroup>) groupRepository.findAllById(groupIds);
	}

	/**
	 * Method used to test the repository and add it manually instead of autowiring it.
	 *
	 * @param mockRepository the mock repository that is used for testing.
	 */
	public void setRepository(GroupRepository mockRepository) {
		this.groupRepository = mockRepository;
	}

	/**
	 * Method that retrieves the StudentGroup based on the groupPath.
	 *
	 * @param  groupPath the groupPath to search for.
	 * @return           the StudentGroup object matching the specified groupPath, or null if not found.
	 */
	public StudentGroup getGroupFromGroupPath(String groupPath) {
		List<StudentGroup> allGroups = (List<StudentGroup>) groupRepository.findAll();
		List<StudentGroup> res = allGroups.stream().filter(x -> x.getGroupPath().equals(groupPath))
				.collect(Collectors.toList());

		if (res.size() != 0) {
			return res.get(0);
		} else
			return null;
	}
}
