package server.service;

import java.util.HashMap;

import org.gitlab4j.api.GitLabApi;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

/**
 * Service that maps users to secrets through their username.
 */
@Service
public class InstanceMappingService {
	@Value("${GITLAB_SERVER_URL}")
	String serverUrl;
	private final HashMap<String, GitLabApi> instanceMap;

	private final GitLabApi rootGitLabAPI;

	/**
	 * noAuthMode determines whether the user requires to be authenticated in order to receive a GitLabAPI
	 * instance.
	 *
	 * developmentMode determines whether to use the server's "hardcoded" GitLabAPI instance or one created
	 * from his personal API key.
	 *
	 */
	@Autowired
	public InstanceMappingService(GitLabApi rootGitLabAPI) {
		this.rootGitLabAPI = rootGitLabAPI;
		this.instanceMap = new HashMap<>();
	}

	/**
	 * Method to add a new GitLabApi instance to the map.
	 *
	 * @param username username of the user
	 * @param apiKey   API key of the user
	 */
	public void addInstance(String username, String apiKey) {
		GitLabApi gitLabApi = new GitLabApi(serverUrl, apiKey);
		instanceMap.put(username, gitLabApi);
	}

	/**
	 * Method to remove a GitLab instance from the map.
	 *
	 * @param username username of the user
	 */
	public void removeInstance(String username) {
		instanceMap.remove(username);
	}

	/**
	 * Method to get a GitLabApi instance from the map based on a username.
	 *
	 * @return associated GitLabApi instance
	 */
	public GitLabApi getInstance(String username) {
		if (username == null)
			return null;

		GitLabApi gitLabApi = instanceMap.get(username);
		return gitLabApi;
	}
}
