package server.service;

import java.util.List;

import org.gitlab4j.api.GitLabApi;
import org.gitlab4j.api.GitLabApiException;
import org.gitlab4j.api.models.*;
import org.springframework.stereotype.Service;

import server.model.MemberRequestDTO;
import server.model.ProjectRequestDTO;

/**
 * Service that allows interacting with GitLab's ProjectAPI.
 */
@Service
public class ProjectAPIService {

	private final String EMAIL_REGEX = "^[A-Za-z0-9._+-]+@(student\\.)?tudelft\\.nl$";

	/**
	 *
	 * @param  gitLabApi          GitLabApi instance
	 * @param  projectRequestDTO  DTO containing information about the project
	 *
	 * @return                    the created project
	 * @throws GitLabApiException if any exception occurs
	 */
	public Project createProjectInNamespace(GitLabApi gitLabApi, ProjectRequestDTO projectRequestDTO)
			throws GitLabApiException {
		Project projectSpec = new Project()
				.withName(projectRequestDTO.getName())
				.withDescription(projectRequestDTO.getDescription())
				.withIssuesEnabled(projectRequestDTO.isIssuesEnabled())
				.withMergeRequestsEnabled(projectRequestDTO.isMergeRequestsEnabled())
				.withWikiEnabled(projectRequestDTO.isWikiEnabled())
				.withSnippetsEnabled(projectRequestDTO.isSnippetsEnabled())
				.withPublic(projectRequestDTO.isPublic());

		Project createdProject = gitLabApi.getProjectApi().createProject(projectRequestDTO.getNamespaceId(),
				projectSpec);

		try {
			PushRules pushRules = new PushRules().withAuthorEmailRegex(EMAIL_REGEX);

			gitLabApi.getProjectApi().createPushRules(createdProject.getId(), pushRules);
		} catch (GitLabApiException e) {
		}

		return createdProject;
	}

	/**
	 *
	 * @param  gitLabApi          GitLab API instance
	 *
	 * @return                    returns a list of all projects
	 *
	 * @throws GitLabApiException if any exception occurs
	 */
	public List<Project> getProjects(GitLabApi gitLabApi) throws GitLabApiException {
		return gitLabApi.getProjectApi().getProjects();
	}

	/**
	 * Adds a new member to a project in the GitLabApi namespace instance.
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  dto                MemberRequestDTO containing information about the path of the project, the
	 *                            userId, accessLevel and expiration
	 *
	 * @return                    member instance representing the new member
	 *
	 * @throws GitLabApiException if any exception occurs
	 */
	public Member addProjectMember(GitLabApi gitLabApi, MemberRequestDTO dto) throws GitLabApiException {
		return gitLabApi.getProjectApi().addMember(dto.getIdOrPath(), dto.getUserId(), dto.getAccessLevel(),
				dto.getExpiresAt());
	}

	/**
	 * Updates a member in a project in the GitLabApi namespace instance.
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  dto                MemberRequestDTO containing information about the user and the project
	 *
	 * @return                    member instance representing the updated member
	 *
	 * @throws GitLabApiException if any exception occurs
	 */
	public Member updateProjectMember(GitLabApi gitLabApi, MemberRequestDTO dto) throws GitLabApiException {
		return gitLabApi.getProjectApi().updateMember(dto.getIdOrPath(), dto.getUserId(),
				dto.getAccessLevel(), dto.getExpiresAt());
	}

	/**
	 * Removes a member from a project in the GitLabApi namespace instance.
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  dto                MemberRequestDTO containing information about the user and the project
	 *
	 * @throws GitLabApiException if any exception occurs
	 */
	public void removeProjectMember(GitLabApi gitLabApi, MemberRequestDTO dto)
			throws GitLabApiException {
		gitLabApi.getProjectApi().removeMember(dto.getIdOrPath(), dto.getUserId());
	}

	public Project forkProject(GitLabApi gitLabApi, Object projectIdOrPath, Long namespaceId)
			throws GitLabApiException {
		Project createdProject = gitLabApi.getProjectApi().forkProject(projectIdOrPath, namespaceId);

		try {
			PushRules pushRules = new PushRules().withAuthorEmailRegex(EMAIL_REGEX);

			gitLabApi.getProjectApi().createPushRules(createdProject.getId(), pushRules);
		} catch (GitLabApiException e) {
		}

		return createdProject;
	}

	/**
	 * Gets a list of project team members.
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  projectIdOrPath    the project in the form of a Long(ID), String(path), or Project instance
	 *
	 * @return                    a list of the members belonging to the specified project
	 * @throws GitLabApiException if any exception occurs
	 */
	public List<Member> getMembers(GitLabApi gitLabApi, Object projectIdOrPath) throws GitLabApiException {
		return gitLabApi.getProjectApi().getMembers(projectIdOrPath);
	}

	/**
	 * Creates an initial commit in a given project.
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  projectIdOrPath    the project in the form of a Long(ID), String(path), or Project instance
	 * @param  projectName        the name of the project
	 * @throws GitLabApiException if any exception occurs
	 */
	public Commit createInitialCommit(GitLabApi gitLabApi, Object projectIdOrPath, String projectName)
			throws GitLabApiException {

		CommitAction readmeAction = new CommitAction()
				.withAction(CommitAction.Action.CREATE)
				.withFilePath("README.md")
				.withContent("# " + projectName);

		CommitPayload commitPayload = new CommitPayload()
				.withBranch("master")
				.withCommitMessage("Initial commit")
				.withAction(readmeAction);

		return gitLabApi.getCommitsApi().createCommit(projectIdOrPath, commitPayload);
	}

	/**
	 * Removes project with all resources(issues, merge requests etc).
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  projectIdOrPath    the project in the form of a Long(ID), String(path), or Project instance
	 *
	 * @throws GitLabApiException if any exception occurs
	 */
	public void deleteProject(GitLabApi gitLabApi, Object projectIdOrPath) throws GitLabApiException {
		gitLabApi.getProjectApi().deleteProject(projectIdOrPath);
	}

	/**
	 * Updates a project. Null values are not updated Either id or path must be provided
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  project            the Project instance with the configuration for the new project
	 *
	 * @return                    a Project instance with the newly updated project info
	 * @throws GitLabApiException if any exception occurs
	 */
	public Project updateProject(GitLabApi gitLabApi, Project project) throws GitLabApiException {
		return gitLabApi.getProjectApi().updateProject(project);
	}

	/**
	 * Gets a project.
	 *
	 * @param  gitLabApi          GitLab API instance
	 * @param  projectIdOrPath    the path or id of the project
	 * @return                    a Project instance from the given path or id
	 * @throws GitLabApiException if any exception occurs
	 */
	public Project getProject(GitLabApi gitLabApi, Object projectIdOrPath) throws GitLabApiException {
		return gitLabApi.getProjectApi().getProject(projectIdOrPath);
	}

}
