package server.service;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;

import org.gitlab4j.api.GitLabApi;
import org.gitlab4j.api.GitLabApiException;
import org.gitlab4j.api.models.User;
import org.springframework.data.util.Pair;
import org.springframework.stereotype.Service;

import server.entity.Account;
import server.entity.StudentGroup;
import server.service.util.ThreadedUtils;

/**
 * Service that allows to interact with GitLab's UserAPI.
 */
@Service
public class UserAPIService {

	ThreadedUtils threadedUtils = new ThreadedUtils();

	/**
	 * Creates a new GitLab user with the given email and username. The username used is supposed to be the
	 * NetID.
	 *
	 * @param  gitLabApi          GitLabAPI instance
	 * @param  email              Email address of the user to be created.
	 * @param  username           Username of the user to be created.
	 * @return                    Created User
	 * @throws GitLabApiException if any error occurs
	 */
	public User createUser(GitLabApi gitLabApi, String email, String username, String name)
			throws GitLabApiException {
		User user = new User()
				.withEmail(email)
				.withUsername(username)
				.withName(name);
		return gitLabApi.getUserApi().createUser(user, null, true);
	}

	/**
	 * Returns all the users in the given GitLab instance.
	 *
	 * @param  gitLabApi          GitLabAPI instance
	 * @return                    List of users
	 * @throws GitLabApiException if any error occurs
	 */
	public List<User> getUsers(GitLabApi gitLabApi) throws GitLabApiException {
		return gitLabApi.getUserApi().getUsers();
	}

	/**
	 * Service method that takes a map of groups with students then creates GitLab accounts for the students
	 * who don't have one already. Continues with disregard for account creation failures. Returns a pair
	 * consisting of a list of accounts that couldn't be created and a list of all existing (after creation)
	 * user accounts in the map.
	 *
	 * @param  gitLabApi          GitLabAPI instance
	 * @param  studentGroupSetMap Map of student groups and sets of accounts
	 * @return                    A pair of a list of users that couldn't be created and a list of existing
	 *                            users.
	 */
	public Pair<List<Account>, List<User>> verifyAccounts(GitLabApi gitLabApi,
			Map<StudentGroup, Set<Account>> studentGroupSetMap) {
		// Get usernames from imported list
		Set<Account> accounts = new HashSet<>();
		for (Set<Account> set : studentGroupSetMap.values()) {
			accounts.addAll(set);
		}

		// Find all students with emails not already on GitLab
		Map<Account, CompletableFuture<User>> userJobs = new HashMap<>();
		for (Account account : accounts) {
			try {
				//System.out.println(account.getEmail() + " " + account);
				CompletableFuture<User> asyncUser = threadedUtils.asyncGetUserByEmail(gitLabApi, account);
				userJobs.put(account, asyncUser);
			} catch (GitLabApiException e) {
				throw new RuntimeException(e);
			}
		}

		userJobs.values().forEach(CompletableFuture::join);

		// This list will contain both created and already existing users
		List<User> students = new ArrayList<>();

		List<Account> usersToBeCreated = userJobs
				.entrySet()
				.stream()
				.filter(x -> {
					try {
						User user = x.getValue().get();
						if (user == null) {
							return true;
						} else {
							students.add(user);
							return false;
						}
					} catch (InterruptedException | ExecutionException e) {
						throw new RuntimeException(e);
					}
				})
				.map(Map.Entry::getKey)
				.toList();

		// Create needed student accounts and log failing ones
		List<Account> failedStudents = new ArrayList<>();

		for (Account account : usersToBeCreated) {
			try {
				//System.out.println(account.getEmail() + " " + account.getNetId() + " " + account.getName());
				User user = createUser(gitLabApi, account.getEmail(), account.getNetId(), account.getName());
				students.add(user);
			} catch (GitLabApiException e) {
				failedStudents.add(account);
				//System.out.println(e.getMessage());
			}
		}

		// TODO : Display warnings if accounts couldn't be created

		return Pair.of(failedStudents, students);
	}

	/**
	 * Get a user by the email address.
	 *
	 * @param  gitLabApi          GitLabAPI instance
	 * @param  email              the email of the user
	 *
	 * @return                    User entity based on the email of the user
	 * @throws GitLabApiException if any error occurs
	 */
	public User getUserByEmail(GitLabApi gitLabApi, String email) throws GitLabApiException {
		return gitLabApi.getUserApi().getUserByEmail(email);
	}

	/**
	 * Get the currently authenticated user.
	 *
	 * @param  gitLabApi          GitLabAPI instance
	 * @return                    User entity of currently authenticated user
	 * @throws GitLabApiException if any error occurs
	 */

	public User getCurrentUser(GitLabApi gitLabApi) throws GitLabApiException {
		return gitLabApi.getUserApi().getCurrentUser();
	}

	/**
	 * Get the user by the username.
	 *
	 * @param  gitLabApi          GitLabAPI instance
	 * @param  username           The username used to search
	 * @return                    User entity of currently authenticated user
	 * @throws GitLabApiException if any error occurs
	 */
	public User getUserByUsername(GitLabApi gitLabApi, String username) throws GitLabApiException {
		return gitLabApi.getUserApi().getUser(username);
	}
}
