package server.service.statisticsService;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.gitlab4j.api.GitLabApi;
import org.gitlab4j.api.GitLabApiException;
import org.gitlab4j.api.models.*;
import org.springframework.data.util.Pair;
import org.springframework.stereotype.Service;

/**
 * Service for retrieving group statistics from GitLab.
 */
@Service
public class GroupStatisticsService {

	/**
	 * Method that retrieves the total number of commits for the specified group.
	 *
	 * @param  gitLabApi the GitLabApi instance
	 * @param  groupPath the ID of the group
	 * @return           the total number of commits
	 */
	public int getTotalCommitsCount(GitLabApi gitLabApi, String groupPath) {
		try {
			return gitLabApi.getCommitsApi().getCommits(groupPath).size();
		} catch (GitLabApiException e) {
			// Handle any exceptions that occur during the API call
			e.printStackTrace();
			return -1; // Return -1 to indicate an error occurred
		}
	}

	/**
	 * Method that retrieves the number of issues in the specified group.
	 *
	 * @param  gitLabApi the GitLabApi instance
	 * @param  groupPath the ID of the group
	 * @return           the number of issues
	 */
	public int getIssuesCount(GitLabApi gitLabApi, String groupPath) {
		try {
			return gitLabApi.getIssuesApi().getIssues(groupPath).size();
		} catch (GitLabApiException e) {
			// Handle any exceptions that occur during the API call
			e.printStackTrace();
			return -1; // Return -1 to indicate an error occurred
		}
	}

	/**
	 * Method that retrieves the number of merge requests in the specified group.
	 *
	 * @param  gitLabApi the GitLabApi instance
	 * @param  groupPath the ID of the group
	 * @return           the number of merge requests
	 */
	public int getMergeRequestsCount(GitLabApi gitLabApi, String groupPath) {
		try {
			return gitLabApi.getMergeRequestApi().getMergeRequests(groupPath).size();
		} catch (GitLabApiException e) {
			// Handle any exceptions that occur during the API call
			e.printStackTrace();
			return -1; // Return -1 to indicate an error occurred
		}
	}

	/**
	 * Method that retrieves the names of participants in the specified group.
	 *
	 * @param  gitLabApi the GitLabApi instance
	 * @param  groupPath the ID of the group
	 * @return           the list of participant names
	 */
	public List<String> getParticipantsNames(GitLabApi gitLabApi, String groupPath) {
		try {
			Group group = gitLabApi.getGroupApi().getGroup(groupPath);

			return gitLabApi.getGroupApi()
					.getMembers(group)
					.stream()
					.map(x -> x.getName())
					.collect(Collectors.toList());
		} catch (GitLabApiException e) {
			// Handle any exceptions that occur during the API call
			e.printStackTrace();
			return null; // Return null to indicate an error occurred
		}
	}

	/**
	 * Method that retrieves the lines of code contributed by each user in the specified repository.
	 *
	 * @param  gitLabApi      the GitLabApi instance
	 * @param  repositoryPath the path of the repository
	 * @return                a map of usernames to their lines of code count
	 */
	public Map<String, Pair<Integer, Integer>> getUsersLinesOfCodeCount(GitLabApi gitLabApi,
			String repositoryPath) {
		try {
			Map<String, Pair<Integer, Integer>> userLinesOfCodeCount = new HashMap<>();

			List<MergeRequest> mergeRequests = gitLabApi.getMergeRequestApi()
					.getMergeRequests(repositoryPath);
			String masterBranch = gitLabApi.getProjectApi().getProject(repositoryPath).getDefaultBranch();

			List<MergeRequest> correctMRs = mergeRequests.stream()
					.filter(x -> x.getTargetBranch().equals(masterBranch))
					.collect(Collectors.toList());

			for (MergeRequest mr : correctMRs) {
				List<Commit> commits = gitLabApi.getMergeRequestApi().getCommits(repositoryPath,
						mr.getIid().intValue());

				for (Commit commit : commits) {
					Commit fullcommit = gitLabApi.getCommitsApi().getCommit(repositoryPath, commit.getId());
					CommitStats changes = fullcommit.getStats();
					String userName = fullcommit.getAuthorName();

					if (changes != null) {
						int linesOfCodeCountAdded = changes.getAdditions();
						int linesOfCodeCountRemoved = changes.getDeletions();

						Pair<Integer, Integer> oldPair = userLinesOfCodeCount.getOrDefault(userName,
								Pair.of(0, 0));
						Pair<Integer, Integer> newPair = Pair.of(oldPair.getFirst() + linesOfCodeCountAdded,
								oldPair.getSecond() + linesOfCodeCountRemoved);

						userLinesOfCodeCount.put(userName, newPair);
					}

				}

			}

			return userLinesOfCodeCount;
		} catch (GitLabApiException e) {
			e.printStackTrace();
			return null;
		}
	}

}
