package server.service.statisticsService;

import java.util.List;
import java.util.stream.Collectors;

import org.gitlab4j.api.GitLabApi;
import org.gitlab4j.api.GitLabApiException;
import org.gitlab4j.api.models.*;
import org.springframework.stereotype.Service;

/**
 * Service for retrieving statistics for a specific person from GitLab.
 */
@Service
public class IndividualStatisticsService {

	/**
	 * Retrieves the number of commits made by the specified user in the given repository.
	 *
	 * @param  gitLabApi      the GitLabApi instance
	 * @param  repositoryPath the path of the repository
	 * @param  userName       the Name of the user
	 * @return                the number of commits made by the user in the repository
	 */
	public int getUserCommitCount(GitLabApi gitLabApi, String repositoryPath, String userName) {
		try {
			return (int) gitLabApi.getCommitsApi().getCommits(repositoryPath).stream()
					.filter(x -> x.getAuthorName().equals(userName)).count();
		} catch (GitLabApiException e) {
			e.printStackTrace();
			return -1;
		}
	}

	/**
	 * Retrieves the number of merge requests assigned by the specified user in the given repository.
	 *
	 * @param  gitLabApi      the GitLabApi instance
	 * @param  repositoryPath the path of the repository
	 * @param  userName       the Name of the user
	 * @return                the number of merge requests created by the user in the repository
	 */
	public int getUserMergeRequestsCount(GitLabApi gitLabApi, String repositoryPath, String userName) {
		try {
			return (int) gitLabApi
					.getMergeRequestApi()
					.getMergeRequests(repositoryPath)
					.stream()
					.filter(x -> x.getAssignees().stream().map(y -> y.getName()).collect(Collectors.toList())
							.contains(userName))
					.count();
		} catch (GitLabApiException e) {
			e.printStackTrace();
			return -1;
		}
	}

	/**
	 * Retrieves the number of issues assigned to the specified user in the given repository.
	 *
	 * @param  gitLabApi      the GitLabApi instance
	 * @param  repositoryPath the path of the repository
	 * @param  userName       the Name of the user
	 * @return                the number of issues created by the user in the repository
	 */
	public int getUserIssuesCount(GitLabApi gitLabApi, String repositoryPath, String userName) {
		try {
			return (int) gitLabApi
					.getIssuesApi()
					.getIssues(repositoryPath)
					.stream()
					.filter(x -> x.getAssignees().stream().map(y -> y.getName()).collect(Collectors.toList())
							.contains(userName))
					.count();
		} catch (GitLabApiException e) {
			e.printStackTrace();
			return -1;
		}
	}

	public int getUserCommentsCountOnMerge(GitLabApi gitLabApi, String repositoryPath, String userName) {
		try {
			int totalComments = 0;

			// Get the list of merge requests for the repository
			List<MergeRequest> mergeRequests = gitLabApi.getMergeRequestApi()
					.getMergeRequests(repositoryPath);

			// Iterate over each merge request
			for (MergeRequest mergeRequest : mergeRequests) {
				// Get the list of comments for the merge request
				List<Note> comments = gitLabApi.getNotesApi().getMergeRequestNotes(repositoryPath,
						mergeRequest.getIid());
				// Iterate over each comment
				for (Note comment : comments) {
					// Check if the comment author matches the provided userName
					if (comment.getAuthor().getName().equals(userName)) {
						totalComments++;
					}
				}
			}

			return totalComments;
		} catch (GitLabApiException e) {
			e.printStackTrace();
			return -1;
		}
	}
}
