package server.statistics;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import server.statistics.util.StreamConsumer;

/**
 * The CommandRunner class is responsible for executing command-line scripts or commands and managing their
 * output and logging.
 */
@Component
public class CommandRunner {

	@Value("${gitbull.statistics.logging-level}")
	private String logging;

	/**
	 * Executes a script or command provided as a list of command-line arguments.
	 *
	 * @param  cmd                  The list of command-line arguments representing the script or command to
	 *                              execute.
	 * @throws IOException          If an I/O error occurs.
	 * @throws InterruptedException If the execution is interrupted.
	 */
	public void runScript(List<String> cmd) throws IOException, InterruptedException {
		ArrayList<List<String>> commands = new ArrayList<>();
		String runner = System.getProperty("os.name").startsWith("Win") ? "wsl.exe" : "sh";

		for (String command : cmd) {
			commands.add(Stream.concat(Stream.of(runner), Stream.of(command.split(" ")))
					.collect(Collectors.toList()));
		}

		runConsequently(commands);
	}

	/**
	 * Executes a list of commands consequently.
	 *
	 * @param  commands             The list of commands to execute.
	 * @throws IOException          If an I/O error occurs.
	 * @throws InterruptedException If the execution is interrupted.
	 */
	private void runConsequently(List<List<String>> commands)
			throws IOException, InterruptedException {
		ProcessBuilder builder = new ProcessBuilder();
		builder.directory(new File("./statistics"));

		StreamConsumer sc;
		Process process;

		for (List<String> cmd : commands) {
			if (cmd.get(1).equals("MOVETO")) {
				builder.directory(new File("./statistics" + (cmd.get(1) == null ? "" : cmd.get(2))));
			}

			builder.command(cmd);

			try {
				process = builder.start();
			} catch (IOException e) {
				checkWSL(e);
				return;
			}

			//if logging output of commands, then print their outputs. otherwise, shreds the output of commands.
			Consumer<String> consumer = logging.equals("FULL") || logging.equals("OUTPUT")
					? System.out::println
					: (x) -> {
					};
			sc = new StreamConsumer(process.getInputStream(), consumer);

			ExecutorService executorService = Executors.newSingleThreadExecutor();
			Future<?> future = executorService.submit(sc);

			process.waitFor();
			if (logging.equals("FULL") || logging.equals("LOG")) {
				System.out.println("Succesfully ran: " + String.join(" ", cmd));
			}
		}
	}

	/**
	 * Checks if the IOException is caused by a missing Windows Subsystem for Linux (WSL) or other required
	 * dependencies. If WSL is missing, an informative message is printed; otherwise, the original exception
	 * is thrown.
	 *
	 * @param  e           The IOException to check.
	 * @throws IOException If WSL is not installed and other required dependencies are missing.
	 */
	static void checkWSL(IOException e) throws IOException {
		if (e.getMessage().startsWith("Cannot run program")) {
			System.out.println(
					"""
											\n============================ STATISTIC SCRIPT INSTALL ============================
							Windows Subsystem for Linux, git and Python are needed for using statistics scripts.
							Install WSL and try running the program again: aka.ms/wsl
							Also install git and Python on your machine.
							Installation of GitInspector and Geit has been aborted.
							==================================================================================
							""");
		} else {
			throw e;
		}
	}
}
