package server.statistics;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Stream;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import server.statistics.util.StatisticsEntry;
import server.util.FancyPrint;

/**
 * Class that runs the statistics for repositories.
 */
@Component
public class StatisticsRunner implements Runnable {

	@Value("${gitbull.statistics.username}")
	private String username;

	@Value("${gitbull.statistics.api-key}")
	private String token;

	@Value("${gitbull.statistics.gitlab-url}")
	private String gitlabUrl;

	@Autowired
	private CommandRunner commandRunner;

	/**
	 * A list of repositories and weeks left to run statistics for.
	 */
	private List<StatisticsEntry> repositories;

	/**
	 * Get repositories. Not used outside testing, for now.
	 *
	 * @return the list of repositories currently tracked
	 */
	public List<StatisticsEntry> getRepositories() {
		return repositories;
	}

	/**
	 * Adds a repository to the list of tracked repositories with the specified path and duration.
	 *
	 * @param path  The path of the repository to be added. If the path does not start with '/', it is
	 *              prepended with '/'.
	 * @param weeks The duration in weeks for which the repository will be tracked.
	 */
	public void addRepository(String path, Integer weeks) {
		if (path.charAt(0) != '/') {
			path = "/" + path;
		}
		FancyPrint.println("statistics", "Now tracking " + path + " for " + weeks + " weeks.");
		if (weeks > 0) {
			repositories.add(new StatisticsEntry(path, weeks));
		}
	}

	public StatisticsRunner(CommandRunner commandRunner) {
		this.commandRunner = commandRunner;
		repositories = new ArrayList<>();
	}

	/**
	 * Executes the analysis for each repository in the list of repositories. This method is invoked when the
	 * `StatisticsRunner` is started as a separate thread. It performs the following steps for each
	 * repository: - Analyzes the repository by calling the `analyzeRepository` method. - Checks if the
	 * analysis period for the repository has ended and marks it for deletion if necessary. - Removes the
	 * marked repositories from the list of repositories.
	 */
	@Override
	public void run() {
		if (repositories.size() == 0) {
			return;
		}
		FancyPrint.println("scheduler", "Repository analysis started. Analysing " + repositories.size()
				+ " repositories.");
		ArrayList<StatisticsEntry> toBeDeleted = new ArrayList<>();

		for (StatisticsEntry repo : repositories) {
			List<String> courseDetails = Stream.of(repo.getCourseName().split("/")).toList();
			try {
				analyzeRepository(repo.getCourseName(), repo.getAtWeek(), courseDetails.get(1),
						courseDetails.get(2));
				FancyPrint.println("scheduler", "Succesfully analysed " + repo.getCourseName());
				if (repo.countWeek()) {
					FancyPrint.println("scheduler",
							"The analysis period for " + repo.getCourseName() + " has ended.");
					toBeDeleted.add(repo);
				}
			} catch (IOException | InterruptedException e) {
				throw new RuntimeException(e);
			}
		}

		//have to first mark for deletion, then delete after iteration
		//to avoid a ConcurrentModificationException
		for (StatisticsEntry repo : toBeDeleted) {
			repositories.remove(repo);
		}
	}

	/**
	 * Analyzes a repository by performing the following steps: - Clones the repository from the GitLab
	 * server. - Runs analysis using the Geit tool. - Deletes the cloned repository. - Moves the generated
	 * analysis results to the appropriate directory.
	 *
	 * @param  pathToRepo           The path to the repository to be analyzed.
	 * @param  weekNumber           The current week number for analysis.
	 * @param  courseName           The name of the course associated with the repository.
	 * @param  courseEdition        The edition of the course associated with the repository.
	 * @return                      A list of commands executed during the analysis process.
	 * @throws IOException          If an I/O error occurs.
	 * @throws InterruptedException If the current thread is interrupted while waiting for the analysis
	 *                              process to complete.
	 */
	public ArrayList<String> analyzeRepository(String pathToRepo, int weekNumber, String courseName,
			String courseEdition)
			throws IOException, InterruptedException {
		ArrayList<String> commands = new ArrayList<>();

		/*
		 * Clone repo
		 * gitlabUrl contains the 'https://', but we already added it before the username and token
		 * hence the call to .substring(8), which removes the first 8 characters ('https://')
		*/
		commands.add("MOVETO /repo");
		commands.add(
				"git clone https://" + username + ":" + token + "@" + gitlabUrl.substring(8) + pathToRepo);

		String[] splitRepo = pathToRepo.split("/");
		String repoName = splitRepo[splitRepo.length - 1];

		//only running Geit for now
		//GitInspector seems to require an older version of Python (3.7)
		//how do we even ensure that? i'm not doing Docker.
		//we'll skip it for now.

		//run analysis
		commands.add("MOVETO /python/geit");
		commands.add("python3 geit.py --target-repo ./../../repo/" + repoName);

		//delete repo
		commands.add("MOVETO /repo");
		commands.add("rm -r " + repoName);

		commands.add("MOVETO /");
		commands.add("mkdir results");

		commands.add("mv ./python/geit/index_" + repoName + ".html ./results/" + courseName + "_"
				+ courseEdition + "_" + repoName.toLowerCase() + "_" + repoName + "_" + weekNumber + ".html");

		commandRunner.runScript(commands);
		return commands;
	}
}
