package server.statistics;

import javax.annotation.PostConstruct;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import server.util.FancyPrint;
import it.sauronsoftware.cron4j.InvalidPatternException;
import it.sauronsoftware.cron4j.Scheduler;

/**
 * Class that schedules the statistics.
 */
@Component
public class StatisticsScheduler {

	@Autowired
	private StatisticsRunner statisticsRunner;

	@Value("${gitbull.statistics.enable-scheduler}")
	private boolean enabled;

	@Value("${gitbull.statistics.schedule}")
	private String schedule;

	private Scheduler scheduler;

	/**
	 * Cron4j uses a string pattern to define the schedule. Here are some basic preset patterns for testing.
	 * Full documentation w/ examples
	 * <a href="http://www.sauronsoftware.it/projects/cron4j/manual.php">here.</a>
	 *
	 * Set the actual value to be used in application.properties.
	 */
	private final String REAL_everyMondayAt2AM = "* 2 * * Mon";
	private final String TESTING_everyMinute = "* * * * *";
	private final String TESTING_everyEvenMinute = "*/2 * * * *";
	private final String TESTING_every5Minutes = "*/5 * * * *";

	//run a task at any minute (0-59)
	//for example, if you set this to 36
	//it will run at 12:36, 13:36, 14:36, etc;
	private final int TESTING_minuteToRunTaskAt = 5;
	private final String TESTING_whateverTime = TESTING_minuteToRunTaskAt + " * * * *";

	/**
	 * Initializes the statistics scheduler and schedules the statistics runner task if enabled.
	 */
	@PostConstruct
	public void init() {
		if (!enabled) {
			FancyPrint.println("scheduler",
					"The statistics scheduler is turned off. To use it, enable it in application.properties.");
			return;
		}
		FancyPrint.println("scheduler", "Started statistics scheduler.");

		try {
			scheduler.schedule(schedule, statisticsRunner);
			scheduler.start();
		} catch (InvalidPatternException e) {
			FancyPrint.println("scheduler",
					"Invalid schedule pattern in application.properties. The statistics module has been turned off.");
		}
	}

	public StatisticsScheduler(StatisticsRunner statisticsRunner) {
		this.statisticsRunner = statisticsRunner;
		this.scheduler = new Scheduler();
	}

}
