package server.controller;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;

import java.util.ArrayList;
import java.util.List;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;

import nl.tudelft.labracore.lib.security.user.DefaultRole;
import nl.tudelft.labracore.lib.security.user.Person;

import org.gitlab4j.api.GitLabApi;
import org.gitlab4j.api.UserApi;
import org.gitlab4j.api.models.User;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.*;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.core.io.FileSystemResource;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.multipart.MultipartFile;

import server.entity.Access;
import server.entity.CourseEdition;
import server.entity.StudentGroup;
import server.model.ApiKeyDTO;
import server.model.GeitStatisticsRequestDTO;
import server.model.UserCourseInfoDTO;
import server.security.GitBullSecurity;
import server.service.*;

@ExtendWith(MockitoExtension.class)
public class FrontendControllerTest {
	@InjectMocks
	private FrontendController frontendController;

	@Mock
	private HttpServletRequest request;

	@Mock
	private AccessService accessService;

	@Mock
	private AccountService accountService;

	@Mock
	private InstanceMappingService instanceMappingService;

	@Mock
	private CourseEditionService courseService;

	@Mock
	private GroupService groupService;

	@Mock
	private GitBullSecurity gitBullSecurity;

	@Mock
	private GitLabApi gitLabApi;

	@Mock
	private UserApi userApi;

	@BeforeEach
	public void setup() {
		MockitoAnnotations.openMocks(this);
	}

	@Test
	public void testWhoami_ValidSecret_ReturnsDisplayName() {
		String secret = "valid_secret";
		String decodedSecret = "decoded_secret";
		String displayName = "John Doe";
		Person authenticatedUser = new Person();
		authenticatedUser.setDisplayName(displayName);

		when(gitBullSecurity.decodeSecretString(secret)).thenReturn(decodedSecret);
		when(gitBullSecurity.getAuthenticatedUser(decodedSecret)).thenReturn(authenticatedUser);

		ResponseEntity<String> response = frontendController.whoami(secret);

		assertEquals(HttpStatus.OK, response.getStatusCode());
		assertEquals("From GitBull security via secret (proper): " + displayName, response.getBody());
	}

	@Test
	public void testWhoami_InvalidSecret_ReturnsUnauthorized() {
		String secret = "invalid_secret";
		String decodedSecret = "decoded_secret";

		when(gitBullSecurity.decodeSecretString(secret)).thenReturn(decodedSecret);
		when(gitBullSecurity.getAuthenticatedUser(decodedSecret)).thenReturn(null);

		ResponseEntity<String> response = frontendController.whoami(secret);

		assertEquals(HttpStatus.UNAUTHORIZED, response.getStatusCode());
		assertNull(response.getBody());
	}

	@Test
	public void testCheckConnection_ReturnsOk() {
		ResponseEntity<String> response = frontendController.checkConnection();

		assertEquals(HttpStatus.OK, response.getStatusCode());
		assertEquals("Hello from GitBull!", response.getBody());
	}

	@Test
	void getFile_ShouldReturnBadRequestWhenInvalidWeek() {
		GeitStatisticsRequestDTO dto = new GeitStatisticsRequestDTO();
		dto.setRepoPath("repo/path");
		dto.setWeek(0);

		ResponseEntity<FileSystemResource> response = frontendController.getFile(dto);

		assertEquals(HttpStatus.BAD_REQUEST, response.getStatusCode());
		assertNull(response.getBody());
	}

	@Test
	void getFile_ShouldReturnFileSystemResourceWithProperHeaders() {
		String currentUser = "john.doe";
		String repoPath = "repo/path";
		Integer week = 5;

		GeitStatisticsRequestDTO dto = new GeitStatisticsRequestDTO();
		dto.setSecret("secret");
		dto.setRepoPath(repoPath);
		dto.setWeek(week);

		when(gitBullSecurity.getCurrentUsername(any(String.class))).thenReturn(currentUser);
		when(instanceMappingService.getInstance(any(String.class))).thenReturn(gitLabApi);

		FileSystemResource fileSystemResource = mock(FileSystemResource.class);

		HttpHeaders expectedHeaders = new HttpHeaders();
		expectedHeaders.add("Content-Disposition", "attachment; filename=\"file.html\"");

		ResponseEntity<FileSystemResource> expectedResponse = ResponseEntity.status(HttpStatus.OK)
				.headers(expectedHeaders)
				.body(fileSystemResource);

		ResponseEntity<FileSystemResource> response = frontendController.getFile(dto);

		assertEquals(expectedResponse.getStatusCode(), response.getStatusCode());
		assertEquals(expectedHeaders, response.getHeaders());
		assertNotNull(response.getBody());
	}

	@Test
	void handleFileUpload_ShouldReturnOkWithSuccessMessage() {
		MultipartFile file = mock(MultipartFile.class);
		when(request.getServletContext()).thenReturn(mock(ServletContext.class));
		when(request.getServletContext().getRealPath("/uploads/")).thenReturn("uploads/");

		ResponseEntity<String> response = frontendController.handleFileUpload(file);

		assertEquals(HttpStatus.OK, response.getStatusCode());
		assertEquals("File uploaded", response.getBody());
	}

	@Test
	void isLoggedIn_ShouldReturnTrueWhenAuthenticatedUserExists() {
		String secret = "secret";
		String decodedSecret = "decodedSecret";
		Person person = new Person();

		doReturn(decodedSecret).when(gitBullSecurity).decodeSecretString(secret);
		doReturn(person).when(gitBullSecurity).getAuthenticatedUser(decodedSecret);

		ResponseEntity<Boolean> response = frontendController.isLoggedIn(secret);

		assertEquals(HttpStatus.OK, response.getStatusCode());
		assertTrue(response.getBody());
	}

	@Test
	void isLoggedIn_ShouldReturnFalseWhenAuthenticatedUserDoesNotExist() {
		String secret = "secret";
		String decodedSecret = "decodedSecret";

		doReturn(decodedSecret).when(gitBullSecurity).decodeSecretString(secret);
		doReturn(null).when(gitBullSecurity).getAuthenticatedUser(decodedSecret);

		ResponseEntity<Boolean> response = frontendController.isLoggedIn(secret);

		assertEquals(HttpStatus.OK, response.getStatusCode());
		assertFalse(response.getBody());
	}

	@Test
	void getRole_ShouldReturnRoleWhenAuthenticatedUserExists() {
		String secret = "secret";
		String decodedSecret = "decodedSecret";
		Person person = new Person();
		person.setDefaultRole(DefaultRole.ADMIN);

		when(gitBullSecurity.decodeSecretString(secret)).thenReturn(decodedSecret);
		when(gitBullSecurity.getAuthenticatedUser(decodedSecret)).thenReturn(person);

		ResponseEntity<String> response = frontendController.getRole(secret);

		assertEquals(HttpStatus.OK, response.getStatusCode());
		assertEquals(DefaultRole.ADMIN.toString(), response.getBody());
	}

	@Test
	public void testGetCourses_ValidSecret_ReturnsUserCourseInfoDTOList() {
		String secret = "valid_secret";
		String decodedSecret = "decoded_secret";
		String netId = "johndoe";
		Long courseId = 1L;
		Long groupId = 2L;
		String groupPath = "group_path";
		String courseName = "Course";
		String courseEdition = "2023";
		String coursePath = "course_path";
		int role = 1;

		Access access = new Access();
		access.setCourseEditionID(courseId);
		access.setGroupID(groupId);
		access.setRole(role);

		CourseEdition course = new CourseEdition();
		course.setCourseName(courseName);
		course.setEdition(courseEdition);
		course.setCoursePath(coursePath);

		StudentGroup group = new StudentGroup();
		group.setGroupPath(groupPath);

		List<Access> accesses = new ArrayList<>();
		accesses.add(access);

		List<Long> courseIds = new ArrayList<>();
		courseIds.add(courseId);

		List<CourseEdition> courses = new ArrayList<>();
		courses.add(course);

		List<Long> groupIds = new ArrayList<>();
		groupIds.add(groupId);

		List<StudentGroup> groups = new ArrayList<>();
		groups.add(group);

		when(gitBullSecurity.decodeSecretString(secret)).thenReturn(decodedSecret);
		when(gitBullSecurity.getCurrentUsername(decodedSecret)).thenReturn(netId);
		when(accessService.fetchAllAccessesOfAccount(netId)).thenReturn(accesses);
		when(courseService.fetchAllCoursesOfListOfIds(courseIds)).thenReturn(courses);
		when(groupService.fetchAllGroupsFromId(groupIds)).thenReturn(groups);

		ResponseEntity<List<UserCourseInfoDTO>> response = frontendController.getCourses(secret);

		assertEquals(HttpStatus.OK, response.getStatusCode());
		assertNotNull(response.getBody());
		assertEquals(1, response.getBody().size());

		UserCourseInfoDTO uci = response.getBody().get(0);
		assertEquals(courseName, uci.getCourseName());
		assertEquals(courseEdition, uci.getCourseEdition());
		assertEquals(coursePath, uci.getCoursePath());
		assertEquals("STUDENT", uci.getRole());
		assertEquals(groupPath, uci.getGroupPath());

		role = 2;
		access.setRole(role);

		when(accessService.fetchAllAccessesOfAccount(netId)).thenReturn(accesses);

		response = frontendController.getCourses(secret);

		assertEquals(HttpStatus.OK, response.getStatusCode());
		assertNotNull(response.getBody());
		assertEquals(1, response.getBody().size());

		uci = response.getBody().get(0);
		assertEquals(courseName, uci.getCourseName());
		assertEquals(courseEdition, uci.getCourseEdition());
		assertEquals(coursePath, uci.getCoursePath());
		assertEquals("TA", uci.getRole());
		assertNull(uci.getGroupPath());

		// Test case 3: Valid secret with role TEACHER
		role = 3;
		access.setRole(role);

		when(accessService.fetchAllAccessesOfAccount(netId)).thenReturn(accesses);

		response = frontendController.getCourses(secret);

		assertEquals(HttpStatus.OK, response.getStatusCode());
		assertNotNull(response.getBody());
		assertEquals(1, response.getBody().size());

		uci = response.getBody().get(0);
		assertEquals(courseName, uci.getCourseName());
		assertEquals(courseEdition, uci.getCourseEdition());
		assertEquals(coursePath, uci.getCoursePath());
		assertEquals("TEACHER", uci.getRole());
		assertNull(uci.getGroupPath());
	}

	@Test
	public void testGetCourses_InvalidSecret_ReturnsUnauthorized() {
		String secret = "invalid_secret";
		String decodedSecret = "decoded_secret";

		when(gitBullSecurity.decodeSecretString(secret)).thenReturn(decodedSecret);
		when(gitBullSecurity.getCurrentUsername(decodedSecret)).thenReturn(null);

		ResponseEntity<List<UserCourseInfoDTO>> response = frontendController.getCourses(secret);

		assertEquals(HttpStatus.UNAUTHORIZED, response.getStatusCode());
		assertNull(response.getBody());
	}

	@Test
	public void testAddKey() {
		String secret = "testSecret";
		String apiKey = "testApiKey";
		String currentUser = "testUser";

		ApiKeyDTO apiKeyDTO = new ApiKeyDTO();
		apiKeyDTO.setSecret(secret);
		apiKeyDTO.setApiKey(apiKey);

		when(gitBullSecurity.getCurrentUsername(secret)).thenReturn(currentUser);

		ResponseEntity<Void> response = frontendController.addKey(apiKeyDTO);

		verify(gitBullSecurity).getCurrentUsername(secret);
		verify(accountService).addAPIKey(currentUser, apiKey);
		verify(instanceMappingService).addInstance(currentUser, apiKey);

		assertEquals(HttpStatus.OK, response.getStatusCode());
	}

	@Test
	public void testAddKeyUnauthorized() {
		String secret = "testSecret";
		String apiKey = "testApiKey";
		String currentUser = "";

		ApiKeyDTO apiKeyDTO = new ApiKeyDTO();
		apiKeyDTO.setSecret(secret);
		apiKeyDTO.setApiKey(apiKey);

		when(gitBullSecurity.getCurrentUsername(secret)).thenReturn(currentUser);

		ResponseEntity<Void> response = frontendController.addKey(apiKeyDTO);

		verify(gitBullSecurity).getCurrentUsername(secret);
		verifyNoInteractions(accountService);
		verifyNoInteractions(instanceMappingService);

		assertEquals(HttpStatus.UNAUTHORIZED, response.getStatusCode());
	}

	@Test
	public void testAddKeyBadRequest() {
		String secret = "testSecret";
		String apiKey = "";

		ApiKeyDTO apiKeyDTO = new ApiKeyDTO();
		apiKeyDTO.setSecret(secret);
		apiKeyDTO.setApiKey(apiKey);

		ResponseEntity<Void> response = frontendController.addKey(apiKeyDTO);

		assertEquals(HttpStatus.BAD_REQUEST, response.getStatusCode());
	}

	@Test
	public void testRemoveKey() {
		String secret = "testSecret";
		String currentUser = "testUser";
		GitLabApi gitLabApi = mock(GitLabApi.class);

		ApiKeyDTO apiKeyDTO = new ApiKeyDTO();
		apiKeyDTO.setSecret(secret);

		when(gitBullSecurity.getCurrentUsername(secret)).thenReturn(currentUser);
		when(instanceMappingService.getInstance(currentUser)).thenReturn(gitLabApi);

		ResponseEntity<String> response = frontendController.removeKey(apiKeyDTO);

		verify(gitBullSecurity).getCurrentUsername(secret);
		verify(instanceMappingService).getInstance(currentUser);
		verify(accountService).removeAPIKey(currentUser);
		verify(instanceMappingService).removeInstance(currentUser);

		assertEquals(HttpStatus.OK, response.getStatusCode());
		assertEquals("API key successfully removed.", response.getBody());
	}

	@Test
	public void testRemoveKeyUnauthorized() {
		String secret = "testSecret";
		String currentUser = "";

		ApiKeyDTO apiKeyDTO = new ApiKeyDTO();
		apiKeyDTO.setSecret(secret);

		when(gitBullSecurity.getCurrentUsername(secret)).thenReturn(currentUser);

		ResponseEntity<String> response = frontendController.removeKey(apiKeyDTO);

		verify(gitBullSecurity).getCurrentUsername(secret);
		verifyNoInteractions(instanceMappingService);
		verifyNoInteractions(accountService);

		assertEquals(HttpStatus.UNAUTHORIZED, response.getStatusCode());
		assertEquals(null, response.getBody());
	}

	@Test
	public void testRemoveKeyNotFound() {
		String secret = "testSecret";
		String currentUser = "testUser";

		ApiKeyDTO apiKeyDTO = new ApiKeyDTO();
		apiKeyDTO.setSecret(secret);

		when(gitBullSecurity.getCurrentUsername(secret)).thenReturn(currentUser);
		when(instanceMappingService.getInstance(currentUser)).thenReturn(null);

		ResponseEntity<String> response = frontendController.removeKey(apiKeyDTO);

		verify(gitBullSecurity).getCurrentUsername(secret);
		verify(instanceMappingService).getInstance(currentUser);
		verifyNoInteractions(accountService);

		assertEquals(HttpStatus.OK, response.getStatusCode());
		assertEquals("API key not found.", response.getBody());
	}

	@Test
	public void testKeyExists() throws Exception {
		String secret = "testSecret";
		String currentUser = "testUser";

		ApiKeyDTO apiKeyDTO = new ApiKeyDTO();
		apiKeyDTO.setSecret(secret);

		when(gitBullSecurity.getCurrentUsername(secret)).thenReturn(currentUser);
		when(instanceMappingService.getInstance(currentUser)).thenReturn(gitLabApi);
		when(gitLabApi.getUserApi()).thenReturn(userApi);
		User user = new User();
		user.setUsername("cool");
		when(userApi.getCurrentUser()).thenReturn(user);
		when(gitLabApi.getUserApi()).thenReturn(userApi);

		ResponseEntity<Boolean> response = frontendController.keyExists(apiKeyDTO);

		verify(gitBullSecurity).getCurrentUsername(secret);
		verify(instanceMappingService).getInstance(currentUser);
		verify(gitLabApi).getUserApi();
		verify(userApi).getCurrentUser();

		assertEquals(HttpStatus.OK, response.getStatusCode());
		assertEquals(true, response.getBody());
	}

	@Test
	public void testKeyExistsUnauthorized() {
		String secret = "testSecret";
		String currentUser = "";

		ApiKeyDTO apiKeyDTO = new ApiKeyDTO();
		apiKeyDTO.setSecret(secret);

		when(gitBullSecurity.getCurrentUsername(secret)).thenReturn(currentUser);

		ResponseEntity<Boolean> response = frontendController.keyExists(apiKeyDTO);

		verify(gitBullSecurity).getCurrentUsername(secret);
		verifyNoInteractions(instanceMappingService);
		verifyNoInteractions(gitLabApi);

		assertEquals(HttpStatus.UNAUTHORIZED, response.getStatusCode());
		assertEquals(null, response.getBody());
	}

	@Test
	public void testGetCourses_AccessesCoursesMismatch_ReturnsNotFound() {
		String secret = "valid_secret";
		String decodedSecret = "decoded_secret";
		String netId = "johndoe";
		Long courseId = 1L;
		Long groupId = 2L;
		String groupPath = "group_path";

		Access access = new Access();
		access.setCourseEditionID(courseId);
		access.setGroupID(groupId);
		access.setRole(1);

		List<Access> accesses = new ArrayList<>();
		accesses.add(access);

		List<Long> courseIds = new ArrayList<>();
		courseIds.add(courseId);

		List<CourseEdition> courses = new ArrayList<>();

		List<Long> groupIds = new ArrayList<>();
		groupIds.add(groupId);

		StudentGroup group = new StudentGroup();
		group.setGroupPath(groupPath);

		List<StudentGroup> groups = new ArrayList<>();
		groups.add(group);

		when(gitBullSecurity.decodeSecretString(secret)).thenReturn(decodedSecret);
		when(gitBullSecurity.getCurrentUsername(decodedSecret)).thenReturn(netId);
		when(accessService.fetchAllAccessesOfAccount(netId)).thenReturn(accesses);
		when(courseService.fetchAllCoursesOfListOfIds(courseIds)).thenReturn(courses);
		when(groupService.fetchAllGroupsFromId(groupIds)).thenReturn(groups);

		ResponseEntity<List<UserCourseInfoDTO>> response = frontendController.getCourses(secret);

		assertEquals(HttpStatus.NOT_FOUND, response.getStatusCode());
		assertNull(response.getBody());
	}

	//
	@Test
	void testDeleteEdition_DeletedCourseEditionAndAccess_ReturnsDeletedEdition() {
		String courseName = "CourseA";
		String edition = "2023";
		String secret = "valid_secret";
		String decodedSecret = "decoded_secret";
		String netId = "johndoe";

		CourseEdition deletedEdition = CourseEdition.builder()
				.courseEditionID(1L)
				.courseName(courseName)
				.edition(edition)
				.build();

		Access accessDeleted = new Access();
		accessDeleted.setAccessID(1L);
		accessDeleted.setCourseEditionID(deletedEdition.getCourseEditionID());

		List<Access> userAccesses = new ArrayList<>();
		userAccesses.add(accessDeleted);

		when(gitBullSecurity.decodeSecretString(secret)).thenReturn(decodedSecret);
		when(gitBullSecurity.getCurrentUsername(decodedSecret)).thenReturn(netId);
		when(courseService.deleteCourseEdition(courseName, edition)).thenReturn(deletedEdition);
		when(accessService.fetchAllAccessesOfAccount(netId)).thenReturn(userAccesses);

		ResponseEntity response = frontendController.deleteEdition(courseName, edition, secret);

		verify(courseService).deleteCourseEdition(courseName, edition);
		verify(accessService).deleteAccess(accessDeleted.getAccessID());

		assertEquals(HttpStatus.OK, response.getStatusCode());
		assertEquals(deletedEdition, response.getBody());
	}

}
