package server.controller;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.*;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.asyncDispatch;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.content;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import static org.springframework.test.web.servlet.setup.MockMvcBuilders.standaloneSetup;

import java.util.*;

import nl.tudelft.labracore.api.dto.PersonDetailsDTO;
import nl.tudelft.labracore.api.dto.PersonSummaryDTO;

import org.gitlab4j.api.GitLabApi;
import org.gitlab4j.api.GitLabApiException;
import org.gitlab4j.api.models.*;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;
import org.springframework.context.annotation.PropertySource;
import org.springframework.data.util.Pair;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.MvcResult;
import org.springframework.test.web.servlet.request.MockMvcRequestBuilders;
import org.springframework.test.web.servlet.result.MockMvcResultMatchers;

import server.csvconverter.CSVService;
import server.entity.Account;
import server.entity.CourseEdition;
import server.entity.CourseName;
import server.entity.StudentGroup;
import server.labraService.LabraConnectionService;
import server.logging.LogType;
import server.logging.LogWriter;
import server.model.*;
import server.security.GitBullSecurity;
import server.service.*;

import com.fasterxml.jackson.databind.ObjectMapper;

@PropertySource("classpath:application-test.properties")
public class GitLabControllerTest {
	private MockMvc mockMvc;
	private ObjectMapper objectMapper;

	@Mock
	private ProjectAPIService projectAPIService;

	@Mock
	private GroupAPIService groupAPIService;
	@Mock
	private GitLabApi gitLabApi;

	@Mock
	private UserAPIService userAPIService;

	@Mock
	private LabraConnectionService labraConnectionService;

	@Mock
	private InstanceMappingService instanceMappingService;

	@Mock
	private CourseNameService courseNameService;

	@Mock
	private GroupService groupDBService;

	@Mock
	private CourseEditionService courseEditionService;

	@Mock
	private AccessService accessService;

	@Mock
	private LogWriter logWriter;

	@Mock
	private CSVService csvService;

	@Mock
	private GitBullSecurity gitBullSecurity;

	@Spy
	@InjectMocks
	private GitLabController gitLabController;

	@BeforeEach
	public void setUp() throws GitLabApiException {
		MockitoAnnotations.openMocks(this);
		mockMvc = standaloneSetup(gitLabController).build();
		objectMapper = new ObjectMapper();

		when(userAPIService.getCurrentUser(any(GitLabApi.class)))
				.thenReturn(new User().withUsername("test-instance"));
		when(instanceMappingService.getInstance("admin")).thenReturn(gitLabApi);
		when(gitBullSecurity.getCurrentUsername("secret")).thenReturn("admin");

		doNothing().when(logWriter).writeActionLog(any(LogType.class), any(LogType.class), any(String.class),
				any(String.class));
		doNothing().when(logWriter).writeActionLog(any(LogType.class), any(LogType.class), any(String.class),
				any(String.class), any(String.class));
	}

	@Test
	public void getCourseNames() throws Exception {
		mockMvc.perform(MockMvcRequestBuilders.post("/api/gitlab/getCourseNames"));
		verify(labraConnectionService).getAllCourseNames();

	}

	@Test
	public void createCourseTest() throws Exception {
		GroupRequestDTO course = new GroupRequestDTO()
				.builder()
				.name("Test group")
				.secret("secret")
				.description("This is a simple test")
				.build();

		Group group = new Group().withName("Test group").withDescription("This is a simple test");

		when(groupAPIService.createGroup(gitLabApi, group.getName(), group.getDescription(),
				group.getName().replaceAll("\\s", "").toLowerCase(), false,
				null)).thenReturn(group);

		when(courseNameService.saveCourseName(any(CourseName.class))).thenReturn(new CourseName());

		mockMvc.perform(MockMvcRequestBuilders.post("/api/gitlab/createCourse")
				.contentType(MediaType.APPLICATION_JSON)
				.content(objectMapper.writeValueAsString(course)))
				.andExpect(status().isCreated())
				.andExpect(MockMvcResultMatchers.jsonPath("$.name").value(course.getName()))
				.andExpect(MockMvcResultMatchers.jsonPath("$.description").value(course.getDescription()));

		verify(logWriter).writeActionLog(eq(LogType.INFO), eq(LogType.COURSE_CREATED), any(String.class),
				any(String.class));
	}

	@Test
	public void createCourseFail() throws Exception {
		GroupRequestDTO course = new GroupRequestDTO()
				.builder()
				.name("Test group")
				.secret("secret")
				.description("This is a simple test")
				.build();

		when(instanceMappingService.getInstance(any(String.class))).thenReturn(null);

		mockMvc.perform(MockMvcRequestBuilders.post("/api/gitlab/createCourse")
				.contentType(MediaType.APPLICATION_JSON)
				.content(objectMapper.writeValueAsString(course)))
				.andExpect(status().isForbidden());

		verify(logWriter, times(0)).writeActionLog(any(LogType.class), any(LogType.class), any(String.class),
				any(String.class));

		Group group = new Group().withName("Test group").withDescription("This is a simple test");

		when(groupAPIService.createGroup(gitLabApi, group.getName(), group.getDescription(),
				group.getName().replaceAll("\\s", "").toLowerCase(), false,
				null)).thenThrow(new GitLabApiException("Nope", 500));

		when(courseNameService.saveCourseName(any())).thenReturn(new CourseName());

		when(instanceMappingService.getInstance(any(String.class))).thenReturn(gitLabApi);

		mockMvc.perform(MockMvcRequestBuilders.post("/api/gitlab/createCourse")
				.contentType(MediaType.APPLICATION_JSON)
				.content(objectMapper.writeValueAsString(course)))
				.andExpect(status().isInternalServerError());

		verify(logWriter, times(1)).writeActionLog(eq(LogType.ERROR), any(LogType.class), any(String.class),
				any(String.class), eq("Nope"));
	}

	@Test
	public void createCourseEditionTest() throws Exception {
		GroupRequestDTO course = new GroupRequestDTO()
				.builder()
				.name("Hard Times")
				.description("Gonna take you down and laugh when you cry")
				.membershipLock(false)
				.secret("secret")
				.visibility("private")
				.parentId(66L)
				.build();

		Group group = new Group().withName("Hard Times")
				.withDescription("Gonna take you down and laugh when you cry");

		when(groupAPIService.getGroupId(any(GitLabApi.class), any(String.class))).thenReturn(26L);
		when(groupAPIService.createSubgroup(any(GitLabApi.class), any(String.class), any(String.class),
				any(String.class), anyBoolean(), anyString(), anyLong())).thenReturn(group);

		when(courseEditionService.saveCourseEdition(any())).thenReturn(new CourseEdition());
		when(courseNameService.fetchNameForCoursePath(anyString())).thenReturn("");

		mockMvc.perform(MockMvcRequestBuilders.post("/api/gitlab/createCourseEdition")
				.contentType(MediaType.APPLICATION_JSON)
				.content(objectMapper.writeValueAsString(course))
				.header("coursePath", "after-laughter"))
				.andExpect(status().isCreated())
				.andExpect(MockMvcResultMatchers.jsonPath("$.name").value(course.getName()))
				.andExpect(MockMvcResultMatchers.jsonPath("$.description").value(course.getDescription()));

		verify(logWriter).writeActionLog(eq(LogType.INFO), eq(LogType.EDITION_CREATED), any(String.class),
				any(String.class));
	}

	@Test
	public void createCourseEditionFail() throws Exception {
		GroupRequestDTO course = new GroupRequestDTO()
				.builder()
				.name("Hard Times")
				.description("Gonna take you down and laugh when you cry")
				.membershipLock(false)
				.visibility("private")
				.secret("secret")
				.parentId(66L)
				.build();

		String url = "/api/gitlab/createCourseEdition";

		// failure when grabbing gitlab instance

		CourseEdition courseEdition = new CourseEdition();
		courseEdition.setCourseName("Hard Times");
		courseEdition.setEdition("2022-2023");
		courseEdition.setCoursePath("/after-laughter" + "/" + "2022-2023");

		when(gitBullSecurity.getCurrentUsername("secret")).thenReturn(null);
		//when(instanceMappingService.getInstance(anyString())).thenReturn(null);
		when(courseNameService.fetchNameForCoursePath("/after-laughter")).thenReturn("Hard Times");
		when(courseEditionService.saveCourseEdition(any(CourseEdition.class)))
				.thenReturn(new CourseEdition());

		mockMvc.perform(MockMvcRequestBuilders.post(url)
				.contentType(MediaType.APPLICATION_JSON)
				.header("coursePath", "/after-laughter")
				.content(objectMapper.writeValueAsString(course)))
				.andExpect(status().isUnauthorized());

		// failure when getting the parent id

		GitLabApiException e1 = new GitLabApiException("little rain cloud", 500);
		when(gitBullSecurity.getCurrentUsername("secret")).thenReturn("admin");
		when(groupAPIService.getGroupId(any(GitLabApi.class), any(String.class))).thenThrow(e1);

		mockMvc.perform(MockMvcRequestBuilders.post(url)
				.contentType(MediaType.APPLICATION_JSON)
				.content(objectMapper.writeValueAsString(course))
				.header("coursePath", "after-laughter"))
				.andExpect(status().isInternalServerError());

		// failure when creating subgroup for edition

		GitLabApiException e2 = new GitLabApiException("little rain cloud", 404);
		when(groupAPIService.getGroupId(any(GitLabApi.class), any(String.class))).thenReturn(26L);
		when(groupAPIService.createSubgroup(any(GitLabApi.class), any(String.class), any(String.class),
				any(String.class), anyBoolean(), anyString(), anyLong())).thenThrow(e2);

		verify(logWriter, times(1)).writeActionLog(eq(LogType.ERROR), eq(LogType.EDITION_CREATED),
				any(String.class), any(String.class));

		mockMvc.perform(MockMvcRequestBuilders.post(url)
				.contentType(MediaType.APPLICATION_JSON)
				.content(objectMapper.writeValueAsString(course))
				.header("coursePath", "after-laughter"))
				.andExpect(status().isNotFound());

		verify(logWriter, times(2)).writeActionLog(eq(LogType.ERROR), eq(LogType.EDITION_CREATED),
				any(String.class), any(String.class));
	}

	public void preSetUp() throws Exception {
		CourseEdition courseEditionTest = CourseEdition.builder().courseEditionID(50L).build();
		when(courseEditionService.getCourseEditionFromCoursePath(anyString())).thenReturn(courseEditionTest);

		Long parentId = 123L;
		Group group = new Group().withPath("ttt").withDescription("mighty")
				.withName("THE LIGHTS ON YOUR FACE").withId(-1L).withParentId(-1L).withFullPath("fullpath");

		Project project = new Project().withName("lol").withId(-2L).withDescription("lol");

		Account account = new Account("test-account", "@tudelft");
		User user = new User().withName("ken").withUsername("kenny").withEmail("kennymail@student.tudelft.nl")
				.withId(-1L);
		Member member = new Member().withName("test-account").withEmail("mail@tudelft.nl");

		when(instanceMappingService.getInstance("admin")).thenReturn(gitLabApi);
		when(gitBullSecurity.getCurrentUsername("secret")).thenReturn("admin");
		when(groupAPIService.getGroupId(eq(gitLabApi), eq("/testcourse/2023"))).thenReturn(parentId);
		when(userAPIService.verifyAccounts(any(GitLabApi.class), any(HashMap.class)))
				.thenReturn(Pair.of(List.of(account), List.of(user)));
		when(groupAPIService.createSubgroup(any(GitLabApi.class), anyString(), anyString(), anyString(),
				anyBoolean(), anyString(), anyLong())).thenReturn(group);
		when(groupAPIService.createSubgroupUpdated(any(GitLabApi.class), anyString(), anyString(),
				anyString(),
				anyString(), anyLong(), any(GroupParams.DefaultBranchProtectionLevel.class)))
						.thenReturn(group);
		when(labraConnectionService.getStaffEmailsForCourseEdition(any(String.class), any(String.class)))
				.thenReturn(List.of(List.of("tamail"), List.of("headtamail"), List.of("teachermail")));
		when(groupAPIService.addGroupMember(any(GitLabApi.class), any(Object.class), anyLong(), anyInt(),
				any(Date.class))).thenReturn(member);
		when(projectAPIService.createInitialCommit(any(GitLabApi.class), any(Object.class),
				any(String.class)))
						.thenReturn(new Commit().withMessage("message!"));
		when(projectAPIService.forkProject(any(GitLabApi.class), any(Object.class), anyLong()))
				.thenReturn(project);
		when(projectAPIService.createProjectInNamespace(any(GitLabApi.class), any(ProjectRequestDTO.class)))
				.thenReturn(project);
		when(userAPIService.getUserByEmail(any(GitLabApi.class), any(String.class))).thenReturn(user);
	}

	@Test
	public void setUpTest() throws Exception {
		String courseName = "Test CourseEdition";
		String courseEdition = "2023";
		String forkPath = "";

		preSetUp();

		String testContent = "OrgDefinedId,Username,Last Name,First Name,Email,SP,End-of-Line Indicator\n" +
				"#5678901,#lipsum1@tudelft.nl,Lorem,Ipsum,L.Ipsum1@student.tudelft.nl,1A,#\n" +
				"#5678902,#lipsum2@tudelft.nl,Lorem,Ipsum,L.Ipsum2@student.tudelft.nl,1A,#\n" +
				"#5678903,#lipsum3@tudelft.nl,Lorem,Ipsum,L.Ipsum3@student.tudelft.nl,1A,#\n" +
				"#5678904,#lipsum4@tudelft.nl,Lorem,Ipsum,L.Ipsum4@student.tudelft.nl,1A,#\n" +
				"#5678905,#lipsum5@tudelft.nl,Lorem,Ipsum,L.Ipsum5@student.tudelft.nl,1A,#\n" +
				"#6666666,#kenny@tudelft.nl,kenny,kenny,kennymail@student.tudelft.nl,2A,#\n" +
				"#5678907,#lipsum7@tudelft.nl,Lorem,Ipsum,L.Ipsum7@student.tudelft.nl,1A,#\n" +
				"#5678908,#regular@tudelft.nl,Lorem,Ipsum,L.Ipsum0@student.tudelft.nl,2A,#\n" +
				"#7777777,#comma@tudelft.nl,Funny,Name,Bro,FunnyNameBro@student.tudelft.nl,2A,#\n" +
				"#0000000,#invalid@tudeft.nl,Invalid,User,invalid@student.tudeft.nl,2A,#\n";

		CSVService realService = new CSVService();

		Map<StudentGroup, Set<Account>> studentGroupSetMap = realService.extractGroups(testContent, 5);

		when(csvService.extractGroups(testContent, 5)).thenReturn(studentGroupSetMap);
		StudentGroup studentGroup = new StudentGroup();
		studentGroup.setName("2A");
		studentGroup.setGroupPath("/2A");
		studentGroup.setGroupID(1L);
		studentGroup.setCourseID(0L);
		when(groupDBService.saveGroup(any(StudentGroup.class))).thenReturn(studentGroup);

		System.out.println(studentGroupSetMap);

		SetUpRequestDTO dto = new SetUpRequestDTO();
		dto.setCourseEdition(courseEdition);
		dto.setForkPath(forkPath);
		dto.setFileContent(testContent);
		dto.setSecret("secret");
		dto.setDefaultBranchProtectionLevel(GroupParams.DefaultBranchProtectionLevel.FULLY_PROTECTED);
		dto.setSpIndex(5);
		dto.setEnableIndividualRepositories(true);
		dto.setIndividualRepoForkPath("");
		dto.setCourseName(courseName);
		dto.setCoursePath("");
		dto.setRepoVisibility(true);
		dto.setGroupVisibility("true");

		//when(userAPIService.verifyAccounts(gitLabApi, testContent))

		MvcResult result = mockMvc.perform(MockMvcRequestBuilders.post("/api/gitlab/setUp")
				.contentType(MediaType.APPLICATION_JSON)
				.content(objectMapper.writeValueAsString(dto))).andReturn();

		mockMvc.perform(asyncDispatch(result))
				.andExpect(status().isOk());

		verify(instanceMappingService, times(1)).getInstance(any(String.class));
		verify(groupAPIService, times(1)).getGroupId(eq(gitLabApi), eq("testcourseedition/2023"));
		verify(userAPIService, times(1)).verifyAccounts(eq(gitLabApi), any(HashMap.class));
	}

	@Test
	public void setUpFailGroup() throws Exception {
		String courseName = "Test CourseEdition";
		String courseEdition = "2023";
		String forkPath = "";

		preSetUp();

		String testContent = "OrgDefinedId,Username,Last Name,First Name,Email,SP,End-of-Line Indicator\n" +
				"#5678901,#lipsum1@tudelft.nl,Lorem,Ipsum,L.Ipsum1@student.tudelft.nl,1A,#\n" +
				"#5678902,#lipsum2@tudelft.nl,Lorem,Ipsum,L.Ipsum2@student.tudelft.nl,1A,#\n" +
				"#5678903,#lipsum3@tudelft.nl,Lorem,Ipsum,L.Ipsum3@student.tudelft.nl,1A,#\n" +
				"#5678904,#lipsum4@tudelft.nl,Lorem,Ipsum,L.Ipsum4@student.tudelft.nl,1A,#\n" +
				"#5678905,#lipsum5@tudelft.nl,Lorem,Ipsum,L.Ipsum5@student.tudelft.nl,1A,#\n" +
				"#6666666,#kenny@tudelft.nl,kenny,kenny,kennymail@student.tudelft.nl,2A,#\n" +
				"#5678907,#lipsum7@tudelft.nl,Lorem,Ipsum,L.Ipsum7@student.tudelft.nl,1A,#\n" +
				"#5678908,#regular@tudelft.nl,Lorem,Ipsum,L.Ipsum0@student.tudelft.nl,2A,#\n" +
				"#7777777,#comma@tudelft.nl,Funny,Name,Bro,FunnyNameBro@student.tudelft.nl,2A,#\n" +
				"#0000000,#invalid@tudeft.nl,Invalid,User,invalid@student.tudeft.nl,2A,#\n";

		CSVService realService = new CSVService();

		Map<StudentGroup, Set<Account>> studentGroupSetMap = realService.extractGroups(testContent, 5);

		when(csvService.extractGroups(testContent, 5)).thenReturn(studentGroupSetMap);
		StudentGroup studentGroup = new StudentGroup();
		studentGroup.setName("2A");
		studentGroup.setGroupPath("/2A");
		studentGroup.setGroupID(1L);
		studentGroup.setCourseID(0L);
		when(groupDBService.saveGroup(any(StudentGroup.class))).thenThrow(new NullPointerException("Hahaha"));

		System.out.println(studentGroupSetMap);

		SetUpRequestDTO dto = new SetUpRequestDTO();
		dto.setCourseEdition(courseEdition);
		dto.setForkPath(forkPath);
		dto.setFileContent(testContent);
		dto.setSecret("secret");
		dto.setDefaultBranchProtectionLevel(GroupParams.DefaultBranchProtectionLevel.FULLY_PROTECTED);
		dto.setSpIndex(5);
		dto.setEnableIndividualRepositories(true);
		dto.setIndividualRepoForkPath("");
		dto.setCourseName(courseName);
		dto.setCoursePath("");
		dto.setRepoVisibility(true);
		dto.setGroupVisibility("true");

		//when(userAPIService.verifyAccounts(gitLabApi, testContent))

		MvcResult result = mockMvc.perform(MockMvcRequestBuilders.post("/api/gitlab/setUp")
				.contentType(MediaType.APPLICATION_JSON)
				.content(objectMapper.writeValueAsString(dto))).andReturn();

		mockMvc.perform(asyncDispatch(result))
				.andExpect(status().isOk());

		verify(instanceMappingService, times(1)).getInstance(any(String.class));
		verify(groupAPIService, times(1)).getGroupId(eq(gitLabApi), eq("testcourseedition/2023"));
		verify(userAPIService, times(1)).verifyAccounts(eq(gitLabApi), any(HashMap.class));
	}

	@Test
	public void setUpFailInstance() throws Exception {
		String courseName = "Test CourseEdition";
		String courseEdition = "2023";
		String forkPath = "";

		preSetUp();

		when(instanceMappingService.getInstance(any(String.class))).thenReturn(null);

		SetUpRequestDTO requestDTO = new SetUpRequestDTO();
		requestDTO.setCourseEdition(courseEdition);
		requestDTO.setForkPath(forkPath);
		requestDTO.setFileContent("hi");
		requestDTO.setSecret("secret");
		requestDTO.setDefaultBranchProtectionLevel(GroupParams.DefaultBranchProtectionLevel.FULLY_PROTECTED);
		requestDTO.setSpIndex(5);
		requestDTO.setEnableIndividualRepositories(true);
		requestDTO.setIndividualRepoForkPath("");
		requestDTO.setCourseName(courseName);
		requestDTO.setCoursePath("");
		requestDTO.setRepoVisibility(true);
		requestDTO.setGroupVisibility("true");

		MvcResult result = mockMvc.perform(MockMvcRequestBuilders.post("/api/gitlab/setUp")
				.contentType(MediaType.APPLICATION_JSON)
				.content(objectMapper.writeValueAsString(requestDTO))).andReturn();

		mockMvc.perform(asyncDispatch(result))
				.andExpect(status().isUnauthorized())
				.andExpect(content().json("[\"You are not authorized.\"]"));
	}

	@Test
	public void setUpFailGetGroupID() throws Exception {
		String courseName = "Test CourseEdition";
		String courseEdition = "2023";
		String forkPath = "";

		preSetUp();

		GitLabApiException e = new GitLabApiException("Fetching ID failed!", 500);
		when(groupAPIService.getGroupId(any(GitLabApi.class), any(String.class))).thenThrow(e);

		SetUpRequestDTO requestDTO = new SetUpRequestDTO();
		requestDTO.setCourseEdition(courseEdition);
		requestDTO.setForkPath(forkPath);
		requestDTO.setFileContent("hi");
		requestDTO.setSecret("secret");
		requestDTO.setDefaultBranchProtectionLevel(GroupParams.DefaultBranchProtectionLevel.FULLY_PROTECTED);
		requestDTO.setSpIndex(5);
		requestDTO.setEnableIndividualRepositories(true);
		requestDTO.setIndividualRepoForkPath("");
		requestDTO.setCourseName(courseName);
		requestDTO.setCoursePath("");
		requestDTO.setRepoVisibility(true);
		requestDTO.setGroupVisibility("true");

		MvcResult result = mockMvc.perform(MockMvcRequestBuilders.post("/api/gitlab/setUp")
				.contentType(MediaType.APPLICATION_JSON)
				.content(objectMapper.writeValueAsString(requestDTO))).andReturn();

		mockMvc.perform(asyncDispatch(result))
				.andExpect(status().is(500));

		String message = result.getResponse().getContentAsString();
		assertTrue(message.contains("Couldn't get parent group!"));
	}

	@Test
	public void testAddStaff() throws Exception {
		String courseName = "CourseEdition A";
		String courseEdition = "2023";

		List<String> tas = Arrays.asList("ta1@example.com", "ta2@example.com");
		List<String> headTas = Arrays.asList("headta1@example.com", "headta2@example.com");
		List<String> lecturers = Arrays.asList("lecturer1@example.com", "lecturer2@example.com");

		List<List<String>> staffList = new ArrayList<>();
		staffList.add(tas);
		staffList.add(headTas);
		staffList.add(lecturers);

		when(labraConnectionService.getStaffEmailsForCourseEdition(courseName, courseEdition))
				.thenReturn(staffList);

		User user1 = new User();
		user1.setId(123L);
		User user2 = new User();
		user2.setId(456L);
		User user3 = new User();
		user3.setId(789L);
		User user4 = new User();
		user4.setId(7L);
		User user5 = new User();
		user5.setId(8L);
		User user6 = new User();
		user6.setId(9L);

		MemberRequestDTO memberRequestDTO1 = new MemberRequestDTO(123L,
				"/coursea/2023", 40, null, "secret");

		MemberRequestDTO memberRequestDTO2 = new MemberRequestDTO(456L,
				"/coursea/2023", 40, null, "secret");

		MemberRequestDTO memberRequestDTO3 = new MemberRequestDTO(789L,
				"/coursea/2023", 40, null, "secret");

		MemberRequestDTO memberRequestDTO4 = new MemberRequestDTO(7L,
				"/coursea/2023", 40, null, "secret");

		MemberRequestDTO memberRequestDTO5 = new MemberRequestDTO(8L,
				"/coursea/2023", 40, null, "secret");

		MemberRequestDTO memberRequestDTO6 = new MemberRequestDTO(9L,
				"/coursea/2023", 40, null, "secret");

		when(userAPIService.getUserByEmail(eq(gitLabApi), eq("ta1@example.com"))).thenReturn(user1);
		when(userAPIService.getUserByEmail(eq(gitLabApi), eq("ta2@example.com"))).thenReturn(user2);

		when(userAPIService.getUserByEmail(eq(gitLabApi), eq("headta1@example.com"))).thenReturn(user3);
		when(userAPIService.getUserByEmail(eq(gitLabApi), eq("headta2@example.com"))).thenReturn(user4);

		when(userAPIService.getUserByEmail(eq(gitLabApi), eq("lecturer1@example.com"))).thenReturn(user5);
		when(userAPIService.getUserByEmail(eq(gitLabApi), eq("lecturer2@example.com"))).thenReturn(user6);

		when(groupAPIService.addGroupMember(any(GitLabApi.class), eq(memberRequestDTO1))).thenReturn(null);
		when(groupAPIService.addGroupMember(any(GitLabApi.class), eq(memberRequestDTO2))).thenReturn(null);
		when(groupAPIService.addGroupMember(any(GitLabApi.class), eq(memberRequestDTO3))).thenReturn(null);
		when(groupAPIService.addGroupMember(any(GitLabApi.class), eq(memberRequestDTO4))).thenReturn(null);
		when(groupAPIService.addGroupMember(any(GitLabApi.class), eq(memberRequestDTO5))).thenReturn(null);
		when(groupAPIService.addGroupMember(any(GitLabApi.class), eq(memberRequestDTO6))).thenReturn(null);

		when(courseEditionService.getCourseEditionFromCoursePath(anyString()))
				.thenReturn(new CourseEdition());

		StaffRequestDTO staffRequestDTO = new StaffRequestDTO();
		staffRequestDTO.setCourseEdition(courseEdition);
		staffRequestDTO.setCourseName(courseName);
		staffRequestDTO.setEditionPath(courseName + "/" + courseEdition);
		staffRequestDTO.setSecret("secret");

		gitLabController.importStaff(staffRequestDTO);

		//verify(gitLabController).addStaff(courseName, courseEdition);
	}

	@Test
	public void importStaffTest() throws GitLabApiException {

		when(courseEditionService.getCourseEditionFromCoursePath(anyString()))
				.thenReturn(new CourseEdition());
		String courseName = "Test Course";
		String courseEdition = "Test Edition";
		String editionPath = "test/path";

		List<List<String>> staffList = new ArrayList<>();
		List<String> tas = new ArrayList<>();
		tas.add("ta1@example.com");
		List<String> headTas = new ArrayList<>();
		headTas.add("headta1@example.com");
		List<String> lecturers = new ArrayList<>();
		lecturers.add("lecturer1@example.com");
		staffList.add(tas);
		staffList.add(headTas);
		staffList.add(lecturers);

		when(instanceMappingService.getInstance("admin")).thenReturn(null);

		StaffRequestDTO staffRequestDTO = new StaffRequestDTO(null, editionPath, courseName, courseEdition,
				"secret");

		ResponseEntity<List<String>> resultNull = gitLabController.importStaff(staffRequestDTO);
		assertEquals(resultNull.getStatusCode(), HttpStatus.UNAUTHORIZED);
		assertNull(resultNull.getBody());

		when(labraConnectionService.getStaffEmailsForCourseEdition(courseName, courseEdition))
				.thenReturn(staffList);
		when(instanceMappingService.getInstance("admin")).thenReturn(gitLabApi);
		when(userAPIService.getUserByUsername(gitLabApi, "ta1")).thenReturn(new User());
		when(userAPIService.getUserByUsername(gitLabApi, "headta1")).thenReturn(new User());
		when(userAPIService.getUserByUsername(gitLabApi, "lecturer1")).thenReturn(new User());
		when(groupAPIService.addGroupMember(any(GitLabApi.class), anyString(), anyLong(), eq(40), eq(null)))
				.thenReturn(new Member().withName("hihihahhahahahahhaiahahahahhahah"));

		ResponseEntity<List<String>> response = gitLabController.importStaff(staffRequestDTO);

		verify(labraConnectionService).getStaffEmailsForCourseEdition(courseName, courseEdition);
		verify(userAPIService).getUserByUsername(gitLabApi, "ta1");
		verify(userAPIService).getUserByUsername(gitLabApi, "headta1");
		verify(userAPIService).getUserByUsername(gitLabApi, "lecturer1");
		verify(groupAPIService, times(1)).addGroupMember(gitLabApi, "test/path", null, 40, null);
		assertEquals(HttpStatus.OK, response.getStatusCode());
		assertEquals(0, response.getBody().size());
	}

	@Test
	public void addStaffToGroupsTest() throws GitLabApiException {

		Map<String, String> staffGroupsMap = new HashMap<>();
		String editionPath = "path";

		when(instanceMappingService.getInstance("admin")).thenReturn(null);
		StaffRequestDTO staffRequestDTO = new StaffRequestDTO();
		staffRequestDTO.setStaffGroupsMap(staffGroupsMap);
		staffRequestDTO.setEditionPath(editionPath);
		staffRequestDTO.setSecret("secret");
		ResponseEntity resultNull = gitLabController
				.addStaffToGroups(staffRequestDTO);
		assertEquals(HttpStatus.UNAUTHORIZED, resultNull.getStatusCode());
		assertNull(resultNull.getBody());

		String clusterA = "apple, pear, orange";
		String clusterB = "TOMATO";
		String clusterC = "specialties";

		staffGroupsMap.put("scary@tudelft.nl", clusterA);
		staffGroupsMap.put("evil@tudelft.nl", clusterB);
		staffGroupsMap.put("missing@tudelft.nl", null);
		staffGroupsMap.put("invisible@tudelft.nl", clusterC);

		User scaryUser = new User().withName("I am so scary ahahaha").withBio("Scary person").withId(666L)
				.withName("Scary-Name");
		User evilUser = new User().withName("I am so evil muahaha").withBio("Evil person").withId(-666L)
				.withName("Evil-Name");
		User invisibleUser = new User().withName(" ").withBio(" ").withId(0L).withName(" ");
		when(instanceMappingService.getInstance("admin")).thenReturn(gitLabApi);

		when(userAPIService.getUserByEmail(gitLabApi, "scary@tudelft.nl")).thenReturn(scaryUser);
		when(userAPIService.getUserByEmail(gitLabApi, "evil@tudelft.nl")).thenReturn(evilUser);
		when(userAPIService.getUserByEmail(gitLabApi, "missing@tudelft.nl"))
				.thenThrow(new GitLabApiException("He is missing!", 404));
		when(userAPIService.getUserByEmail(gitLabApi, "invisible@tudelft.nl")).thenReturn(invisibleUser);

		when(groupAPIService.addGroupMember(any(GitLabApi.class), anyString(), anyLong(), eq(40), eq(null)))
				.thenReturn(new Member());
		when(groupAPIService.addGroupMember(any(GitLabApi.class), eq("path/specialties"), eq(0L), eq(40),
				eq(null))).thenThrow(new GitLabApiException("Couldn't find him!", 404));

		when(courseEditionService.getCourseEditionFromCoursePath(anyString()))
				.thenReturn(new CourseEdition());

		ResponseEntity<List<String>> result = gitLabController
				.addStaffToGroups(staffRequestDTO);

		List<String> expectedLog = List.of(
				"Couldn't add staff with mail [scary@tudelft.nl] to group path [path]. Reason: User could not be found.",
				"Couldn't add staff with mail [invisible@tudelft.nl] to group path [path]. Reason: User could not be found.",
				"Couldn't add staff with mail [missing@tudelft.nl] to group path [path]. Reason: User could not be found.",
				"Couldn't add staff with mail [evil@tudelft.nl] to group path [path]. Reason: User could not be found.");

		assertEquals(HttpStatus.OK, result.getStatusCode());
		assertEquals(expectedLog, result.getBody());
	}

	@Test
	void testAddTeacherToCourse() throws GitLabApiException {
		String secret = "secret";
		String username = "teacher@example.com";
		String coursePath = "course/path";
		String currentUsername = "current_user";
		Long userId = 123L;
		String displayName = "Teacher";
		TeacherCourseRequestDTO teacherCourseRequestDTO = new TeacherCourseRequestDTO();
		teacherCourseRequestDTO.setSecret(secret);
		teacherCourseRequestDTO.setUsername(username);
		teacherCourseRequestDTO.setCoursePath(coursePath);

		GitLabApi gitLabApi = mock(GitLabApi.class);
		User user = new User();
		user.setId(userId);
		when(gitBullSecurity.getCurrentUsername(secret)).thenReturn(currentUsername);
		when(instanceMappingService.getInstance(currentUsername)).thenReturn(gitLabApi);
		when(userAPIService.getUserByEmail(gitLabApi, username.toLowerCase())).thenReturn(null);
		PersonDetailsDTO personDetailsDTO = new PersonDetailsDTO();
		personDetailsDTO.setUsername(username);
		personDetailsDTO.setDisplayName(displayName);
		when(labraConnectionService.getPerson(username)).thenReturn(personDetailsDTO);
		when(userAPIService.createUser(gitLabApi, username, username, displayName)).thenReturn(user);

		Member member = new Member();
		member.setId(userId);
		when(groupAPIService.addGroupMember(gitLabApi, coursePath, userId, 50, null)).thenReturn(member);

		ResponseEntity<?> response = gitLabController.addTeacherToCourse(teacherCourseRequestDTO);

		assertEquals(response.getStatusCode(), HttpStatus.OK);
		assertEquals(response.getBody(), member);
	}

	@Test
	public void initProjectTest() throws GitLabApiException {
		Long namespaceID = 123L;
		String projectName = "Test Project";
		String forkPath = null;
		Boolean repoVisibility = true;
		String currentUser = "user1";

		ProjectRequestDTO projectRequestDTO = new ProjectRequestDTO("testproject", projectName,
				"", namespaceID, true, true, true, true, repoVisibility);

		Project createdProject = new Project();
		createdProject.setId(456L);

		when(projectAPIService.createProjectInNamespace(gitLabApi, projectRequestDTO))
				.thenReturn(createdProject);

		Project result = gitLabController.initProject(gitLabApi, namespaceID, projectName,
				forkPath, repoVisibility, currentUser);

		verify(projectAPIService, times(1)).createProjectInNamespace(gitLabApi, projectRequestDTO);
		verify(projectAPIService, times(1)).createInitialCommit(any(GitLabApi.class), eq(456L), anyString());

		assertEquals(createdProject, result);
	}

	@Test
	public void initProjectForkTest() throws GitLabApiException {
		Long namespaceID = 123L;
		String projectName = "Test Project";
		String forkPath = "existing-project";
		Boolean repoVisibility = true;
		String currentUser = "user1";

		Project forkedProject = new Project();
		forkedProject.setId(789L);

		when(projectAPIService.getProject(gitLabApi, forkPath)).thenReturn(mock(Project.class));
		when(projectAPIService.forkProject(any(GitLabApi.class), any(Project.class), eq(namespaceID)))
				.thenReturn(forkedProject);

		Project result = gitLabController.initProject(gitLabApi, namespaceID, projectName,
				forkPath, repoVisibility, currentUser);

		verify(projectAPIService, times(1)).getProject(any(GitLabApi.class), eq(forkPath));
		verify(projectAPIService, times(1)).forkProject(any(GitLabApi.class), any(Project.class),
				eq(namespaceID));

		assertEquals(forkedProject, result);
	}

	@Test
	public void getCourseStaffTest() {
		StaffRequestDTO dto = new StaffRequestDTO();
		dto.setSecret("secret");
		dto.setCourseEdition("2022-2023");
		dto.setCourseName("test-course");

		List<PersonSummaryDTO> expectedSummary = List.of(new PersonSummaryDTO());
		when(gitBullSecurity.getCurrentUsername(dto.getSecret())).thenReturn(null);

		ResponseEntity<List<PersonSummaryDTO>> fail = gitLabController.getCourseStaff(dto);
		assertEquals(HttpStatus.UNAUTHORIZED, fail.getStatusCode());
		assertNull(fail.getBody());

		when(gitBullSecurity.getCurrentUsername(dto.getSecret())).thenReturn("admin");

		when(labraConnectionService.getTeacherAssistantsForCourseEdition(dto.getCourseName(),
				dto.getCourseEdition()))
						.thenReturn(List.of(new PersonSummaryDTO()));

		ResponseEntity<List<PersonSummaryDTO>> result = gitLabController.getCourseStaff(dto);
		assertEquals(HttpStatus.OK, result.getStatusCode());
		assertEquals(expectedSummary, result.getBody());
	}

}
