package server.controller;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.mockito.Mockito.*;

import java.util.*;

import org.gitlab4j.api.GitLabApi;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.data.util.Pair;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;

import server.entity.Statistics;
import server.model.GroupStatisticsRequestDTO;
import server.security.GitBullSecurity;
import server.service.InstanceMappingService;
import server.service.statisticsService.GroupStatisticsService;
import server.service.statisticsService.IndividualStatisticsService;

public class GitLabStatisticsControllerTest {

	@Mock
	private GroupStatisticsService groupStatisticsService;

	@Mock
	private IndividualStatisticsService individualStatisticsService;

	@Mock
	private InstanceMappingService instanceMappingService;

	@Mock
	private GitBullSecurity gitBullSecurity;

	@Mock
	private GitLabApi gitLabApi;

	private GitLabStatisticsController controller;

	@BeforeEach
	public void setup() {
		MockitoAnnotations.openMocks(this);
		controller = new GitLabStatisticsController(
				groupStatisticsService,
				individualStatisticsService,
				instanceMappingService,
				gitBullSecurity);
	}

	@Test
	public void testGetStatsForGroup() {
		// Mock the dependencies
		GroupStatisticsRequestDTO dto = new GroupStatisticsRequestDTO();
		dto.setSecret("testSecret");
		dto.setGroupPath("groupPath");
		dto.setRepoPath("repoPath");

		String currentUser = "testUser";
		when(gitBullSecurity.getCurrentUsername(dto.getSecret())).thenReturn(currentUser);
		when(instanceMappingService.getInstance(currentUser)).thenReturn(gitLabApi);
		when(groupStatisticsService.getParticipantsNames(gitLabApi, dto.getGroupPath()))
				.thenReturn(Arrays.asList("user1", "user2"));

		Map<String, Pair<Integer, Integer>> userLinesInfo = mock(Map.class);
		when(groupStatisticsService.getUsersLinesOfCodeCount(gitLabApi, dto.getGroupPath()))
				.thenReturn(userLinesInfo);

		when(individualStatisticsService.getUserCommentsCountOnMerge(gitLabApi, dto.getRepoPath(), "user1"))
				.thenReturn(5);
		when(individualStatisticsService.getUserCommentsCountOnMerge(gitLabApi, dto.getRepoPath(), "user2"))
				.thenReturn(3);
		when(userLinesInfo.getOrDefault("user1", Pair.of(0, 0))).thenReturn(Pair.of(10, 2));
		when(userLinesInfo.getOrDefault("user2", Pair.of(0, 0))).thenReturn(Pair.of(8, 4));
		when(individualStatisticsService.getUserIssuesCount(gitLabApi, dto.getRepoPath(), "user1"))
				.thenReturn(2);
		when(individualStatisticsService.getUserIssuesCount(gitLabApi, dto.getRepoPath(), "user2"))
				.thenReturn(4);
		when(individualStatisticsService.getUserMergeRequestsCount(gitLabApi, dto.getRepoPath(), "user1"))
				.thenReturn(3);
		when(individualStatisticsService.getUserMergeRequestsCount(gitLabApi, dto.getRepoPath(), "user2"))
				.thenReturn(1);
		when(individualStatisticsService.getUserCommitCount(gitLabApi, dto.getRepoPath(), "user1"))
				.thenReturn(7);
		when(individualStatisticsService.getUserCommitCount(gitLabApi, dto.getRepoPath(), "user2"))
				.thenReturn(6);

		// Perform the test
		ResponseEntity<List<Statistics>> response = controller.getStatsForGroup(dto);
		assertEquals(HttpStatus.OK, response.getStatusCode());

		List<Statistics> expectedStats = new ArrayList<>();
		expectedStats.add(new Statistics("user1", "7", "2", "3", "10", "2", "5"));
		expectedStats.add(new Statistics("user2", "6", "4", "1", "8", "4", "3"));

		assertEquals(expectedStats, response.getBody());

		// Verify the interactions
		verify(gitBullSecurity).getCurrentUsername(dto.getSecret());
		verify(instanceMappingService).getInstance(currentUser);
		verify(groupStatisticsService).getParticipantsNames(gitLabApi, dto.getGroupPath());
		verify(groupStatisticsService).getUsersLinesOfCodeCount(gitLabApi, dto.getGroupPath());
		verify(individualStatisticsService).getUserCommentsCountOnMerge(gitLabApi, dto.getRepoPath(),
				"user1");
		verify(individualStatisticsService).getUserCommentsCountOnMerge(gitLabApi, dto.getRepoPath(),
				"user2");
		verify(individualStatisticsService).getUserIssuesCount(gitLabApi, dto.getRepoPath(), "user1");
		verify(individualStatisticsService).getUserIssuesCount(gitLabApi, dto.getRepoPath(), "user2");
		verify(individualStatisticsService).getUserMergeRequestsCount(gitLabApi, dto.getRepoPath(), "user1");
		verify(individualStatisticsService).getUserMergeRequestsCount(gitLabApi, dto.getRepoPath(), "user2");
		verify(individualStatisticsService).getUserCommitCount(gitLabApi, dto.getRepoPath(), "user1");
		verify(individualStatisticsService).getUserCommitCount(gitLabApi, dto.getRepoPath(), "user2");
	}

	@Test
	public void testGetStatsForGroup_Unauthorized() {
		// Mock the dependencies
		GroupStatisticsRequestDTO dto = new GroupStatisticsRequestDTO();
		dto.setSecret("testSecret");
		dto.setGroupPath("groupPath");
		dto.setRepoPath("repoPath");

		String currentUser = "testUser";
		when(gitBullSecurity.getCurrentUsername(dto.getSecret())).thenReturn(currentUser);
		when(instanceMappingService.getInstance(currentUser)).thenReturn(null); // Return null for GitLabApi instance

		// Perform the test
		ResponseEntity<List<Statistics>> response = controller.getStatsForGroup(dto);
		assertEquals(HttpStatus.UNAUTHORIZED, response.getStatusCode());

		// Verify the interactions
		verify(gitBullSecurity).getCurrentUsername(dto.getSecret());
		verify(instanceMappingService).getInstance(currentUser);
		// Verify that the controller returns UNAUTHORIZED without interacting with other services
		verifyNoInteractions(groupStatisticsService, individualStatisticsService);
	}

	@Test
	public void testGetStatsForAllGroups_Success() {
		// Mock the dependencies
		GroupStatisticsRequestDTO dto = new GroupStatisticsRequestDTO();
		dto.setSecret("testSecret");
		dto.setGroupPaths(Arrays.asList("groupPath1", "groupPath2"));
		dto.setRepoPaths(Arrays.asList("repoPath1", "repoPath2"));

		String currentUser = "testUser";
		GitLabApi gitLabApi = mock(GitLabApi.class);
		when(gitBullSecurity.getCurrentUsername(dto.getSecret())).thenReturn(currentUser);
		when(instanceMappingService.getInstance(currentUser)).thenReturn(gitLabApi);

		// Mock the group statistics service
		List<String> group1Names = Arrays.asList("user1", "user2");
		List<String> group2Names = Arrays.asList("user3", "user4");

		when(groupStatisticsService.getParticipantsNames(gitLabApi, "groupPath1")).thenReturn(group1Names);
		when(groupStatisticsService.getParticipantsNames(gitLabApi, "groupPath2")).thenReturn(group2Names);

		Map<String, Pair<Integer, Integer>> group1UserLinesInfo = new HashMap<>();
		group1UserLinesInfo.put("user1", Pair.of(10, 5));
		group1UserLinesInfo.put("user2", Pair.of(20, 15));
		when(groupStatisticsService.getUsersLinesOfCodeCount(gitLabApi, "repoPath1"))
				.thenReturn(group1UserLinesInfo);

		Map<String, Pair<Integer, Integer>> group2UserLinesInfo = new HashMap<>();
		group2UserLinesInfo.put("user3", Pair.of(8, 3));
		group2UserLinesInfo.put("user4", Pair.of(15, 10));
		when(groupStatisticsService.getUsersLinesOfCodeCount(gitLabApi, "repoPath2"))
				.thenReturn(group2UserLinesInfo);

		// Mock the individual statistics service
		when(individualStatisticsService.getUserCommentsCountOnMerge(gitLabApi, "repoPath1", "user1"))
				.thenReturn(3);
		when(individualStatisticsService.getUserCommentsCountOnMerge(gitLabApi, "repoPath1", "user2"))
				.thenReturn(5);
		when(individualStatisticsService.getUserCommentsCountOnMerge(gitLabApi, "repoPath2", "user3"))
				.thenReturn(2);
		when(individualStatisticsService.getUserCommentsCountOnMerge(gitLabApi, "repoPath2", "user4"))
				.thenReturn(4);

		when(individualStatisticsService.getUserIssuesCount(gitLabApi, "repoPath1", "user1")).thenReturn(2);
		when(individualStatisticsService.getUserIssuesCount(gitLabApi, "repoPath1", "user2")).thenReturn(4);
		when(individualStatisticsService.getUserIssuesCount(gitLabApi, "repoPath2", "user3")).thenReturn(1);
		when(individualStatisticsService.getUserIssuesCount(gitLabApi, "repoPath2", "user4")).thenReturn(3);

		when(individualStatisticsService.getUserMergeRequestsCount(gitLabApi, "repoPath1", "user1"))
				.thenReturn(3);
		when(individualStatisticsService.getUserMergeRequestsCount(gitLabApi, "repoPath1", "user2"))
				.thenReturn(2);
		when(individualStatisticsService.getUserMergeRequestsCount(gitLabApi, "repoPath2", "user3"))
				.thenReturn(4);
		when(individualStatisticsService.getUserMergeRequestsCount(gitLabApi, "repoPath2", "user4"))
				.thenReturn(1);

		when(individualStatisticsService.getUserCommitCount(gitLabApi, "repoPath1", "user1")).thenReturn(10);
		when(individualStatisticsService.getUserCommitCount(gitLabApi, "repoPath1", "user2")).thenReturn(15);
		when(individualStatisticsService.getUserCommitCount(gitLabApi, "repoPath2", "user3")).thenReturn(8);
		when(individualStatisticsService.getUserCommitCount(gitLabApi, "repoPath2", "user4")).thenReturn(12);

		// Perform the test
		ResponseEntity<List<List<Statistics>>> response = controller.getStatsForAllGroups(dto);
		assertEquals(HttpStatus.OK, response.getStatusCode());

		List<List<Statistics>> expectedStats = new ArrayList<>();
		List<Statistics> group1Stats = new ArrayList<>();
		group1Stats.add(new Statistics("user1", "10", "2", "3", "10", "5", "3"));
		group1Stats.add(new Statistics("user2", "15", "4", "2", "20", "15", "5"));
		expectedStats.add(group1Stats);

		List<Statistics> group2Stats = new ArrayList<>();
		group2Stats.add(new Statistics("user3", "8", "1", "4", "8", "3", "2"));
		group2Stats.add(new Statistics("user4", "12", "3", "1", "15", "10", "4"));
		expectedStats.add(group2Stats);

		assertEquals(expectedStats, response.getBody());

		// Verify the interactions
		verify(gitBullSecurity).getCurrentUsername(dto.getSecret());
		verify(instanceMappingService).getInstance(currentUser);
		verify(groupStatisticsService).getParticipantsNames(gitLabApi, "groupPath1");
		verify(groupStatisticsService).getParticipantsNames(gitLabApi, "groupPath2");
		verify(groupStatisticsService).getUsersLinesOfCodeCount(gitLabApi, "repoPath1");
		verify(groupStatisticsService).getUsersLinesOfCodeCount(gitLabApi, "repoPath2");
		verify(individualStatisticsService).getUserCommentsCountOnMerge(gitLabApi, "repoPath1", "user1");
		verify(individualStatisticsService).getUserCommentsCountOnMerge(gitLabApi, "repoPath1", "user2");
		verify(individualStatisticsService).getUserCommentsCountOnMerge(gitLabApi, "repoPath2", "user3");
		verify(individualStatisticsService).getUserCommentsCountOnMerge(gitLabApi, "repoPath2", "user4");
		verify(individualStatisticsService).getUserIssuesCount(gitLabApi, "repoPath1", "user1");
		verify(individualStatisticsService).getUserIssuesCount(gitLabApi, "repoPath1", "user2");
		verify(individualStatisticsService).getUserIssuesCount(gitLabApi, "repoPath2", "user3");
		verify(individualStatisticsService).getUserIssuesCount(gitLabApi, "repoPath2", "user4");
		verify(individualStatisticsService).getUserMergeRequestsCount(gitLabApi, "repoPath1", "user1");
		verify(individualStatisticsService).getUserMergeRequestsCount(gitLabApi, "repoPath1", "user2");
		verify(individualStatisticsService).getUserMergeRequestsCount(gitLabApi, "repoPath2", "user3");
		verify(individualStatisticsService).getUserMergeRequestsCount(gitLabApi, "repoPath2", "user4");
		verify(individualStatisticsService).getUserCommitCount(gitLabApi, "repoPath1", "user1");
		verify(individualStatisticsService).getUserCommitCount(gitLabApi, "repoPath1", "user2");
		verify(individualStatisticsService).getUserCommitCount(gitLabApi, "repoPath2", "user3");
		verify(individualStatisticsService).getUserCommitCount(gitLabApi, "repoPath2", "user4");
	}

	@Test
	public void testGetStatsForAllGroups_Unauthorized() {
		// Mock dependencies
		GroupStatisticsService groupStatisticsService = mock(GroupStatisticsService.class);
		IndividualStatisticsService individualStatisticsService = mock(IndividualStatisticsService.class);
		InstanceMappingService instanceMappingService = mock(InstanceMappingService.class);
		GitBullSecurity gitBullSecurity = mock(GitBullSecurity.class);

		// Create the controller instance
		GitLabStatisticsController controller = new GitLabStatisticsController(
				groupStatisticsService, individualStatisticsService, instanceMappingService, gitBullSecurity);

		// Prepare the test data
		GroupStatisticsRequestDTO dto = new GroupStatisticsRequestDTO();
		dto.setGroupPaths(Arrays.asList("groupPath1", "groupPath2"));
		dto.setRepoPaths(Arrays.asList("repoPath1", "repoPath2"));

		// Mock the behavior of GitBullSecurity to return null GitLabApi instance
		when(gitBullSecurity.getCurrentUsername(dto.getSecret())).thenReturn("currentUser");
		when(instanceMappingService.getInstance("currentUser")).thenReturn(null);

		// Perform the test
		ResponseEntity<List<List<Statistics>>> response = controller.getStatsForAllGroups(dto);

		// Verify the response
		assertEquals(HttpStatus.UNAUTHORIZED, response.getStatusCode());

		// Verify the interactions
		verify(gitBullSecurity).getCurrentUsername(dto.getSecret());
		verify(instanceMappingService).getInstance("currentUser");
		verifyNoInteractions(groupStatisticsService);
		verifyNoInteractions(individualStatisticsService);
	}

}
