package server.controller;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNull;
import static org.mockito.Mockito.*;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.content;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import static org.springframework.test.web.servlet.setup.MockMvcBuilders.standaloneSetup;

import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

import org.gitlab4j.api.GitLabApi;
import org.gitlab4j.api.GitLabApiException;
import org.gitlab4j.api.GroupApi;
import org.gitlab4j.api.models.*;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.request.MockMvcRequestBuilders;
import org.springframework.test.web.servlet.result.MockMvcResultMatchers;

import server.entity.Access;
import server.entity.CourseEdition;
import server.entity.StudentGroup;
import server.model.BatchRequestDTO;
import server.model.GroupRequestDTO;
import server.model.UserRequestDTO;
import server.security.GitBullSecurity;
import server.service.*;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

public class GroupManagementControllerTest {
	private MockMvc mockMvc;
	private ObjectMapper objectMapper;

	@Mock
	private GroupAPIService groupAPIService;
	@Mock
	private GitLabApi gitLabApi;

	@Mock
	private GitBullSecurity gitBullSecurity;
	@Mock
	private UserAPIService userAPIService;

	@Mock
	private GroupService groupDBService;
	@Mock
	private CourseEditionService courseEditionDBService;
	@Mock
	private AccessService accessDBService;

	@InjectMocks
	private GroupManagementController groupManagementController;

	@Mock
	private InstanceMappingService instanceMappingService;

	@Mock
	private GroupApi groupApi;

	@BeforeEach
	public void setUp() throws GitLabApiException {
		MockitoAnnotations.openMocks(this);
		mockMvc = standaloneSetup(groupManagementController).build();
		objectMapper = new ObjectMapper();
		when(gitBullSecurity.getCurrentUsername("secret")).thenReturn("admin");
		when(instanceMappingService.getInstance("admin")).thenReturn(gitLabApi);
		when(instanceMappingService.getInstance("gitbull")).thenReturn(gitLabApi);
		when(userAPIService.getCurrentUser(null))
				.thenReturn(new User().withUsername("test-instance"));
		when(userAPIService.getCurrentUser(any(GitLabApi.class)))
				.thenReturn(new User().withUsername("test-instance"));
		when(gitLabApi.getGroupApi()).thenReturn(groupApi);

		when(groupDBService.getGroupFromGroupPath(anyString())).thenReturn(new StudentGroup());
		when(courseEditionDBService.getCourseEditionFromCoursePath(anyString()))
				.thenReturn(new CourseEdition());
		Access newAccess = Access.builder().accessID(1L).groupID(1L).role(1).accountID("username")
				.courseEditionID(2L).build();
		when(accessDBService.getAccessFromUserEditionGroup(anyString(), anyLong(), anyLong()))
				.thenReturn(newAccess);
		when(accessDBService.saveAccess(any())).thenReturn(newAccess);
	}

	private String requestDTOToJson(GroupRequestDTO requestDTO) throws JsonProcessingException {
		ObjectMapper objectMapper = new ObjectMapper();
		return objectMapper.writeValueAsString(requestDTO);
	}

	@Test
	void testDeleteGroup_Success() throws Exception {
		GroupRequestDTO requestDTO = new GroupRequestDTO();
		requestDTO.setName("Test Group");
		requestDTO.setDescription("description");
		requestDTO.setPath("path");
		requestDTO.setMembershipLock(true);
		requestDTO.setVisibility("public");
		requestDTO.setSecret("secret");

		doNothing().when(groupAPIService).deleteGroup(gitLabApi, requestDTO);

		mockMvc.perform(MockMvcRequestBuilders.delete("/api/group/deleteGroup")
				.contentType(MediaType.APPLICATION_JSON)
				.content(requestDTOToJson(requestDTO)))
				.andExpect(status().isOk())
				.andExpect(content().string("Group " + requestDTO.getName() + " deleted successfully"));
	}

	@Test
	public void testAddGroupMemberByEmail() throws Exception {

		UserRequestDTO userRequestDTO = new UserRequestDTO();
		userRequestDTO.setCourseName("CourseEdition A");
		userRequestDTO.setCourseEdition("2023");
		userRequestDTO.setGroupPath("Group 1");
		userRequestDTO.setEmail("test@example.com");
		userRequestDTO.setSecret("secret");
		userRequestDTO.setAccessLevel(10);

		SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
		userRequestDTO.setExpiresAt(format.parse("2023-12-31"));

		User user = new User();
		user.setId(123L);
		when(userAPIService.getUserByEmail(gitLabApi, userRequestDTO.getEmail().toLowerCase()))
				.thenReturn(user);

		Member member = new Member();
		member.setId(789L);
		member.setAccessLevel(AccessLevel.forValue(10));
		member.setExpiresAt(format.parse("2023-12-31"));
		when(groupAPIService.addGroupMember(gitLabApi, "courseeditiona/2023/group1", 123L, 10,
				format.parse("2023-12-31")))
						.thenReturn(member);

		when(groupDBService.getGroupFromGroupPath(anyString())).thenReturn(new StudentGroup());
		when(courseEditionDBService.getCourseEditionFromCoursePath(anyString()))
				.thenReturn(new CourseEdition());
		Access newAccess = Access.builder().accessID(1L).groupID(1L).role(1).accountID("username")
				.courseEditionID(2L).build();
		when(accessDBService.getAccessFromUserEditionGroup(anyString(), anyLong(), anyLong()))
				.thenReturn(newAccess);
		when(accessDBService.saveAccess(any())).thenReturn(newAccess);

		mockMvc.perform(MockMvcRequestBuilders.post("/api/group/addGroupMemberByEmail")
				.contentType(MediaType.APPLICATION_JSON)
				.content(objectMapper.writeValueAsString(userRequestDTO)))
				.andExpect(status().isOk())
				.andExpect(MockMvcResultMatchers.jsonPath("$.accessLevel").value(10))
				.andExpect(MockMvcResultMatchers.jsonPath("$.expiresAt").value(format.parse("2023-12-31")));
	}

	@Test
	public void testAddGroupMemberByEmailAlternative() throws Exception {

		UserRequestDTO userRequestDTO = new UserRequestDTO();
		userRequestDTO.setCourseName("CourseEdition A");
		userRequestDTO.setCourseEdition("2023");
		userRequestDTO.setEmail("test@example.com");
		userRequestDTO.setSecret("secret");
		userRequestDTO.setAccessLevel(10);

		SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
		userRequestDTO.setExpiresAt(format.parse("2023-12-31"));

		User user = new User();
		user.setId(123L);
		when(userAPIService.getUserByEmail(gitLabApi, userRequestDTO.getEmail().toLowerCase()))
				.thenReturn(user);

		Member member = new Member();
		member.setId(789L);
		member.setAccessLevel(AccessLevel.forValue(10));
		member.setExpiresAt(format.parse("2023-12-31"));
		when(groupAPIService.addGroupMember(gitLabApi, "courseeditiona/2023", 123L, 10,
				format.parse("2023-12-31")))
						.thenReturn(member);

		when(groupDBService.getGroupFromGroupPath(anyString())).thenReturn(new StudentGroup());
		when(courseEditionDBService.getCourseEditionFromCoursePath(anyString()))
				.thenReturn(new CourseEdition());
		Access newAccess = Access.builder().accessID(1L).groupID(1L).role(1).accountID("username")
				.courseEditionID(2L).build();
		when(accessDBService.getAccessFromUserEditionGroup(anyString(), anyLong(), anyLong()))
				.thenReturn(newAccess);
		when(accessDBService.saveAccess(any())).thenReturn(newAccess);

		mockMvc.perform(MockMvcRequestBuilders.post("/api/group/addGroupMemberByEmail")
				.contentType(MediaType.APPLICATION_JSON)
				.content(objectMapper.writeValueAsString(userRequestDTO)))
				.andExpect(status().isOk())
				.andExpect(MockMvcResultMatchers.jsonPath("$.accessLevel").value(10))
				.andExpect(MockMvcResultMatchers.jsonPath("$.expiresAt").value(format.parse("2023-12-31")));
	}

	@Test
	public void testRemoveGroupMemberByEmail() throws Exception {

		UserRequestDTO userRequestDTO = new UserRequestDTO();
		userRequestDTO.setCourseName("CourseEdition A");
		userRequestDTO.setCourseEdition("2023");
		userRequestDTO.setGroupPath("Group 1");
		userRequestDTO.setEmail("test@example.com");
		userRequestDTO.setSecret("secret");

		User user = new User();
		user.setId(123L);
		when(userAPIService.getUserByEmail(gitLabApi, userRequestDTO.getEmail().toLowerCase()))
				.thenReturn(user);

		when(groupDBService.getGroupFromGroupPath(anyString())).thenReturn(new StudentGroup());
		when(courseEditionDBService.getCourseEditionFromCoursePath(anyString()))
				.thenReturn(new CourseEdition());
		Access newAccess = Access.builder().accessID(1L).groupID(1L).role(1).accountID("username")
				.courseEditionID(2L).build();
		when(accessDBService.getAccessFromUserEditionGroup(any(), anyLong(), anyLong()))
				.thenReturn(newAccess);
		when(accessDBService.saveAccess(any())).thenReturn(newAccess);

		mockMvc.perform(MockMvcRequestBuilders.post("/api/group/removeGroupMemberByEmail")
				.contentType(MediaType.APPLICATION_JSON)
				.content(objectMapper.writeValueAsString(userRequestDTO)))
				.andExpect(status().isOk())
				.andExpect(MockMvcResultMatchers.jsonPath("$.id").value(123L));
	}

	@Test
	public void testRemoveGroupMemberByEmailAlternative() throws Exception {

		UserRequestDTO userRequestDTO = new UserRequestDTO();
		userRequestDTO.setCourseName("CourseEdition A");
		userRequestDTO.setCourseEdition("2023");
		userRequestDTO.setEmail("test@example.com");
		userRequestDTO.setSecret("secret");

		User user = new User();
		user.setId(123L);
		when(userAPIService.getUserByEmail(gitLabApi, userRequestDTO.getEmail().toLowerCase()))
				.thenReturn(user);

		when(groupDBService.getGroupFromGroupPath(anyString())).thenReturn(new StudentGroup());
		when(courseEditionDBService.getCourseEditionFromCoursePath(anyString()))
				.thenReturn(new CourseEdition());
		Access newAccess = Access.builder().accessID(1L).groupID(1L).role(1).accountID("username")
				.courseEditionID(2L).build();
		when(accessDBService.getAccessFromUserEdition(any(), anyLong()))
				.thenReturn(newAccess);
		when(accessDBService.saveAccess(any())).thenReturn(newAccess);

		mockMvc.perform(MockMvcRequestBuilders.post("/api/group/removeGroupMemberByEmail")
				.contentType(MediaType.APPLICATION_JSON)
				.content(objectMapper.writeValueAsString(userRequestDTO)))
				.andExpect(status().isOk())
				.andExpect(MockMvcResultMatchers.jsonPath("$.id").value(123L));
	}

	@Test
	public void testUpdateGroupMemberByEmail() throws Exception {
		UserRequestDTO userRequestDTO = new UserRequestDTO();
		userRequestDTO.setCourseName("CourseEdition A");
		userRequestDTO.setCourseEdition("2023");
		userRequestDTO.setGroupPath("Group 1");
		userRequestDTO.setEmail("test@example.com");
		userRequestDTO.setSecret("secret");
		userRequestDTO.setAccessLevel(10);
		SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
		userRequestDTO.setExpiresAt(format.parse("2023-12-31"));

		User user = new User();
		user.setId(123L);
		when(userAPIService.getUserByEmail(gitLabApi, userRequestDTO.getEmail().toLowerCase()))
				.thenReturn(user);

		Member member = new Member();
		member.setId(789L);
		member.setAccessLevel(AccessLevel.forValue(10));
		member.setExpiresAt(format.parse("2023-12-31"));
		when(groupAPIService.updateGroupMember(gitLabApi, "courseeditiona/2023/group1", 123L, 10,
				format.parse("2023-12-31")))
						.thenReturn(member);

		when(groupDBService.getGroupFromGroupPath(anyString())).thenReturn(new StudentGroup());
		when(courseEditionDBService.getCourseEditionFromCoursePath(anyString()))
				.thenReturn(new CourseEdition());
		Access newAccess = Access.builder().accessID(1L).groupID(1L).role(1).accountID("username")
				.courseEditionID(2L).build();
		when(accessDBService.getAccessFromUserEditionGroup(any(), anyLong(), anyLong()))
				.thenReturn(newAccess);
		when(accessDBService.saveAccess(any())).thenReturn(newAccess);

		mockMvc.perform(MockMvcRequestBuilders.put("/api/group/updateGroupMemberByEmail")
				.contentType(MediaType.APPLICATION_JSON)
				.content(objectMapper.writeValueAsString(userRequestDTO)))
				.andExpect(status().isOk())
				.andExpect(MockMvcResultMatchers.jsonPath("$.id").value(789L))
				.andExpect(MockMvcResultMatchers.jsonPath("$.accessLevel").value(10))
				.andExpect(MockMvcResultMatchers.jsonPath("$.expiresAt").value(format.parse("2023-12-31")));
	}

	@Test
	public void testUpdateGroupMemberByEmailAlternative() throws Exception {
		UserRequestDTO userRequestDTO = new UserRequestDTO();
		userRequestDTO.setCourseName("CourseEdition A");
		userRequestDTO.setCourseEdition("2023");
		userRequestDTO.setEmail("test@example.com");
		userRequestDTO.setSecret("secret");
		userRequestDTO.setAccessLevel(10);
		SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
		userRequestDTO.setExpiresAt(format.parse("2023-12-31"));

		User user = new User();
		user.setId(123L);
		when(userAPIService.getUserByEmail(gitLabApi, userRequestDTO.getEmail().toLowerCase()))
				.thenReturn(user);

		Member member = new Member();
		member.setId(789L);
		member.setAccessLevel(AccessLevel.forValue(10));
		member.setExpiresAt(format.parse("2023-12-31"));
		when(groupAPIService.updateGroupMember(gitLabApi, "courseeditiona/2023", 123L, 10,
				format.parse("2023-12-31")))
						.thenReturn(member);

		when(groupDBService.getGroupFromGroupPath(anyString())).thenReturn(new StudentGroup());
		when(courseEditionDBService.getCourseEditionFromCoursePath(anyString()))
				.thenReturn(new CourseEdition());
		Access newAccess = Access.builder().accessID(1L).groupID(1L).role(1).accountID("username")
				.courseEditionID(2L).build();
		when(accessDBService.getAccessFromUserEdition(any(), anyLong()))
				.thenReturn(newAccess);
		when(accessDBService.saveAccess(any())).thenReturn(newAccess);

		mockMvc.perform(MockMvcRequestBuilders.put("/api/group/updateGroupMemberByEmail")
				.contentType(MediaType.APPLICATION_JSON)
				.content(objectMapper.writeValueAsString(userRequestDTO)))
				.andExpect(status().isOk())
				.andExpect(MockMvcResultMatchers.jsonPath("$.id").value(789L))
				.andExpect(MockMvcResultMatchers.jsonPath("$.accessLevel").value(10))
				.andExpect(MockMvcResultMatchers.jsonPath("$.expiresAt").value(format.parse("2023-12-31")));
	}

	@Test
	public void testAddMembersInBatches() throws Exception {
		UserRequestDTO user1 = new UserRequestDTO();
		user1.setCourseName("CourseEdition A");
		user1.setCourseEdition("2023");
		user1.setGroupPath("Group 1");
		user1.setEmail("test1@example.com");
		user1.setAccessLevel(10);
		user1.setSecret("secret");

		UserRequestDTO user2 = new UserRequestDTO();
		user2.setCourseName("CourseEdition B");
		user2.setCourseEdition("2024");
		user2.setGroupPath("Group 2");
		user2.setEmail("test2@example.com");
		user2.setSecret("secret");
		user2.setAccessLevel(20);

		List<UserRequestDTO> users = Arrays.asList(user1, user2);

		User user = new User();
		user.setId(123L);
		when(userAPIService.getUserByEmail(gitLabApi, user1.getEmail().toLowerCase())).thenReturn(user);

		Member member1 = new Member();
		member1.setId(789L);
		member1.setAccessLevel(AccessLevel.forValue(10));
		member1.setExpiresAt(new Date());
		Member member2 = new Member();
		member2.setId(890L);
		member2.setAccessLevel(AccessLevel.forValue(20));
		member2.setExpiresAt(new Date());

		User user22 = new User();
		user22.setId(124L);

		when(userAPIService.getUserByEmail(gitLabApi, user2.getEmail().toLowerCase())).thenReturn(user22);

		when(groupAPIService.addGroupMember(gitLabApi, "/courseeditiona/2023/group1", user.getId(),
				user1.getAccessLevel(), user1.getExpiresAt()))
						.thenReturn(member1);
		when(groupAPIService.addGroupMember(gitLabApi, "/courseeditionb/2024/group2", user22.getId(),
				user2.getAccessLevel(), user2.getExpiresAt()))
						.thenReturn(member2);

		BatchRequestDTO batchRequestDTO = new BatchRequestDTO();
		batchRequestDTO.setUsers(users);
		batchRequestDTO.setSecret("secret");

		mockMvc.perform(MockMvcRequestBuilders.post("/api/group/addMembersInBatches")
				.contentType(MediaType.APPLICATION_JSON)
				.content(objectMapper.writeValueAsString(batchRequestDTO)))
				.andExpect(status().isOk())
				.andExpect(MockMvcResultMatchers.jsonPath("$[0].accessLevel").value(10))
				.andExpect(MockMvcResultMatchers.jsonPath("$[1].accessLevel").value(20));
	}

	@Test
	public void testRemoveMembersInBatches() throws Exception {
		UserRequestDTO user1 = new UserRequestDTO();
		user1.setCourseName("CourseEdition A");
		user1.setCourseEdition("2023");
		user1.setGroupPath("Group 1");
		user1.setEmail("test1@example.com");
		user1.setSecret("secret");

		UserRequestDTO user2 = new UserRequestDTO();
		user2.setCourseName("CourseEdition B");
		user2.setCourseEdition("2024");
		user2.setGroupPath("Group 2");
		user2.setEmail("test2@example.com");
		user2.setSecret("secret");

		List<UserRequestDTO> users = Arrays.asList(user1, user2);

		User user = new User();
		user.setId(123L);
		when(userAPIService.getUserByEmail(gitLabApi, user1.getEmail().toLowerCase())).thenReturn(user);

		User user22 = new User();
		user22.setId(123L);
		when(userAPIService.getUserByEmail(gitLabApi, user2.getEmail().toLowerCase())).thenReturn(user22);

		BatchRequestDTO batchRequestDTO = new BatchRequestDTO();
		batchRequestDTO.setUsers(users);
		batchRequestDTO.setSecret("secret");

		mockMvc.perform(MockMvcRequestBuilders.delete("/api/group/removeMembersInBatches")
				.contentType(MediaType.APPLICATION_JSON)
				.content(objectMapper.writeValueAsString(batchRequestDTO)))
				.andExpect(status().isOk());
	}

	@Test
	public void testGetSubgroupPaths() throws Exception {
		String groupPath = "step-step-step";

		GroupRequestDTO groupRequestDTO = new GroupRequestDTO();
		groupRequestDTO.setPath(groupPath);
		groupRequestDTO.setSecret("secret");

		List<String> expectedPaths = List.of("left", "foot", "right", "foot", "levitate");

		when(groupAPIService.getSubgroupPaths(gitLabApi, groupPath)).thenReturn(expectedPaths);
		ResponseEntity<List<String>> result = groupManagementController.getSubgroupPaths(groupRequestDTO);

		assertEquals(expectedPaths, result.getBody());
		assertEquals(HttpStatus.CREATED, result.getStatusCode());
	}

	@Test
	public void testGetSubgroupPathsFailInstance() throws Exception {
		String groupPath = "step-step-step";

		when(instanceMappingService.getInstance("admin")).thenReturn(null);
		List<Group> subgroups = List.of(new Group().withPath("left"), new Group().withPath("foot"),
				new Group().withPath("right"), new Group().withPath("foot"),
				new Group().withPath("levitate"));
		when(groupApi.getSubGroups(groupPath)).thenReturn(subgroups);

		GroupRequestDTO groupRequestDTO = new GroupRequestDTO();
		groupRequestDTO.setSecret("secret");
		groupRequestDTO.setPath(groupPath);

		ResponseEntity<List<String>> result = groupManagementController.getSubgroupPaths(groupRequestDTO);

		assertNull(result.getBody());
		assertEquals(HttpStatus.UNAUTHORIZED, result.getStatusCode());
	}

	@Test
	public void testGetSubgroupPathsFailAPI() throws Exception {
		String groupPath = "step-step-step";

		GroupRequestDTO groupRequestDTO = new GroupRequestDTO();
		groupRequestDTO.setPath(groupPath);
		groupRequestDTO.setSecret("secret");

		when(groupAPIService.getSubgroupPaths(gitLabApi, groupPath))
				.thenThrow(new GitLabApiException("Not really", 404));

		ResponseEntity<List<String>> result = groupManagementController.getSubgroupPaths(groupRequestDTO);

		assertNull(result.getBody());
		assertEquals(HttpStatus.NOT_FOUND, result.getStatusCode());
	}

	@Test
	public void updateGroup() throws Exception {
		GroupRequestDTO requestDTO = new GroupRequestDTO();
		requestDTO.setName("Test Group");
		requestDTO.setDescription("description");
		requestDTO.setPath("path");
		requestDTO.setMembershipLock(true);
		requestDTO.setDefaultBranchProtectionLevel(GroupParams.DefaultBranchProtectionLevel.FULLY_PROTECTED);
		requestDTO.setVisibility("public");
		requestDTO.setSecret("secret");

		Group group = new Group().withPath("path").withName("Test Group").withDescription("description")
				.withVisibility(Visibility.PUBLIC);

		when(instanceMappingService.getInstance("admin")).thenReturn(null);
		ResponseEntity<?> resultNull = groupManagementController.updateGroup(requestDTO);
		assertNull(resultNull.getBody());
		assertEquals(HttpStatus.UNAUTHORIZED, resultNull.getStatusCode());

		when(instanceMappingService.getInstance("admin")).thenReturn(gitLabApi);
		when(groupAPIService.updateGroup(any(GitLabApi.class), eq(requestDTO.getPath()),
				any(GroupParams.class))).thenReturn(group);

		ResponseEntity<?> result = groupManagementController.updateGroup(requestDTO);
		Group requestGroup = (Group) result.getBody();

		assertEquals("Test Group", requestGroup.getName());
		assertEquals("path", requestGroup.getPath());
		assertEquals("description", requestGroup.getDescription());
	}

	@Test
	public void updateGroupAndSubgroups() throws Exception {
		GroupRequestDTO requestDTO = new GroupRequestDTO();
		requestDTO.setName("Test Group");
		requestDTO.setDescription("description");
		requestDTO.setPath("path");
		requestDTO.setMembershipLock(true);
		requestDTO.setDefaultBranchProtectionLevel(GroupParams.DefaultBranchProtectionLevel.FULLY_PROTECTED);
		requestDTO.setVisibility("public");
		requestDTO.setSecret("secret");

		Group group = new Group().withPath("path").withName("Test Group").withDescription("description")
				.withVisibility(Visibility.PUBLIC);

		when(instanceMappingService.getInstance("admin")).thenReturn(null);
		ResponseEntity<?> resultNull = groupManagementController.updateGroupAndSubgroups(requestDTO);
		assertNull(resultNull.getBody());
		assertEquals(HttpStatus.UNAUTHORIZED, resultNull.getStatusCode());

		when(instanceMappingService.getInstance("admin")).thenReturn(gitLabApi);

		when(groupAPIService.getGroup(any(GitLabApi.class), eq("path"))).thenReturn(group);
		when(groupAPIService.updateGroup(any(GitLabApi.class), eq(group), any(GroupParams.class)))
				.thenReturn(group);

		Group group1 = new Group().withName("Name 1").withParentId(0L).withPath("path/go");
		Group group2 = new Group().withName("FailedGroup").withPath("fail");

		when(groupAPIService.getSubgroups(any(GitLabApi.class), any(Group.class)))
				.thenReturn(List.of(group1, group2));
		when(groupAPIService.updateGroup(any(GitLabApi.class), eq(group1), any(GroupParams.class)))
				.thenReturn(new Group().withName("Name 1").withDescription("description").withParentId(0L)
						.withPath("path/go"));
		when(groupAPIService.updateGroup(any(GitLabApi.class), eq(group2), any(GroupParams.class)))
				.thenThrow(new GitLabApiException("NOT FOUND", 404));

		ResponseEntity<?> result = groupManagementController.updateGroupAndSubgroups(requestDTO);
		assertEquals(HttpStatus.OK, result.getStatusCode());

		List<String> resultLog = (List<String>) result.getBody();

		String expectedLog = "Error updating group [FailedGroup]. Reason: 404 NOT FOUND";

		assertEquals(List.of(expectedLog), resultLog);
	}
}
