package server.csvconverter;

import static org.junit.jupiter.api.Assertions.*;
import static org.junit.jupiter.api.Assertions.assertEquals;

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import server.entity.Account;
import server.entity.StudentGroup;

public class CSVServiceTest {

	private CSVService csvService;
	private String testContent;

	@BeforeEach
	void setUp() {
		csvService = new CSVService();
		testContent = "OrgDefinedId,Username,Last Name,First Name,Email,SP,End-of-Line Indicator\n" +
				"#5678901,#lipsum1@tudelft.nl,Lorem,Ipsum,L.Ipsum1@student.tudelft.nl,1A,#\n" +
				"#5678902,#lipsum2@tudelft.nl,Lorem,Ipsum,L.Ipsum2@student.tudelft.nl,1A,#\n" +
				"#5678903,#lipsum3@tudelft.nl,Lorem,Ipsum,L.Ipsum3@student.tudelft.nl,1A,#\n" +
				"#5678904,#lipsum4@tudelft.nl,Lorem,Ipsum,L.Ipsum4@student.tudelft.nl,1A,#\n" +
				"#5678905,#lipsum5@tudelft.nl,Lorem,Ipsum,L.Ipsum5@student.tudelft.nl,1A,#\n" +
				"#5678906,#lipsum6@tudelft.nl,Lorem,Ipsum,L.Ipsum6@student.tudelft.nl,1A,#\n" +
				"#5678907,#lipsum7@tudelft.nl,Lorem,Ipsum,L.Ipsum7@student.tudelft.nl,1A,#\n" +
				"#5678908,#regular@tudelft.nl,Lorem,Ipsum,L.Ipsum0@student.tudelft.nl,2A,#\n" +
				"#7777777,#comma@tudelft.nl,Funny,Name,Bro,FunnyNameBro@student.tudelft.nl,2A,#\n" +
				"#0000000,#invalid@tudeft.nl,Invalid,User,invalid@student.tudeft.nl,2A,#\n";
	}

	@Test
	void testFormatter() {
		String fileContent = "{\"fileContent\":\"Hello\\rWorld!\"}";
		String expectedFormattedContents = "HelloWorld!";

		String formattedContents = csvService.format(fileContent);

		assertEquals(expectedFormattedContents, formattedContents);
	}

	@Test
	void testConvertToEntities() throws Exception {
		Map<StudentGroup, Set<Account>> result = csvService.extractGroups(testContent, 5);

		assertNotNull(result);
		assertFalse(result.isEmpty());
		assertEquals(2, result.size());

		StudentGroup group1 = new StudentGroup("1A");

		assertTrue(result.containsKey(group1));
		Set<Account> group1Accounts = result.get(group1);
		assertEquals(7, group1Accounts.size());

		StudentGroup group2 = new StudentGroup("2A");
		assertTrue(result.containsKey(group2));
		Set<Account> group2Accounts = result.get(group2);
		assertEquals(2, group2Accounts.size());

		assertTrue(group2Accounts.stream().map(x -> x.getNetId()).toList()
				.containsAll(List.of("regular", "comma")));
	}

	@Test
	void testValidateStudentEmail() {
		assertTrue(csvService.validateStudentEmail("L.Ipsum@student.tudelft.nl"));
		assertFalse(csvService.validateStudentEmail("L.Ipsum@student.tudelft.nl,"));
		assertFalse(csvService.validateStudentEmail("L.Ipsum@student.tudeft.nl,"));
		assertFalse(csvService.validateStudentEmail("L.Ipsum@student.tudelft.nl,"));
		assertFalse(csvService.validateStudentEmail("L.Ipsum@student.tudelft.n"));
	}

	@Test
	void testGetName() {
		String fullName = "Yoshi City";
		assertEquals(fullName, csvService
				.getFullName(List.of("#IGNORE", "#NETID", "Yoshi", "City", "EMAIL", "GROUP", "#"), 0));
		String longName = "This Is A Long Name";
		assertEquals(longName, csvService
				.getFullName(List.of("#", "#", "This", "Is", "A", "Long", "Name", "#", "#", "#"), 3));
		String weirdCase = "This Is A";
		assertEquals(weirdCase, csvService
				.getFullName(List.of("#", "#", "This", "Is", "A", "Long", "Name", "#", "#", "#"), 1));
	}

}
