package server.labraService;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.anyList;
import static org.mockito.Mockito.when;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import nl.tudelft.labracore.api.CourseControllerApi;
import nl.tudelft.labracore.api.EditionControllerApi;
import nl.tudelft.labracore.api.PersonControllerApi;
import nl.tudelft.labracore.api.RoleControllerApi;
import nl.tudelft.labracore.api.dto.*;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

class LabraConnectionServiceTest {
	@Mock
	private EditionControllerApi editionControllerApi;

	@Mock
	private RoleControllerApi roleControllerApi;

	@Mock
	private PersonControllerApi personControllerApi;

	@Mock
	private CourseControllerApi courseControllerApi;

	@InjectMocks
	private LabraConnectionService labraConnectionService;

	private Flux<EditionDetailsDTO> allEditions;

	private Flux<RolePersonDetailsDTO> roles;

	private Flux<CourseSummaryDTO> allCourses;

	@BeforeEach
	void setUp() {
		MockitoAnnotations.openMocks(this);

		CourseSummaryDTO c1 = new CourseSummaryDTO();
		c1.setName("Computer Organization");

		CourseSummaryDTO c2 = new CourseSummaryDTO();
		c2.setName("Object Oriented Programming");

		EditionDetailsDTO e1 = new EditionDetailsDTO();
		e1.setId(1L);
		e1.course(c1);
		e1.setName("NOW");
		EditionDetailsDTO e2 = new EditionDetailsDTO();
		e2.setId(2L);
		e2.course(c1);
		e2.setName("21-22");
		EditionDetailsDTO e3 = new EditionDetailsDTO();
		e3.setId(3L);
		e3.course(c2);
		e3.setName("NOW");

		List<EditionDetailsDTO> allEditionsList = new ArrayList<>();
		allEditionsList.add(e1);
		allEditionsList.add(e2);
		allEditionsList.add(e3);

		List<CourseSummaryDTO> allCoursesList = new ArrayList<>();
		allCoursesList.add(c1);
		allCoursesList.add(c2);

		allEditions = Flux.fromIterable(allEditionsList);

		roles = Flux.fromIterable(Arrays.asList(
				createRolePersonDetailsDTO("ta1@example.com", RolePersonDetailsDTO.TypeEnum.TA),
				createRolePersonDetailsDTO("ta2@example.com", RolePersonDetailsDTO.TypeEnum.TA),
				createRolePersonDetailsDTO("headta1@example.com", RolePersonDetailsDTO.TypeEnum.HEAD_TA),
				createRolePersonDetailsDTO("teacher1@example.com", RolePersonDetailsDTO.TypeEnum.TEACHER)));

		allCourses = Flux.fromIterable(allCoursesList);

	}

	@Test
	void testGetAllCourseNames() {
		when(courseControllerApi.getAllCourses()).thenReturn(allCourses);

		List<String> courseNames = labraConnectionService.getAllCourseNames();

		List<String> correctCourseNames = new ArrayList<>();
		correctCourseNames.add("Computer Organization");
		correctCourseNames.add("Object Oriented Programming");

		assertEquals(courseNames, correctCourseNames);
	}

	@Test
	void testGetStaffEmailsForCourse() {
		when(editionControllerApi.getAllEditions()).thenReturn(allEditions);
		when(roleControllerApi.getRolesByEditions(anyList())).thenReturn(roles);

		List<List<String>> correctRes = new ArrayList<>();

		List<String> taMail = new ArrayList<>();
		taMail.add("ta1@example.com");
		taMail.add("ta2@example.com");

		correctRes.add(taMail);

		List<String> headTaMail = new ArrayList<>();
		headTaMail.add("headta1@example.com");

		correctRes.add(headTaMail);

		List<String> teacherMail = new ArrayList<>();
		teacherMail.add("teacher1@example.com");

		correctRes.add(teacherMail);

		List<List<String>> res = labraConnectionService
				.getStaffEmailsForCourseEdition("Computer Organization", "NOW");

		assertEquals(res, correctRes);
	}

	@Test
	public void testGetTeacherAssistantsForCourseEdition() {
		when(editionControllerApi.getAllEditions()).thenReturn(allEditions);
		when(roleControllerApi.getRolesByEditions(anyList())).thenReturn(roles);

		List<PersonSummaryDTO> result = labraConnectionService
				.getTeacherAssistantsForCourseEdition("Computer Organization", "NOW");

		assertEquals(3, result.size());
	}

	private RolePersonDetailsDTO createRolePersonDetailsDTO(String mail, RolePersonDetailsDTO.TypeEnum role) {
		RolePersonDetailsDTO res = new RolePersonDetailsDTO();

		PersonSummaryDTO person = new PersonSummaryDTO();
		person.setEmail(mail);
		res.setPerson(person);
		res.setType(role);

		return res;
	}

	@Test
	public void getPersonTest() {
		PersonDetailsDTO expected = new PersonDetailsDTO();
		expected.setDisplayName("the admin");
		expected.setUsername("admin");
		expected.setEmail("admin@admin");

		when(personControllerApi.getPersonByUsername("admin")).thenReturn(Mono.just(expected));

		PersonDetailsDTO result = labraConnectionService.getPerson("admin");

		assertEquals(expected.getEmail(), result.getEmail());
		assertEquals(expected.getUsername(), result.getUsername());
		assertEquals(expected.getDisplayName(), result.getDisplayName());
	}
}
