package server.logging;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.*;

import java.io.FileWriter;
import java.io.IOException;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

public class LogWriterTest {
	private LogWriter logWriter;
	private FileWriter fileWriter;

	private LocalDateTime time;

	@BeforeEach
	public void setUp() throws IOException {
		logWriter = spy(new LogWriter());
		time = LocalDateTime.now();
		when(logWriter.currentTime()).thenReturn(time);
		logWriter.setLogFailures(true);

		fileWriter = mock(FileWriter.class);
		doNothing().when(fileWriter).write(any(String.class));
		when(logWriter.getFileWriter()).thenReturn(fileWriter);
	}

	@Test
	public void writeMemberLog() throws IOException {
		LogType logType = LogType.ERROR;
		LogType actionType = LogType.USER_ADDED_TO_PROJECT;
		String userInfo = "Scary user";
		String destinationInfo = "Scary project";
		String responsibleUser = "Evil Clown";
		String reason = "he failed";

		logWriter.writeMemberActionLog(logType, actionType, userInfo, destinationInfo, responsibleUser,
				reason);
		logWriter.writeMemberActionLog(logType, actionType, userInfo, destinationInfo, responsibleUser);
		String expectedMessageReason = "[ERROR] ["
				+ time.format(DateTimeFormatter.ofPattern("uuuu-MM-dd HH:mm:ss.SSS"))
				+ "] User Scary user was NOT added by Evil Clown to project Scary project; he failed";
		String expectedMessage = "[ERROR] ["
				+ time.format(DateTimeFormatter.ofPattern("uuuu-MM-dd HH:mm:ss.SSS"))
				+ "] User Scary user was NOT added by Evil Clown to project Scary project";

		assertEquals(expectedMessageReason,
				logWriter.generateMessage(logType, actionType, userInfo, responsibleUser) + " to project "
						+ destinationInfo + "; " + reason);

		assertEquals(expectedMessage,
				logWriter.generateMessage(logType, actionType, userInfo, responsibleUser) + " to project "
						+ destinationInfo);

		verify(fileWriter, times(1)).write(expectedMessageReason + System.lineSeparator());
		verify(fileWriter, times(1)).write(expectedMessage + System.lineSeparator());
	}

	@Test
	public void writeLogInfo() throws IOException {
		LogType logType = LogType.INFO;
		LogType actionType = LogType.COURSE_CREATED;
		String targetInfo = "Scary Course";
		String responsibleUser = "Evil Clown";

		logWriter.writeActionLog(logType, actionType, targetInfo, responsibleUser);

		String expectedMessage = "[INFO] ["
				+ time.format(DateTimeFormatter.ofPattern("uuuu-MM-dd HH:mm:ss.SSS"))
				+ "] Course Scary Course was created by Evil Clown";

		assertEquals(expectedMessage,
				logWriter.generateMessage(logType, actionType, targetInfo, responsibleUser));

		verify(fileWriter, times(1)).write(expectedMessage + System.lineSeparator());
	}

	@Test
	public void writeLogError() throws IOException {
		LogType logType = LogType.ERROR;
		LogType actionType = LogType.COURSE_CREATED;
		String targetInfo = "Scary Course";
		String responsibleUser = "Evil Clown";

		logWriter.writeActionLog(logType, actionType, targetInfo, responsibleUser);

		String expectedMessage = "[ERROR] ["
				+ time.format(DateTimeFormatter.ofPattern("uuuu-MM-dd HH:mm:ss.SSS"))
				+ "] Course Scary Course was NOT created by Evil Clown";

		assertEquals(expectedMessage,
				logWriter.generateMessage(logType, actionType, targetInfo, responsibleUser));

		verify(fileWriter, times(1)).write(expectedMessage + System.lineSeparator());
	}

	@Test
	public void writeLogAdditionalMessage() throws IOException {
		LogType logType = LogType.ERROR;
		LogType actionType = LogType.COURSE_CREATED;
		String targetInfo = "Scary Course";
		String responsibleUser = "Evil Clown";
		String reason = "he failed";

		logWriter.writeActionLog(logType, actionType, targetInfo, responsibleUser, reason);

		String expectedMessage = "[ERROR] ["
				+ time.format(DateTimeFormatter.ofPattern("uuuu-MM-dd HH:mm:ss.SSS"))
				+ "] Course Scary Course was NOT created by Evil Clown; he failed";

		assertEquals(expectedMessage,
				logWriter.generateMessage(logType, actionType, targetInfo, responsibleUser) + "; " + reason);

		verify(fileWriter, times(1)).write(expectedMessage + System.lineSeparator());
	}

	@Test
	public void writeLogErrorsDisabled() throws IOException {
		logWriter.setLogFailures(false);
		LogType logType = LogType.ERROR;
		LogType actionType = LogType.USER_CREATED;
		String targetInfo = "hackerman";
		String responsibleUser = "hater";

		logWriter.writeActionLog(logType, actionType, targetInfo, responsibleUser);

		verify(fileWriter, never()).write(any(String.class));
	}

	@Test
	public void writeLogFailure() throws IOException {
		LogType logType = LogType.ERROR;
		LogType actionType = LogType.COURSE_CREATED;
		String targetInfo = "Scary Course";
		String responsibleUser = "Evil Clown";
		String reason = "he failed";

		doThrow(new IOException("Ahahaha")).when(fileWriter).write(any(String.class));

		logWriter.writeActionLog(logType, actionType, targetInfo, responsibleUser, reason);
	}

	@Test
	public void invalidLogType() {
		RuntimeException e1 = assertThrows(RuntimeException.class, () -> {
			logWriter.writeActionLog(LogType.COURSE_CREATED, LogType.COURSE_CREATED, "this", "will", "fail");
		});

		assertEquals("Unexpected logType: COURSE_CREATED", e1.getMessage());

		RuntimeException e2 = assertThrows(RuntimeException.class, () -> {
			logWriter.writeActionLog(LogType.INFO, LogType.INFO, "this", "will", "fail");
		});

		assertEquals("Unexpected actionType: INFO", e2.getMessage());
	}

	@Test
	public void generateMessageTests() {
		String timestamp = time.format(DateTimeFormatter.ofPattern("uuuu-MM-dd HH:mm:ss.SSS"));

		LogType logType = LogType.INFO;
		LogType actionType = LogType.USER_CREATED;
		String targetInfo = "A";
		String responsibleUser = "B";
		String expectedMessage = "[INFO] [" + timestamp + "] User A was created by B";

		assertEquals(expectedMessage,
				logWriter.generateMessage(logType, actionType, targetInfo, responsibleUser));

		actionType = LogType.USER_DELETED;
		expectedMessage = "[INFO] [" + timestamp + "] User A was deleted by B";
		assertEquals(expectedMessage,
				logWriter.generateMessage(logType, actionType, targetInfo, responsibleUser));

		actionType = LogType.COURSE_DELETED;
		expectedMessage = "[INFO] [" + timestamp + "] Course A was deleted by B";
		assertEquals(expectedMessage,
				logWriter.generateMessage(logType, actionType, targetInfo, responsibleUser));

		actionType = LogType.EDITION_CREATED;
		expectedMessage = "[INFO] [" + timestamp + "] Course edition A was created by B";
		assertEquals(expectedMessage,
				logWriter.generateMessage(logType, actionType, targetInfo, responsibleUser));

		actionType = LogType.EDITION_DELETED;
		expectedMessage = "[INFO] [" + timestamp + "] Course edition A was deleted by B";
		assertEquals(expectedMessage,
				logWriter.generateMessage(logType, actionType, targetInfo, responsibleUser));

		actionType = LogType.GROUP_CREATED;
		expectedMessage = "[INFO] [" + timestamp + "] Group A was created by B";
		assertEquals(expectedMessage,
				logWriter.generateMessage(logType, actionType, targetInfo, responsibleUser));

		actionType = LogType.GROUP_DELETED;
		expectedMessage = "[INFO] [" + timestamp + "] Group A was deleted by B";
		assertEquals(expectedMessage,
				logWriter.generateMessage(logType, actionType, targetInfo, responsibleUser));

		actionType = LogType.PROJECT_CREATED;
		expectedMessage = "[INFO] [" + timestamp + "] Project A was created by B";
		assertEquals(expectedMessage,
				logWriter.generateMessage(logType, actionType, targetInfo, responsibleUser));

		actionType = LogType.PROJECT_DELETED;
		expectedMessage = "[INFO] [" + timestamp + "] Project A was deleted by B";
		assertEquals(expectedMessage,
				logWriter.generateMessage(logType, actionType, targetInfo, responsibleUser));

		actionType = LogType.PROJECT_FORKED;
		expectedMessage = "[INFO] [" + timestamp + "] Project A was forked by B";
		assertEquals(expectedMessage,
				logWriter.generateMessage(logType, actionType, targetInfo, responsibleUser));
	}
}
