package server.service;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.*;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import javax.ws.rs.core.Application;

import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInstance;
import org.mockito.Mock;
import org.springframework.boot.test.context.SpringBootTest;

import server.entity.Access;
import server.repository.AccessRepository;

@SpringBootTest(classes = Application.class)
@TestInstance(TestInstance.Lifecycle.PER_CLASS)
class AccessServiceTest {

	@Mock
	private AccessRepository mockRepository;

	@Test
	public void testSaveAccessNewAccess() {
		AccessService accessService = new AccessService();
		accessService.setRepository(mockRepository);

		Access access = Access.builder().groupID(1L).accountID("testAccount").role(2).courseEditionID(3L)
				.build();
		when(mockRepository.save(access)).thenReturn(access);

		// Call the method under test
		Access savedAccess = accessService.saveAccess(access);

		// Verify that the accessRepository.save method was called
		verify(mockRepository).save(access);

		// Assert the createTime and lastModify fields were set correctly
		assertEquals(access.getLastModified(), access.getTimeCreated());

		assertEquals(access, savedAccess);
	}

	@Test
	public void testSaveAccessUpdate() {
		AccessService accessService = new AccessService();
		accessService.setRepository(mockRepository);

		Access access = Access.builder().groupID(1L).accountID("testAccount").role(2).courseEditionID(3L)
				.build();
		Access accessOld = Access.builder().groupID(1L).accountID("testAccountNew").role(3)
				.courseEditionID(3L)
				.timeCreated(LocalDateTime.now().minusDays(1)).build();
		when(mockRepository.save(access)).thenReturn(access);
		when(mockRepository.findById(any())).thenReturn(Optional.ofNullable(accessOld));

		// Call the method under test
		Access savedAccess = accessService.saveAccess(access);

		// Verify that the accessRepository.save method was called
		verify(mockRepository).save(access);

		// Assert the createTime and lastModify fields were set correctly
		assertTrue(savedAccess.getLastModified().isAfter(savedAccess.getTimeCreated()));

		assertEquals(access, savedAccess);
	}

	@Test
	public void testDeleteAccess() {
		AccessService accessService = new AccessService();
		accessService.setRepository(mockRepository);

		long accessId = 1L;
		doNothing().when(mockRepository).deleteById(accessId);
		accessService.deleteAccess(accessId);
		verify(mockRepository, times(1)).deleteById(accessId);
	}

	@Test
	public void testFetchAllAccesses() {
		AccessService accessService = new AccessService();
		accessService.setRepository(mockRepository);

		List<Access> accesses = new ArrayList<>();
		Access access1 = Access.builder().groupID(1L).accountID("testAccount").role(2).courseEditionID(3L)
				.build();
		Access access2 = Access.builder().groupID(2L).accountID("testAccount").role(1).courseEditionID(4L)
				.build();

		accesses.add(access1);
		accesses.add(access2);
		when(mockRepository.findAll()).thenReturn(accesses);
		List<Access> fetchedAccesses = accessService.fetchAllAccesses();
		assertEquals(accesses, fetchedAccesses);
	}

	@Test
	public void testFetchAllAccessesOfAccount() {
		AccessService accessService = new AccessService();
		accessService.setRepository(mockRepository);

		List<Access> accesses = new ArrayList<>();
		List<Access> goodAccesses = new ArrayList<>();
		Access access1 = Access.builder().groupID(1L).accountID("testAccount").role(2).courseEditionID(3L)
				.build();
		Access access2 = Access.builder().groupID(2L).accountID("testAccount").role(1).courseEditionID(4L)
				.build();
		Access access3 = Access.builder().groupID(2L).accountID("notTheRightOne").role(4).courseEditionID(4L)
				.build();

		accesses.add(access1);
		accesses.add(access2);
		accesses.add(access3);

		goodAccesses.add(access1);
		goodAccesses.add(access2);

		String accountId = "testAccount";

		when(mockRepository.findAll()).thenReturn(accesses);
		List<Access> fetchedAccesses = accessService.fetchAllAccessesOfAccount(accountId);
		assertEquals(goodAccesses, fetchedAccesses);
	}

	@Test
	public void testGetAccessFromUserEdition() {
		AccessService accessService = new AccessService();
		accessService.setRepository(mockRepository);
		String userId = "testUser";
		long editionId = 1L;

		Access access1 = Access.builder()
				.groupID(1L)
				.accountID(userId)
				.role(2)
				.courseEditionID(editionId)
				.build();
		Access access2 = Access.builder()
				.groupID(2L)
				.accountID(userId)
				.role(1)
				.courseEditionID(editionId)
				.build();
		Access access3 = Access.builder()
				.groupID(3L)
				.accountID("otherUser")
				.role(3)
				.courseEditionID(editionId)
				.build();

		List<Access> allAccesses = new ArrayList<>();
		allAccesses.add(access1);
		allAccesses.add(access2);
		allAccesses.add(access3);

		when(mockRepository.findAll()).thenReturn(allAccesses);

		Access result = accessService.getAccessFromUserEdition(userId, editionId);

		assertEquals(access1, result);
	}

	@Test
	public void testGetAccessFromUserEditionGroup() {
		AccessService accessService = new AccessService();
		accessService.setRepository(mockRepository);
		String userId = "testUser";
		long editionId = 1L;
		long groupId = 2L;

		Access access1 = Access.builder()
				.groupID(groupId)
				.accountID(userId)
				.role(2)
				.courseEditionID(editionId)
				.build();
		Access access2 = Access.builder()
				.groupID(groupId)
				.accountID("otherUser")
				.role(1)
				.courseEditionID(editionId)
				.build();
		Access access3 = Access.builder()
				.groupID(3L)
				.accountID(userId)
				.role(3)
				.courseEditionID(editionId)
				.build();

		List<Access> allAccesses = new ArrayList<>();
		allAccesses.add(access1);
		allAccesses.add(access2);
		allAccesses.add(access3);

		when(mockRepository.findAll()).thenReturn(allAccesses);

		Access result = accessService.getAccessFromUserEditionGroup(userId, editionId, groupId);

		assertEquals(access1, result);
	}

	@Test
	public void testGetAccessFromUserEdition_Null() {
		AccessService accessService = new AccessService();
		accessService.setRepository(mockRepository);
		String userId = "testUser";
		long editionId = 1L;

		List<Access> allAccesses = new ArrayList<>();

		when(mockRepository.findAll()).thenReturn(allAccesses);

		Access result = accessService.getAccessFromUserEdition(userId, editionId);

		assertNull(result);
	}

	@Test
	public void testGetAccessFromUserEditionGroup_Null() {
		AccessService accessService = new AccessService();
		accessService.setRepository(mockRepository);
		String userId = "testUser";
		long editionId = 1L;
		long groupId = 2L;

		List<Access> allAccesses = new ArrayList<>();

		when(mockRepository.findAll()).thenReturn(allAccesses);

		Access result = accessService.getAccessFromUserEditionGroup(userId, editionId, groupId);

		assertNull(result);
	}

}
