package server.service;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.*;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import javax.ws.rs.core.Application;

import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInstance;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.springframework.boot.test.context.SpringBootTest;

import server.entity.Account;
import server.repository.AccountRepository;

@SpringBootTest(classes = Application.class)
@TestInstance(TestInstance.Lifecycle.PER_CLASS)
class AccountServiceTest {

	@Mock
	private AccountRepository mockRepository;

	@Mock
	private InstanceMappingService instanceMappingService;
	@InjectMocks
	private AccountService accountService;

	@Test
	public void testRemoveAPIKey() {
		String netID = "testNetID";
		Account account = new Account();
		account.setNetId(netID);
		account.setApiKey("apiKey");

		when(mockRepository.findById(netID)).thenReturn(Optional.of(account));

		accountService.removeAPIKey(netID);

		assertEquals(null, account.getApiKey());
		verify(mockRepository, times(1)).save(account);
		verify(instanceMappingService, times(1)).removeInstance(netID);
	}

	@Test
	public void testAddAPIKey() {

		String netID = "testNetID";
		String apiKey = "testAPIKey";
		Account account = new Account();
		account.setNetId(netID);

		when(mockRepository.findById(netID)).thenReturn(Optional.of(account));

		accountService.addAPIKey(netID, apiKey);

		assertEquals(apiKey, account.getApiKey());
		verify(mockRepository, times(1)).save(account);
		verify(instanceMappingService, times(1)).addInstance(netID, apiKey);
	}

	@Test
	public void testSaveAccountNewAcc() {
		Account account1 = Account.builder().userId("test-user-1").apiKey("test-api-key-1")
				.email("test1@example.com").build();

		when(mockRepository.save(account1)).thenReturn(account1);

		Account savedAccount = accountService.saveAccount(account1);

		assertEquals(savedAccount.getTimeCreated(), savedAccount.getLastModified());
		assertEquals(account1, savedAccount);
	}

	@Test
	public void testSaveAccountUpdate() {
		Account account1 = Account.builder().userId("test-user-1").apiKey("test-api-key-1")
				.email("test1@example.com").build();
		Account account2 = Account.builder().userId("test-user-1").apiKey("test-api-key-2")
				.email("test1@example.com").timeCreated(LocalDateTime.now().minusDays(1)).build();

		when(mockRepository.save(account1)).thenReturn(account1);
		when(mockRepository.findById(any())).thenReturn(Optional.ofNullable(account2));

		Account savedAccount = accountService.saveAccount(account1);

		verify(mockRepository).save(account1);

		assertTrue(savedAccount.getLastModified().isAfter(savedAccount.getTimeCreated()));

		assertEquals(account1, savedAccount);
	}

	@Test
	public void testDeleteAccount() {
		doNothing().when(mockRepository).deleteById("TestId");

		accountService.deleteAccount("TestId");
		verify(mockRepository, times(1)).deleteById("TestId");
	}

	@Test
	public void testFetchAllAccounts() {
		List<Account> accounts = new ArrayList<>();

		Account account1 = Account.builder().userId("test-user-1").apiKey("test-api-key-1")
				.email("test1@example.com").build();
		Account account2 = Account.builder().userId("test-user-2").apiKey("test-api-key-2")
				.email("test2@example.com").build();

		accounts.add(account1);
		accounts.add(account2);

		when(mockRepository.findAll()).thenReturn(accounts);

		List<Account> fetchedAccounts = accountService.fetchAllAccounts();
		assertEquals(accounts, fetchedAccounts);
	}

	@Test
	void testGetAPIKey_existingAccount() {
		String netId = "test-user-1";
		String apiKey = "test-api-key";

		Account account = Account.builder()
				.netId(netId)
				.apiKey(apiKey)
				.email("test@example.com")
				.build();

		when(mockRepository.findById(netId)).thenReturn(Optional.of(account));

		String retrievedApiKey = accountService.getAPIKey(netId);

		assertEquals(apiKey, retrievedApiKey);
	}

	@Test
	void testGetAPIKey_nonExistingAccount() {
		String netId = "test-user-1";

		when(mockRepository.findById(netId)).thenReturn(Optional.empty());

		String retrievedApiKey = accountService.getAPIKey(netId);

		assertNull(retrievedApiKey);
	}

	@Test
	void testGetAccountFromMail_existingAccount() throws Exception {
		String email = "test@example.com";
		Account account = Account.builder()
				.netId("test-user-1")
				.apiKey("test-api-key")
				.email(email)
				.build();

		List<Account> accounts = List.of(account);

		when(mockRepository.findAll()).thenReturn(accounts);

		Account retrievedAccount = accountService.getAccountFromMail(email);

		assertEquals(account, retrievedAccount);
	}

	@Test
	void testGetAccountFromMail_nonExistingAccount() {
		String email = "test@example.com";

		when(mockRepository.findAll()).thenReturn(new ArrayList<>());

		Exception exception = assertThrows(Exception.class, () -> {
			accountService.getAccountFromMail(email);
		});

		assertEquals("User not found in DB", exception.getMessage());
	}

	@Test
	void testGetAccountFromNetId_existingAccount() throws Exception {
		String netId = "test-user-1";
		Account account = Account.builder()
				.netId(netId)
				.apiKey("test-api-key")
				.email("test@example.com")
				.build();

		when(mockRepository.findById(netId)).thenReturn(Optional.of(account));

		Account retrievedAccount = accountService.getAccountFromNetId(netId);

		assertEquals(account, retrievedAccount);
	}

	@Test
	void testGetAccountFromNetId_nonExistingAccount() {
		String netId = "test-user-1";

		when(mockRepository.findById(netId)).thenReturn(Optional.empty());

		Exception exception = assertThrows(Exception.class, () -> {
			accountService.getAccountFromNetId(netId);
		});

		assertEquals("The user with this netId does not exist", exception.getMessage());
	}

}
