package server.service;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.*;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import javax.ws.rs.core.Application;

import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInstance;
import org.mockito.Mock;
import org.springframework.boot.test.context.SpringBootTest;

import server.entity.CourseEdition;
import server.repository.CourseRepository;

@SpringBootTest(classes = Application.class)
@TestInstance(TestInstance.Lifecycle.PER_CLASS)
class CourseEditionServiceTest {

	@Mock
	private CourseRepository courseRepository;

	@Test
	void saveCourseNewCourse() {
		CourseEditionService courseService = new CourseEditionService();
		courseService.setRepository(courseRepository);

		CourseEdition courseEdition = CourseEdition.builder().courseEditionID(1L).courseName("OOPP")
				.edition("22-23")
				.build();
		when(courseRepository.save(courseEdition)).thenReturn(courseEdition);

		CourseEdition savedCourseEdition = courseService.saveCourseEdition(courseEdition);

		verify(courseRepository).save(courseEdition);

		assertEquals(courseEdition.getLastModified(), courseEdition.getTimeCreated());

		assertEquals(courseEdition, savedCourseEdition);
	}

	@Test
	void saveCourseUpdate() {
		CourseEditionService courseService = new CourseEditionService();
		courseService.setRepository(courseRepository);

		CourseEdition courseEdition = CourseEdition.builder().courseEditionID(1L).courseName("OOPP")
				.edition("22-23")
				.build();
		CourseEdition courseEditionOld = CourseEdition.builder().courseEditionID(1L).courseName("OOPP")
				.edition("23-24")
				.timeCreated(LocalDateTime.now().minusDays(1)).build();
		when(courseRepository.save(courseEdition)).thenReturn(courseEdition);

		when(courseRepository.findById(any())).thenReturn(Optional.ofNullable(courseEditionOld));

		CourseEdition savedCourseEdition = courseService.saveCourseEdition(courseEdition);

		verify(courseRepository).save(courseEdition);

		assertTrue(savedCourseEdition.getLastModified().isAfter(savedCourseEdition.getTimeCreated()));

		assertEquals(courseEdition, savedCourseEdition);
	}

	@Test
	void deleteCourse() {
		CourseEditionService courseService = new CourseEditionService();
		courseService.setRepository(courseRepository);

		long courseId = 3L;
		doNothing().when(courseRepository).deleteById(courseId);

		courseService.deleteCourseEdition(courseId);
		verify(courseRepository, times(1)).deleteById(courseId);
	}

	@Test
	void fetchAllCourses() {
		CourseEditionService courseService = new CourseEditionService();
		courseService.setRepository(courseRepository);

		CourseEdition courseEdition1 = CourseEdition.builder().courseEditionID(1L).courseName("OOPP")
				.edition("22-23").build();
		CourseEdition courseEdition2 = CourseEdition.builder().courseEditionID(1L).courseName("OOPP")
				.edition("21-22").build();
		CourseEdition courseEdition3 = CourseEdition.builder().courseEditionID(1L).courseName("SEM")
				.edition("22-23")
				.build();

		List<CourseEdition> courseEditionList = new ArrayList<>();
		courseEditionList.add(courseEdition1);
		courseEditionList.add(courseEdition2);
		courseEditionList.add(courseEdition3);

		when(courseRepository.findAll()).thenReturn(courseEditionList);
		List<CourseEdition> fetchedCours = courseService.fetchAllCourseEditions();

		assertEquals(fetchedCours, courseEditionList);
	}

	@Test
	void fetchAllCourseEditionsOfCourse() {
		CourseEditionService courseService = new CourseEditionService();
		courseService.setRepository(courseRepository);

		CourseEdition courseEdition1 = CourseEdition.builder().courseEditionID(1L).courseName("OOPP")
				.edition("22-23").build();
		CourseEdition courseEdition2 = CourseEdition.builder().courseEditionID(1L).courseName("OOPP")
				.edition("21-22").build();
		CourseEdition courseEdition3 = CourseEdition.builder().courseEditionID(1L).courseName("SEM")
				.edition("22-23")
				.build();

		List<CourseEdition> courseEditionList = new ArrayList<>();
		List<CourseEdition> goodCourseListEdition = new ArrayList<>();
		courseEditionList.add(courseEdition1);
		courseEditionList.add(courseEdition2);
		courseEditionList.add(courseEdition3);

		goodCourseListEdition.add(courseEdition1);
		goodCourseListEdition.add(courseEdition2);

		when(courseRepository.findAll()).thenReturn(courseEditionList);
		List<CourseEdition> fetchedGroups = courseService.fetchAllCourseEditionsOfCourse("OOPP");
		assertEquals(fetchedGroups, goodCourseListEdition);
	}

	@Test
	void getCourseEditionFromCoursePath() {
		CourseEditionService courseService = new CourseEditionService();
		courseService.setRepository(courseRepository);

		List<CourseEdition> courseEditionList = new ArrayList<>();

		CourseEdition courseEdition1 = CourseEdition.builder()
				.courseEditionID(1L)
				.courseName("OOPP")
				.edition("22-23")
				.coursePath("path1")
				.build();

		CourseEdition courseEdition2 = CourseEdition.builder()
				.courseEditionID(2L)
				.courseName("OOPP")
				.edition("21-22")
				.coursePath("path2")
				.build();

		courseEditionList.add(courseEdition1);
		courseEditionList.add(courseEdition2);

		when(courseRepository.findAll()).thenReturn(courseEditionList);

		CourseEdition fetchedCourseEdition = courseService.getCourseEditionFromCoursePath("path1");

		assertEquals(courseEdition1, fetchedCourseEdition);
	}

	@Test
	void fetchAllCoursesOfListOfIds() {
		CourseEditionService courseService = new CourseEditionService();
		courseService.setRepository(courseRepository);

		List<Long> courseIds = List.of(1L, 2L, 3L);

		CourseEdition courseEdition1 = CourseEdition.builder()
				.courseEditionID(1L)
				.courseName("OOPP")
				.edition("22-23")
				.build();

		CourseEdition courseEdition2 = CourseEdition.builder()
				.courseEditionID(2L)
				.courseName("OOPP")
				.edition("21-22")
				.build();

		CourseEdition courseEdition3 = CourseEdition.builder()
				.courseEditionID(3L)
				.courseName("SEM")
				.edition("22-23")
				.build();

		List<CourseEdition> courseEditionList = new ArrayList<>();
		courseEditionList.add(courseEdition1);
		courseEditionList.add(courseEdition2);
		courseEditionList.add(courseEdition3);

		when(courseRepository.findAllById(courseIds)).thenReturn(courseEditionList);

		List<CourseEdition> fetchedCourses = courseService.fetchAllCoursesOfListOfIds(courseIds);

		assertEquals(courseEditionList, fetchedCourses);
	}

	@Test
	void getCourseEditionFromCoursePath_CourseEditionDoesNotExist() {
		CourseEditionService courseService = new CourseEditionService();
		courseService.setRepository(courseRepository);

		List<CourseEdition> courseEditionList = new ArrayList<>();

		when(courseRepository.findAll()).thenReturn(courseEditionList);

		CourseEdition fetchedCourseEdition = courseService.getCourseEditionFromCoursePath("path1");

		assertNull(fetchedCourseEdition);
	}

	@Test
	void deleteNonExistingCourseEdition() {
		CourseEditionService courseService = new CourseEditionService();
		courseService.setRepository(courseRepository);

		String courseName = "NonExistentCourse";
		String edition = "NonExistentEdition";

		when(courseRepository.findAll()).thenReturn(Collections.emptyList());

		CourseEdition deletedCourseEdition = courseService.deleteCourseEdition(courseName, edition);

		verify(courseRepository, never()).delete(any());
		assertNull(deletedCourseEdition);
	}

}
