package server.service;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.*;

import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import org.gitlab4j.api.GitLabApi;
import org.gitlab4j.api.GitLabApiException;
import org.gitlab4j.api.GroupApi;
import org.gitlab4j.api.models.Group;
import org.gitlab4j.api.models.GroupParams;
import org.gitlab4j.api.models.Member;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import server.model.MemberRequestDTO;

public class GroupAPIServiceTest {

	@Mock
	private GitLabApi gitLabApi;

	@Mock
	private GroupApi groupApi;

	private GroupAPIService groupAPIService;

	@BeforeEach
	void setUp() {
		MockitoAnnotations.openMocks(this);
		groupAPIService = new GroupAPIService();
		when(gitLabApi.getGroupApi()).thenReturn(groupApi);
	}

	@Test
	public void testCreateGroup() throws GitLabApiException {
		Group group = new Group();
		when(groupApi.createGroup(any(GroupParams.class))).thenReturn(group);

		GroupAPIService service = new GroupAPIService();

		Group createdGroup = service.createGroup(gitLabApi, "Test Group", "Test description", "test-group",
				true, "public");

		verify(gitLabApi, times(1)).getGroupApi();
		verify(groupApi, times(1)).createGroup(any(GroupParams.class));
		assertEquals(group, createdGroup);
	}

	@Test
	public void testCreateSubgroup() throws GitLabApiException {
		Group group = new Group();
		when(groupApi.createGroup(any(GroupParams.class))).thenReturn(group);

		Group createdGroup = groupAPIService.createSubgroup(gitLabApi, "Test Subgroup", "Test description",
				"test-subgroup", true, "public", 123L);

		verify(gitLabApi, times(1)).getGroupApi();
		verify(groupApi, times(1)).createGroup(any(GroupParams.class));
		assertEquals(group, createdGroup);
	}

	@Test
	public void testGetGroups() throws GitLabApiException {
		Group group1 = new Group();
		Group group2 = new Group();
		List<Group> expectedGroups = Arrays.asList(group1, group2);
		when(groupApi.getGroups()).thenReturn(expectedGroups);

		List<Group> actualGroups = groupAPIService.getGroups(gitLabApi);

		// Verify the method calls and assertions
		verify(gitLabApi, times(1)).getGroupApi();
		verify(groupApi, times(1)).getGroups();
		assertEquals(expectedGroups, actualGroups);
	}

	@Test
	public void testAddGroupMember() throws GitLabApiException {
		Member expectedMember = new Member();
		when(groupApi.addMember(any(), anyLong(), anyInt(), any(Date.class))).thenReturn(expectedMember);

		Member actualMember = groupAPIService.addGroupMember(gitLabApi, "groupIdOrPath", 123L, 50,
				new Date());

		verify(gitLabApi, times(1)).getGroupApi();
		verify(groupApi, times(1)).addMember(any(), anyLong(), anyInt(), any(Date.class));
		assertEquals(expectedMember, actualMember);
	}

	@Test
	public void testUpdateGroupMember() throws GitLabApiException {
		Member expectedMember = new Member();
		when(groupApi.updateMember(any(), anyLong(), anyInt(), any(Date.class))).thenReturn(expectedMember);

		Member actualMember = groupAPIService.updateGroupMember(gitLabApi, "groupIdOrPath", 123L, 50,
				new Date());

		verify(gitLabApi, times(1)).getGroupApi();
		verify(groupApi, times(1)).updateMember(any(), anyLong(), anyInt(), any(Date.class));
		assertEquals(expectedMember, actualMember);
	}

	@Test
	public void testRemoveGroupMember() throws GitLabApiException {
		groupAPIService.removeGroupMember(gitLabApi, "groupIdOrPath", 123L);

		verify(gitLabApi, times(1)).getGroupApi();
		verify(groupApi, times(1)).removeMember("groupIdOrPath", 123L);
	}

	@Test
	public void testGetMembers() throws GitLabApiException {
		List<Member> expectedMembers = Arrays.asList(new Member(), new Member());
		when(groupApi.getMembers("groupIdOrPath")).thenReturn(expectedMembers);

		List<Member> actualMembers = groupAPIService.getMembers(gitLabApi, "groupIdOrPath");

		verify(gitLabApi, times(1)).getGroupApi();
		verify(groupApi, times(1)).getMembers("groupIdOrPath");
		assertEquals(expectedMembers, actualMembers);
	}

	@Test
	public void testDeleteGroup() throws GitLabApiException {
		groupAPIService.deleteGroup(gitLabApi, "groupIdOrPath");

		verify(gitLabApi, times(1)).getGroupApi();
		verify(groupApi, times(1)).deleteGroup("groupIdOrPath");
	}

	@Test
	public void testUpdateGroup() throws Exception {
		GroupParams params = new GroupParams()
				.withName("Updated Group")
				.withDescription("Updated description");

		Group updatedGroup = new Group();
		updatedGroup.setId(1L);
		updatedGroup.setName("Updated Group");
		updatedGroup.setDescription("Updated description");

		when(gitLabApi.getGroupApi()).thenReturn(groupApi);
		when(groupApi.updateGroup(any(), eq(params))).thenReturn(updatedGroup);

		Group result = groupAPIService.updateGroup(gitLabApi, "group-path", params);

		assertEquals(updatedGroup.getId(), result.getId());
		assertEquals(updatedGroup.getName(), result.getName());
		assertEquals(updatedGroup.getDescription(), result.getDescription());
	}

	@Test
	void createGroupUpdated_ShouldCreateGroupAndReturnResult() throws Exception {
		String groupName = "TestGroup";
		String description = "Test group description";
		String path = "test-group";
		GroupParams.DefaultBranchProtectionLevel defaultBranchProtectionLevel = GroupParams.DefaultBranchProtectionLevel.NOT_PROTECTED;
		String visibility = "public";
		Group createdGroup = new Group();
		when(groupApi.createGroup(any())).thenReturn(createdGroup);

		Group result = groupAPIService.createGroupUpdated(gitLabApi, groupName, description, path,
				defaultBranchProtectionLevel, visibility);

		assertEquals(createdGroup, result);
		verify(groupApi).createGroup(any(GroupParams.class));
	}

	@Test
	void createSubgroupUpdated_ShouldCreateSubgroupAndReturnResult() throws Exception {
		String groupName = "TestSubgroup";
		String description = "Test subgroup description";
		String path = "test-subgroup";
		String visibility = "public";
		Long parentId = 123L;
		GroupParams.DefaultBranchProtectionLevel defaultBranchProtectionLevel = GroupParams.DefaultBranchProtectionLevel.NOT_PROTECTED;
		Group createdGroup = new Group();
		when(groupApi.createGroup(any())).thenReturn(createdGroup);

		Group result = groupAPIService.createSubgroupUpdated(gitLabApi, groupName, description, path,
				visibility, parentId, defaultBranchProtectionLevel);

		assertEquals(createdGroup, result);
		verify(groupApi).createGroup(any(GroupParams.class));
	}

	@Test
	void updateGroupMember_ShouldUpdateGroupMemberAndReturnResult() throws Exception {
		MemberRequestDTO dto = new MemberRequestDTO();
		dto.setIdOrPath("groupIdOrPath");
		dto.setUserId(123L);
		dto.setAccessLevel(30);
		dto.setExpiresAt(new Date());
		Member updatedMember = new Member();
		when(groupApi.updateMember(anyString(), anyLong(), anyInt(), any())).thenReturn(updatedMember);

		Member result = groupAPIService.updateGroupMember(gitLabApi, dto);

		assertEquals(updatedMember, result);
		verify(groupApi).updateMember(eq(dto.getIdOrPath()), eq(dto.getUserId()), eq(dto.getAccessLevel()),
				eq(dto.getExpiresAt()));
	}

	@Test
	void removeGroupMember_ShouldRemoveGroupMember() throws Exception {
		MemberRequestDTO dto = new MemberRequestDTO();
		dto.setIdOrPath("groupIdOrPath");
		dto.setUserId(123L);

		groupAPIService.removeGroupMember(gitLabApi, dto);

		verify(groupApi).removeMember(eq(dto.getIdOrPath()), eq(dto.getUserId()));
	}

	@Test
	void addGroupMember_ShouldAddGroupMemberAndReturnResult() throws Exception {
		Object groupIdOrPath = "groupIdOrPath";
		Long userId = 123L;
		Integer accessLevel = 30;
		Date expiresAt = new Date();
		Member addedMember = new Member();
		when(groupApi.addMember(any(), anyLong(), anyInt(), any())).thenReturn(addedMember);

		Member result = groupAPIService.addGroupMember(gitLabApi, groupIdOrPath, userId, accessLevel,
				expiresAt);

		assertEquals(addedMember, result);
		verify(groupApi).addMember(eq(groupIdOrPath), eq(userId), eq(accessLevel), eq(expiresAt));
	}

	@Test
	void updateGroup_ShouldUpdateGroupAndReturnResult() throws Exception {
		Object groupIdOrPath = "groupIdOrPath";
		GroupParams params = new GroupParams();
		Group updatedGroup = new Group();
		when(groupApi.updateGroup(any(), any())).thenReturn(updatedGroup);

		Group result = groupAPIService.updateGroup(gitLabApi, groupIdOrPath, params);

		assertEquals(updatedGroup, result);
		verify(groupApi).updateGroup(eq(groupIdOrPath), eq(params));
	}

	@Test
	void getGroupId_ShouldReturnGroupId() throws Exception {
		Object groupIdOrPath = "groupIdOrPath";
		Group group = new Group();
		group.setId(123L);
		when(groupApi.getGroup(any())).thenReturn(group);

		Long result = groupAPIService.getGroupId(gitLabApi, groupIdOrPath);

		assertEquals(group.getId(), result);
		verify(groupApi).getGroup(eq(groupIdOrPath));
	}

	@Test
	void getSubgroups_ShouldReturnListOfSubgroups() throws Exception {
		Object groupIdOrPath = "groupIdOrPath";
		List<Group> subgroups = List.of(new Group());
		when(groupApi.getSubGroups(any())).thenReturn(subgroups);

		List<Group> result = groupAPIService.getSubgroups(gitLabApi, groupIdOrPath);

		assertEquals(subgroups, result);
		verify(groupApi).getSubGroups(eq(groupIdOrPath));
	}

	@Test
	void getGroup_ShouldReturnGroup() throws Exception {
		Object groupIdOrPath = "groupIdOrPath";
		Group group = new Group();
		when(groupApi.getGroup(any())).thenReturn(group);

		Group result = groupAPIService.getGroup(gitLabApi, groupIdOrPath);

		assertEquals(group, result);
		verify(groupApi).getGroup(eq(groupIdOrPath));
	}

	@Test
	void getSubgroupPaths_ShouldReturnListOfSubgroupPaths() throws Exception {
		Object groupIdOrPath = "groupIdOrPath";
		List<Group> subgroups = Collections.singletonList(new Group());
		when(groupApi.getSubGroups(any())).thenReturn(subgroups);

		List<String> result = groupAPIService.getSubgroupPaths(gitLabApi, groupIdOrPath);

		assertEquals(subgroups.stream().map(Group::getPath).toList(), result);
		verify(groupApi).getSubGroups(eq(groupIdOrPath));
	}

}
