package server.service;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.*;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import javax.ws.rs.core.Application;

import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInstance;
import org.mockito.Mock;
import org.springframework.boot.test.context.SpringBootTest;

import server.entity.StudentGroup;
import server.repository.GroupRepository;

@SpringBootTest(classes = Application.class)
@TestInstance(TestInstance.Lifecycle.PER_CLASS)
class GroupServiceTest {

	@Mock
	private GroupRepository repo;

	@Test
	void saveGroupNewGroup() {
		GroupService service = new GroupService();
		service.setRepository(repo);

		StudentGroup group = StudentGroup.builder().courseID(1L).name("13A").build();
		when(repo.save(group)).thenReturn(group);

		StudentGroup savedGroup = service.saveGroup(group);

		verify(repo).save(group);

		assertEquals(group.getLastModified(), group.getTimeCreated());

		assertEquals(savedGroup, group);
	}

	@Test
	void saveGroupUpdate() {
		GroupService service = new GroupService();
		service.setRepository(repo);

		StudentGroup group = StudentGroup.builder().courseID(1L).name("13A").build();
		StudentGroup groupOld = StudentGroup.builder().courseID(1L).name("13B")
				.timeCreated(LocalDateTime.now().minusDays(1)).build();
		when(repo.save(group)).thenReturn(group);
		when(repo.findById(any())).thenReturn(Optional.ofNullable(groupOld));

		StudentGroup savedGroup = service.saveGroup(group);

		verify(repo).save(group);

		assertTrue(savedGroup.getLastModified().isAfter(savedGroup.getTimeCreated()));

		assertEquals(savedGroup, group);
	}

	@Test
	void deleteGroup() {
		GroupService service = new GroupService();
		service.setRepository(repo);

		long groupId = 4L;
		doNothing().when(repo).deleteById(groupId);
		service.deleteGroup(groupId);
		verify(repo, times(1)).deleteById(groupId);
	}

	@Test
	void fetchAllGroups() {
		GroupService service = new GroupService();
		service.setRepository(repo);

		List<StudentGroup> groups = new ArrayList<>();

		StudentGroup group1 = StudentGroup.builder().courseID(1L).name("13A").build();
		StudentGroup group2 = StudentGroup.builder().courseID(1L).name("13B").build();
		StudentGroup group3 = StudentGroup.builder().courseID(1L).name("14A").build();
		StudentGroup group4 = StudentGroup.builder().courseID(2L).name("AREAS").build();

		groups.add(group1);
		groups.add(group2);
		groups.add(group3);
		groups.add(group4);

		when(repo.findAll()).thenReturn(groups);
		List<StudentGroup> fetchedGroups = service.fetchAllGroups();
		assertEquals(fetchedGroups, groups);
	}

	@Test
	void fetchAllGroupsFromCourse() {
		GroupService service = new GroupService();
		service.setRepository(repo);

		List<StudentGroup> groups = new ArrayList<>();
		List<StudentGroup> goodGroups = new ArrayList<>();

		StudentGroup group1 = StudentGroup.builder().courseID(1L).name("13A").build();
		StudentGroup group2 = StudentGroup.builder().courseID(1L).name("13B").build();
		StudentGroup group3 = StudentGroup.builder().courseID(1L).name("14A").build();
		StudentGroup group4 = StudentGroup.builder().courseID(2L).name("AREAS").build();

		groups.add(group1);
		groups.add(group2);
		groups.add(group3);
		groups.add(group4);

		goodGroups.add(group1);
		goodGroups.add(group2);
		goodGroups.add(group3);

		when(repo.findAll()).thenReturn(groups);
		List<StudentGroup> fetchedGroups = service.fetchAllGroupsFromCourse(1L);
		assertEquals(goodGroups, fetchedGroups);
	}

	@Test
	void getGroupFromGroupPath() {
		GroupService service = new GroupService();
		service.setRepository(repo);

		StudentGroup group1 = StudentGroup.builder().courseID(1L).name("13A").groupPath("group1").build();
		StudentGroup group2 = StudentGroup.builder().courseID(2L).name("13B").groupPath("group2").build();
		StudentGroup group3 = StudentGroup.builder().courseID(3L).name("14A").groupPath("group3").build();

		List<StudentGroup> allGroups = new ArrayList<>();
		allGroups.add(group1);
		allGroups.add(group2);
		allGroups.add(group3);

		when(repo.findAll()).thenReturn(allGroups);

		StudentGroup foundGroup = service.getGroupFromGroupPath("group2");

		assertEquals(group2, foundGroup);
	}

	@Test
	void fetchAllGroupsFromId() {
		GroupService service = new GroupService();
		service.setRepository(repo);

		List<Long> groupIds = List.of(1L, 3L, 5L);

		StudentGroup group1 = StudentGroup.builder().groupID(1L).courseID(1L).name("13A").build();
		StudentGroup group3 = StudentGroup.builder().groupID(3L).courseID(2L).name("13B").build();
		StudentGroup group5 = StudentGroup.builder().groupID(5L).courseID(3L).name("14A").build();

		List<StudentGroup> groups = new ArrayList<>();
		groups.add(group1);
		groups.add(group3);
		groups.add(group5);

		when(repo.findAllById(groupIds)).thenReturn(groups);

		List<StudentGroup> fetchedGroups = service.fetchAllGroupsFromId(groupIds);

		assertEquals(groups, fetchedGroups);
	}

}
