package server.service;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.*;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.regex.Pattern;

import org.gitlab4j.api.CommitsApi;
import org.gitlab4j.api.GitLabApi;
import org.gitlab4j.api.GitLabApiException;
import org.gitlab4j.api.ProjectApi;
import org.gitlab4j.api.models.*;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mock;

import server.model.MemberRequestDTO;
import server.model.ProjectRequestDTO;

public class ProjectAPIServiceTest {
	@Mock
	private GitLabApi gitLabApi;

	@Mock
	private ProjectApi projectApi;

	@Mock
	private CommitsApi commitsApi;

	private ProjectAPIService projectAPIService;

	private MemberRequestDTO sampleMemberRequest;
	private ProjectRequestDTO sampleProjectRequest;

	@BeforeEach
	public void setup() {
		gitLabApi = mock(GitLabApi.class);
		projectApi = mock(ProjectApi.class);
		commitsApi = mock(CommitsApi.class);
		projectAPIService = new ProjectAPIService();
		when(gitLabApi.getProjectApi()).thenReturn(projectApi);
		when(gitLabApi.getCommitsApi()).thenReturn(commitsApi);
		new MemberRequestDTO();
		sampleMemberRequest = MemberRequestDTO.builder().userId(123L).idOrPath("test-path").accessLevel(30)
				.expiresAt(new Date()).build();
		new ProjectRequestDTO();
		sampleProjectRequest = ProjectRequestDTO.builder().projectIdOrPath("test-path").name("test")
				.description("test-description").issuesEnabled(true).isPublic(false).snippetsEnabled(true)
				.mergeRequestsEnabled(true).namespaceId(0L).wikiEnabled(true).namespaceId(5L).build();
	}

	@Test
	public void testCreateProjectInNamespace() throws GitLabApiException {
		Project project = new Project()
				.withName("test")
				.withDescription("test")
				.withIssuesEnabled(true)
				.withMergeRequestsEnabled(true)
				.withWikiEnabled(true)
				.withSnippetsEnabled(true)
				.withPublic(true);

		when(projectApi.createProject(anyLong(), any(Project.class)))
				.thenReturn(project);

		Project result = projectAPIService.createProjectInNamespace(gitLabApi, sampleProjectRequest);

		assertEquals(project, result);
		verify(gitLabApi.getProjectApi(), times(1)).createProject(anyLong(), any(Project.class));
	}

	@Test
	public void testGetProjects() throws GitLabApiException {
		List<Project> projects = List.of(new Project(), new Project());

		when(projectApi.getProjects())
				.thenReturn(projects);

		List<Project> result = projectAPIService.getProjects(gitLabApi);

		assertEquals(projects, result);
		verify(gitLabApi.getProjectApi(), times(1)).getProjects();
	}

	@Test
	public void testAddProjectMember() throws GitLabApiException {
		Member member = new Member()
				.withAccessLevel(AccessLevel.forValue(5))
				.withExpiresAt(new Date())
				.withId(123L);

		when(projectApi.addMember(any(), anyLong(), anyInt(), any(Date.class)))
				.thenReturn(member);

		Member result = projectAPIService.addProjectMember(gitLabApi, sampleMemberRequest);

		assertEquals(member, result);
		verify(gitLabApi.getProjectApi(), times(1)).addMember(any(String.class), anyLong(),
				any(Integer.class), any(Date.class));
	}

	@Test
	public void testUpdateProjectMember() throws GitLabApiException {
		Member member1 = new Member()
				.withAccessLevel(AccessLevel.forValue(5))
				.withExpiresAt(new Date())
				.withId(123L);

		when(projectApi.addMember(any(), anyLong(), anyInt(), any(Date.class)))
				.thenReturn(member1);

		Member result = projectAPIService.addProjectMember(gitLabApi, sampleMemberRequest);

		assertEquals(member1, result);
		verify(gitLabApi.getProjectApi(), times(1)).addMember(any(String.class), anyLong(), anyInt(),
				any(Date.class));

		Member member2 = new Member()
				.withAccessLevel(AccessLevel.forValue(50))
				.withExpiresAt(new Date())
				.withId(123L);

		when(projectApi.updateMember(any(), anyLong(), anyInt(), any(Date.class)))
				.thenReturn(member2);

		Member result2 = projectAPIService.updateProjectMember(gitLabApi, sampleMemberRequest);
		assertEquals(member2, result2);
		verify(gitLabApi.getProjectApi(), times(1)).updateMember(any(), anyLong(), anyInt(), any(Date.class));
	}

	@Test
	public void testRemoveProjectMember() throws GitLabApiException {
		Member member1 = new Member()
				.withAccessLevel(AccessLevel.forValue(5))
				.withExpiresAt(new Date())
				.withId(123L);

		when(projectApi.addMember(any(), anyLong(), anyInt(), any(Date.class)))
				.thenReturn(member1);

		Member result1 = projectAPIService.addProjectMember(gitLabApi, sampleMemberRequest);
		assertEquals(member1, result1);
		verify(gitLabApi.getProjectApi(), times(1)).addMember(any(String.class), anyLong(), anyInt(),
				any(Date.class));

		doNothing().when(projectApi).removeMember(any(), anyLong());
		projectAPIService.removeProjectMember(gitLabApi, sampleMemberRequest);

		when(projectApi.getMembers("test"))
				.thenReturn(new ArrayList<>());

		List<Member> result2 = projectAPIService.getMembers(gitLabApi, "test");
		assertTrue(result2.isEmpty());
		verify(gitLabApi.getProjectApi(), times(1)).removeMember(any(), anyLong());
	}

	@Test
	public void testForkProject() throws GitLabApiException {
		Project project = new Project()
				.withName("test")
				.withDescription("test")
				.withIssuesEnabled(true)
				.withMergeRequestsEnabled(true)
				.withWikiEnabled(true)
				.withSnippetsEnabled(true)
				.withPublic(true);

		when(projectApi.forkProject(any(), anyLong()))
				.thenReturn(project);

		Project result = projectAPIService.forkProject(gitLabApi, project, 10L);
		assertEquals(project, result);
		verify(gitLabApi.getProjectApi(), times(1)).forkProject(any(), anyLong());
	}

	@Test
	public void testUpdateProject() throws GitLabApiException {
		Project project = new Project()
				.withName("test")
				.withDescription("test")
				.withIssuesEnabled(true)
				.withMergeRequestsEnabled(true)
				.withWikiEnabled(true)
				.withSnippetsEnabled(true)
				.withPublic(true);

		when(projectApi.updateProject(any(Project.class)))
				.thenReturn(project);
		Project result = projectAPIService.updateProject(gitLabApi, project);

		assertEquals(project, result);
		verify(gitLabApi.getProjectApi(), times(1)).updateProject(any(Project.class));
	}

	@Test
	public void getMembers() throws GitLabApiException {
		Project project = new Project()
				.withName("test")
				.withDescription("test")
				.withIssuesEnabled(true)
				.withMergeRequestsEnabled(true)
				.withWikiEnabled(true)
				.withSnippetsEnabled(true)
				.withPublic(true);

		when(projectApi.createProject(any(Project.class)))
				.thenReturn(project);

		projectApi.createProject(project);

		List<Member> members = List.of(new Member());
		when(projectApi.getMembers(project)).thenReturn(members);

		List<Member> result = projectAPIService.getMembers(gitLabApi, project);
		assertEquals(members, result);
		verify(gitLabApi.getProjectApi(), times(1)).getMembers(any(Project.class));
	}

	@Test
	public void testDeleteProject() throws GitLabApiException {
		Project project = new Project()
				.withName("test")
				.withDescription("test")
				.withIssuesEnabled(true)
				.withMergeRequestsEnabled(true)
				.withWikiEnabled(true)
				.withSnippetsEnabled(true)
				.withPublic(true);

		doNothing().when(projectApi).deleteProject(any(Project.class));

		projectAPIService.deleteProject(gitLabApi, project);

		verify(gitLabApi.getProjectApi(), times(1)).deleteProject(any(Project.class));
	}

	@Test
	public void createInitialCommit() throws GitLabApiException {
		Commit commit = new Commit();
		when(commitsApi.createCommit(anyLong(), any(CommitPayload.class))).thenReturn(commit);

		projectAPIService.createInitialCommit(gitLabApi, 10L, "Hah");
		verify(gitLabApi.getCommitsApi(), times(1)).createCommit(any(), any(CommitPayload.class));
	}

	@Test
	public void regexTest() throws GitLabApiException {
		Project project = new Project()
				.withName("test")
				.withDescription("test")
				.withIssuesEnabled(true)
				.withMergeRequestsEnabled(true)
				.withWikiEnabled(true)
				.withSnippetsEnabled(true)
				.withPublic(true);

		when(projectApi.createProject(anyLong(), any(Project.class)))
				.thenReturn(project);
		Project createdProject = projectAPIService.createProjectInNamespace(gitLabApi, sampleProjectRequest);

		PushRules emailRule = new PushRules()
				.withAuthorEmailRegex("^[A-Za-z0-9._+-]+@(student\\.)?tudelft\\.nl$");
		when(projectApi.getPushRules(any(Project.class))).thenReturn(emailRule);

		PushRules rules = projectApi.getPushRules(createdProject);
		String regex = rules.getAuthorEmailRegex();

		System.out.println(regex);

		Pattern pattern = Pattern.compile(regex);

		assertTrue(pattern.matcher("T.U.Delft@tudelft.nl").matches());

		assertTrue(pattern.matcher("t.uDelft@student.tudelft.nl").matches());

		assertFalse(pattern.matcher("T.U.Delft@student.nl").matches());
		assertFalse(pattern.matcher("T.U.Delft@").matches());
		assertFalse(pattern.matcher("T.U.Delft@.nl").matches());
		assertFalse(pattern.matcher("T.U.Delft@nl").matches());
		assertFalse(pattern.matcher("").matches());
		assertFalse(pattern.matcher("TUD").matches());
		assertFalse(pattern.matcher("T.U.Delft@tudelft").matches());
		assertFalse(pattern.matcher("T.U.Delft@other.tudelft.nl").matches());
		assertFalse(pattern.matcher("T.U.Delft@student.tudelft..nl").matches());
		assertFalse(pattern.matcher("T.U.Delft@.student.tudelftnl").matches());
	}
}
