package server.service;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.AdditionalAnswers.returnsFirstArg;
import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;

import java.util.*;

import org.gitlab4j.api.GitLabApi;
import org.gitlab4j.api.GitLabApiException;
import org.gitlab4j.api.UserApi;
import org.gitlab4j.api.models.User;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.Mock;
import org.springframework.data.util.Pair;

import server.entity.Account;
import server.entity.StudentGroup;

public class UserAPIServiceTest {
	@Mock
	private GitLabApi gitLabApi;

	@Mock
	private UserApi userApi;

	private UserAPIService userAPIService;

	@BeforeEach
	public void setup() {
		gitLabApi = mock(GitLabApi.class);
		userAPIService = mock(UserAPIService.class);
		userApi = mock(UserApi.class);
		when(gitLabApi.getUserApi()).thenReturn(userApi);
		userAPIService = new UserAPIService();
	}

	@Test
	public void testCreateUser() throws GitLabApiException {
		User peanuts = new User()
				.withEmail("peanuts@tudelft.nl")
				.withUsername("peanuts");

		//when(userApi.createUser(any(User.class), eq(null), anyBoolean())).then(returnsFirstArg());
		when(userApi.createUser(any(User.class), eq(null), anyBoolean())).thenReturn(peanuts);

		User result = userAPIService.createUser(gitLabApi, "peanuts@tudelft.nl", "peanuts", "peanuts-name");

		assertEquals(result, peanuts);
	}

	@Test
	public void testCreateUserFail() throws GitLabApiException {
		when(userApi.createUser(any(User.class), eq(null), eq(true)))
				.thenThrow(new GitLabApiException("doggo"));

		assertThrows(GitLabApiException.class,
				() -> userAPIService.createUser(gitLabApi, "peanuts@tudelft.nl", "peanuts", "peanuts-name"));
	}

	@Test
	public void testGetUsers() throws GitLabApiException {
		User peanuts = new User()
				.withEmail("peanuts@tudelft.nl")
				.withUsername("peanuts");

		User woodstock = peanuts.withBio("not actually peanuts");

		List<User> users = List.of(peanuts, woodstock);

		when(userApi.getUsers()).thenReturn(users);

		assertEquals(users, userAPIService.getUsers(gitLabApi));
	}

	@Test
	public void testGetUserByEmail() throws GitLabApiException {
		User peanuts = new User()
				.withEmail("peanuts@tudelft.nl")
				.withUsername("peanuts");

		when(userApi.getUserByEmail(eq("peanuts@tudelft.nl"))).thenReturn(peanuts);
		when(userApi.getUserByEmail(eq("woodstock@tudelft.nl"))).thenReturn(null);

		User result1 = userAPIService.getUserByEmail(gitLabApi, "peanuts@tudelft.nl");
		User result2 = userAPIService.getUserByEmail(gitLabApi, "woodstock@tudelft.nl");

		assertEquals(peanuts, result1);
		assertNull(result2);
	}

	@Test
	public void verifyAccounts() throws GitLabApiException {
		User blank = new User()
				.withEmail("blank@mail")
				.withUsername("blank");
		User none = new User()
				.withEmail("none@mail")
				.withUsername("none");
		User clockwork = new User()
				.withEmail("clockwork@mail")
				.withUsername("clockwork");

		Account fakeAccount = new Account("fake", "nevermail", "fake");

		Set<Account> set1 = Set.of(new Account("id", blank.getEmail(), blank.getUsername()),
				new Account("id2", none.getEmail(), none.getUsername()));
		Set<Account> set2 = Set.of(fakeAccount);
		Set<Account> set3 = Set.of(new Account("id3", clockwork.getEmail(), clockwork.getUsername()));

		StudentGroup group1 = new StudentGroup("a");
		StudentGroup group2 = new StudentGroup("b");
		StudentGroup group3 = new StudentGroup("snowblind");

		Map<StudentGroup, Set<Account>> map = new HashMap<>();
		map.put(group1, set1);
		map.put(group2, set2);
		map.put(group3, set3);

		User fake = new User().withEmail("nevermail").withUsername("fake");

		when(userApi.getUserByEmail(anyString())).thenReturn(null);
		when(userApi.getUserByEmail(eq("none@mail"))).thenReturn(none);
		when(userApi.createUser(any(User.class), eq(null), anyBoolean())).then(returnsFirstArg());
		when(userApi.createUser(refEq(fake), eq(null), anyBoolean())).thenThrow(new GitLabApiException(""));

		assertThrows(GitLabApiException.class,
				() -> userApi.createUser(fake, null, false));

		Pair<List<Account>, List<User>> pair = userAPIService.verifyAccounts(gitLabApi, map);

		List<Account> failedUsers = pair.getFirst();

		verify(userApi, times(3)).createUser(any(User.class), eq(null), eq(true));

		List<Account> expectedFailed = new ArrayList<>();
		expectedFailed.add(fakeAccount);
		assertEquals(expectedFailed, failedUsers);

	}
}
