package server.service.statisticsService;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.anyInt;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.*;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.gitlab4j.api.*;
import org.gitlab4j.api.models.*;
import org.junit.jupiter.api.Test;
import org.springframework.data.util.Pair;

class GroupStatisticsServiceTest {

	@Test
	public void getTotalCommitsCount() throws GitLabApiException {
		// Create a mock GitLabApi object
		GitLabApi gitLabApi = mock(GitLabApi.class);

		CommitsApi commitsApi = mock(CommitsApi.class);

		when(gitLabApi.getCommitsApi()).thenReturn(commitsApi);

		// Create a mock CommitsApi object and configure it to return a list of commits
		List<Commit> commits = new ArrayList<>();
		commits.add(new Commit());
		commits.add(new Commit());
		commits.add(new Commit());
		when(commitsApi.getCommits(anyString())).thenReturn(commits);

		// Create an instance of GroupStatisticsService
		GroupStatisticsService service = new GroupStatisticsService();

		// Test the method
		int commitCount = service.getTotalCommitsCount(gitLabApi, "groupPath");

		// Verify the result
		assertEquals(3, commitCount);
	}

	@Test
	public void getIssuesCount() throws GitLabApiException {
		// Create a mock GitLabApi object
		GitLabApi gitLabApi = mock(GitLabApi.class);

		IssuesApi issuesApi = mock(IssuesApi.class);

		when(gitLabApi.getIssuesApi()).thenReturn(issuesApi);

		// Create a mock IssuesApi object and configure it to return a list of issues
		List<Issue> issues = new ArrayList<>();
		issues.add(new Issue());
		issues.add(new Issue());
		when(issuesApi.getIssues(anyString())).thenReturn(issues);

		// Create an instance of GroupStatisticsService
		GroupStatisticsService service = new GroupStatisticsService();

		// Test the method
		int issueCount = service.getIssuesCount(gitLabApi, "groupPath");

		// Verify the result
		assertEquals(2, issueCount);
	}

	@Test
	public void getMergeRequestsCount() throws GitLabApiException {
		// Create a mock GitLabApi object
		GitLabApi gitLabApi = mock(GitLabApi.class);

		MergeRequestApi mergeRequestApi = mock(MergeRequestApi.class);

		when(gitLabApi.getMergeRequestApi()).thenReturn(mergeRequestApi);

		// Create a mock MergeRequestApi object and configure it to return a list of merge requests
		List<MergeRequest> mergeRequests = new ArrayList<>();
		mergeRequests.add(new MergeRequest());
		mergeRequests.add(new MergeRequest());
		when(mergeRequestApi.getMergeRequests(anyString())).thenReturn(mergeRequests);

		// Create an instance of GroupStatisticsService
		GroupStatisticsService service = new GroupStatisticsService();

		// Test the method
		int mergeRequestCount = service.getMergeRequestsCount(gitLabApi, "groupPath");

		// Verify the result
		assertEquals(2, mergeRequestCount);
	}

	@Test
	public void getParticipantsNames() throws GitLabApiException {
		// Create a mock GitLabApi object
		GitLabApi gitLabApi = mock(GitLabApi.class);

		GroupApi groupApi = mock(GroupApi.class);

		when(gitLabApi.getGroupApi()).thenReturn(groupApi);

		// Create a mock GroupApi object and configure it to return a list of members
		List<Member> members = new ArrayList<>();
		members.add(new Member().withName("member1"));
		members.add(new Member().withName("member2"));
		members.add(new Member().withName("member3"));

		Group group = new Group();
		when(groupApi.getGroup("groupPath")).thenReturn(group);
		when(groupApi.getMembers(group)).thenReturn(members);

		// Create an instance of GroupStatisticsService
		GroupStatisticsService service = new GroupStatisticsService();

		// Test the method
		List<String> names = service.getParticipantsNames(gitLabApi, "groupPath");

		// Verify the result
		assertEquals(3, names.size());
		assertEquals("member1", names.get(0));
		assertEquals("member2", names.get(1));
		assertEquals("member3", names.get(2));
	}

	@Test
	void getUsersLinesOfCodeCount() throws GitLabApiException {
		// Create a mock GitLabApi object
		GitLabApi gitLabApi = mock(GitLabApi.class);

		// Create a mock MergeRequestApi object
		MergeRequestApi mergeRequestApi = mock(MergeRequestApi.class);
		when(gitLabApi.getMergeRequestApi()).thenReturn(mergeRequestApi);

		// Create a mock ProjectApi object
		ProjectApi projectApi = mock(ProjectApi.class);
		when(gitLabApi.getProjectApi()).thenReturn(projectApi);

		// Create a mock CommitsApi object
		CommitsApi commitsApi = mock(CommitsApi.class);
		when(gitLabApi.getCommitsApi()).thenReturn(commitsApi);

		// Create a mock Commit object
		Commit commit = mock(Commit.class);
		when(commit.getId()).thenReturn("commitId");

		// Create a mock CommitStats object
		CommitStats commitStats = mock(CommitStats.class);
		when(commitStats.getAdditions()).thenReturn(10);
		when(commitStats.getDeletions()).thenReturn(5);

		// Create a mock MergeRequest object
		MergeRequest mergeRequest = mock(MergeRequest.class);
		when(mergeRequest.getIid()).thenReturn(1L);
		when(mergeRequest.getTargetBranch()).thenReturn("master");

		// Create a mock Commit object for full commit details
		Commit fullCommit = mock(Commit.class);
		when(fullCommit.getStats()).thenReturn(commitStats);
		when(fullCommit.getAuthorName()).thenReturn("user1");

		// Create a mock Project object
		Project project = mock(Project.class);
		when(project.getDefaultBranch()).thenReturn("master");

		// Create a list of MergeRequests
		List<MergeRequest> mergeRequests = new ArrayList<>();
		mergeRequests.add(mergeRequest);

		// Create a list of Commits
		List<Commit> commits = new ArrayList<>();
		commits.add(commit);

		// Configure the mock GitLabApi object to return the mock MergeRequests and Commits
		when(mergeRequestApi.getMergeRequests(anyString())).thenReturn(mergeRequests);
		when(gitLabApi.getMergeRequestApi().getCommits(anyString(), anyInt())).thenReturn(commits);
		when(gitLabApi.getCommitsApi().getCommit(anyString(), anyString())).thenReturn(fullCommit);

		// Configure the mock ProjectApi object to return the mock Project
		when(projectApi.getProject(anyString())).thenReturn(project);

		// Create an instance of GroupStatisticsService
		GroupStatisticsService service = new GroupStatisticsService();

		// Test the method
		Map<String, Pair<Integer, Integer>> userLinesOfCodeCount = service.getUsersLinesOfCodeCount(gitLabApi,
				"repositoryPath");

		// Verify the result
		assertEquals(1, userLinesOfCodeCount.size());
		assertEquals(Pair.of(10, 5), userLinesOfCodeCount.get("user1"));
	}

	@Test
	public void getTotalCommitsCountException() throws GitLabApiException {
		// Create a mock GitLabApi object
		GitLabApi gitLabApi = mock(GitLabApi.class);

		CommitsApi commitsApi = mock(CommitsApi.class);

		when(gitLabApi.getCommitsApi()).thenReturn(commitsApi);

		// Configure the mock CommitsApi object to throw a GitLabApiException
		when(commitsApi.getCommits(anyString())).thenThrow(new GitLabApiException("API Exception"));

		// Create an instance of GroupStatisticsService
		GroupStatisticsService service = new GroupStatisticsService();

		// Test the method
		int commitCount = service.getTotalCommitsCount(gitLabApi, "groupPath");

		// Verify the result
		assertEquals(-1, commitCount);
	}

	@Test
	public void getIssuesCountException() throws GitLabApiException {
		// Create a mock GitLabApi object
		GitLabApi gitLabApi = mock(GitLabApi.class);

		IssuesApi issuesApi = mock(IssuesApi.class);

		when(gitLabApi.getIssuesApi()).thenReturn(issuesApi);

		// Configure the mock IssuesApi object to throw a GitLabApiException
		when(issuesApi.getIssues(anyString())).thenThrow(new GitLabApiException("API Exception"));

		// Create an instance of GroupStatisticsService
		GroupStatisticsService service = new GroupStatisticsService();

		// Test the method
		int issueCount = service.getIssuesCount(gitLabApi, "groupPath");

		// Verify the result
		assertEquals(-1, issueCount);
	}

	@Test
	public void getMergeRequestsCountException() throws GitLabApiException {
		// Create a mock GitLabApi object
		GitLabApi gitLabApi = mock(GitLabApi.class);

		MergeRequestApi mergeRequestApi = mock(MergeRequestApi.class);

		when(gitLabApi.getMergeRequestApi()).thenReturn(mergeRequestApi);

		// Configure the mock MergeRequestApi object to throw a GitLabApiException
		when(mergeRequestApi.getMergeRequests(anyString()))
				.thenThrow(new GitLabApiException("API Exception"));

		// Create an instance of GroupStatisticsService
		GroupStatisticsService service = new GroupStatisticsService();

		// Test the method
		int mergeRequestCount = service.getMergeRequestsCount(gitLabApi, "groupPath");

		// Verify the result
		assertEquals(-1, mergeRequestCount);
	}

	@Test
	public void getParticipantsNamesException() throws GitLabApiException {
		// Create a mock GitLabApi object
		GitLabApi gitLabApi = mock(GitLabApi.class);

		GroupApi groupApi = mock(GroupApi.class);

		when(gitLabApi.getGroupApi()).thenReturn(groupApi);

		// Configure the mock GroupApi object to throw a GitLabApiException
		when(groupApi.getGroup(anyString())).thenThrow(new GitLabApiException("API Exception"));

		// Create an instance of GroupStatisticsService
		GroupStatisticsService service = new GroupStatisticsService();

		// Test the method
		List<String> names = service.getParticipantsNames(gitLabApi, "groupPath");

		// Verify the result
		assertNull(names);
	}

	@Test
	public void getUsersLinesOfCodeCountException() throws GitLabApiException {
		// Create a mock GitLabApi object
		GitLabApi gitLabApi = mock(GitLabApi.class);

		RepositoryApi repositoryApi = mock(RepositoryApi.class);
		MergeRequestApi mergeRequestApi = mock(MergeRequestApi.class);
		ProjectApi projectApi = mock(ProjectApi.class);

		when(gitLabApi.getRepositoryApi()).thenReturn(repositoryApi);
		when(gitLabApi.getMergeRequestApi()).thenReturn(mergeRequestApi);
		when(gitLabApi.getProjectApi()).thenReturn(projectApi);

		// Configure the mock RepositoryApi object to throw a GitLabApiException
		when(repositoryApi.getContributors(anyString())).thenThrow(new GitLabApiException("API Exception"));
		when(projectApi.getProject(anyString())).thenThrow(new GitLabApiException("API Exception"));

		// Create an instance of GroupStatisticsService
		GroupStatisticsService service = new GroupStatisticsService();

		// Test the method
		Map<String, Pair<Integer, Integer>> linesOfCode = service.getUsersLinesOfCodeCount(gitLabApi,
				"groupPath");

		// Verify the result
		assertNull(linesOfCode);
	}

}
