package server.service.statisticsService;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import java.util.ArrayList;
import java.util.List;

import org.gitlab4j.api.*;
import org.gitlab4j.api.models.*;
import org.junit.jupiter.api.Test;

class IndividualStatisticsServiceTest {

	@Test
	public void testGetUserCommitCount() throws GitLabApiException {
		// Create a mock GitLabApi object
		GitLabApi gitLabApi = mock(GitLabApi.class);

		// Create a mock CommitsApi object and configure it to return a list of commits
		CommitsApi commitsApi = mock(CommitsApi.class);
		when(gitLabApi.getCommitsApi()).thenReturn(commitsApi);
		Commit c1 = new Commit().withAuthorName("author1");
		Commit c2 = new Commit().withAuthorName("author1");
		Commit c3 = new Commit().withAuthorName("author2");
		List<Commit> commits = new ArrayList<>();
		commits.add(c1);
		commits.add(c2);
		commits.add(c3);
		when(commitsApi.getCommits(anyString())).thenReturn(commits);

		// Create an instance of IndividualStatisticsService
		IndividualStatisticsService service = new IndividualStatisticsService();

		// Test the method
		int commitCount = service.getUserCommitCount(gitLabApi, "repositoryPath", "author1");

		// Verify the result
		assertEquals(2, commitCount);
	}

	@Test
	public void testGetUserMergeRequestsCount() throws GitLabApiException {
		// Create a mock GitLabApi object
		GitLabApi gitLabApi = mock(GitLabApi.class);

		// Create a mock MergeRequestApi object and configure it to return a list of merge requests
		MergeRequestApi mergeRequestApi = mock(MergeRequestApi.class);
		when(gitLabApi.getMergeRequestApi()).thenReturn(mergeRequestApi);

		Assignee a1 = new Assignee().withName("assignee1");
		Assignee a2 = new Assignee().withName("assignee2");
		Assignee a3 = new Assignee().withName("assignee3");
		List<Assignee> first = new ArrayList<>();
		first.add(a1);
		first.add(a2);
		List<Assignee> second = new ArrayList<>();
		second.add(a1);
		second.add(a3);
		List<Assignee> third = new ArrayList<>();
		third.add(a2);

		MergeRequest m1 = new MergeRequest();
		m1.setAssignees(first);
		MergeRequest m2 = new MergeRequest();
		m2.setAssignees(second);
		MergeRequest m3 = new MergeRequest();
		m3.setAssignees(third);

		List<MergeRequest> mergeRequests = new ArrayList<>();
		mergeRequests.add(m1);
		mergeRequests.add(m2);
		mergeRequests.add(m3);

		when(mergeRequestApi.getMergeRequests(anyString())).thenReturn(mergeRequests);

		// Create an instance of IndividualStatisticsService
		IndividualStatisticsService service = new IndividualStatisticsService();

		// Test the method
		int mergeRequestCount = service.getUserMergeRequestsCount(gitLabApi, "repositoryPath",
				"assignee2");

		// Verify the result
		assertEquals(2, mergeRequestCount);
	}

	@Test
	public void testGetUserIssuesCount() throws GitLabApiException {
		// Create a mock GitLabApi object
		GitLabApi gitLabApi = mock(GitLabApi.class);

		// Create a mock IssuesApi object and configure it to return a list of issues
		IssuesApi issuesApi = mock(IssuesApi.class);

		when(gitLabApi.getIssuesApi()).thenReturn(issuesApi);

		Assignee a1 = new Assignee().withName("assignee1");
		Assignee a2 = new Assignee().withName("assignee2");
		Assignee a3 = new Assignee().withName("assignee3");
		List<Assignee> first = new ArrayList<>();
		first.add(a1);
		first.add(a2);
		List<Assignee> second = new ArrayList<>();
		second.add(a1);
		second.add(a3);
		List<Assignee> third = new ArrayList<>();
		third.add(a2);

		Issue i1 = new Issue();
		i1.setAssignees(first);
		Issue i2 = new Issue();
		i2.setAssignees(second);
		Issue i3 = new Issue();
		i3.setAssignees(third);

		List<Issue> issues = new ArrayList<>();
		issues.add(i1);
		issues.add(i2);
		issues.add(i3);

		when(issuesApi.getIssues(anyString())).thenReturn(issues);

		// Create an instance of IndividualStatisticsService
		IndividualStatisticsService service = new IndividualStatisticsService();

		// Test the method
		int issueCount = service.getUserIssuesCount(gitLabApi, "repositoryPath", "assignee1");

		// Verify the result
		assertEquals(2, issueCount);
	}

	@Test
	public void testGetUserCommentsCountOnMerge() throws GitLabApiException {
		// Create a mock GitLabApi object
		GitLabApi gitLabApi = mock(GitLabApi.class);

		// Create a mock MergeRequestApi object and configure it to return a list of merge requests
		MergeRequestApi mergeRequestApi = mock(MergeRequestApi.class);
		when(gitLabApi.getMergeRequestApi()).thenReturn(mergeRequestApi);

		// Create a mock NotesApi object and configure it to return a list of comments
		NotesApi notesApi = mock(NotesApi.class);
		when(gitLabApi.getNotesApi()).thenReturn(notesApi);

		// Create some mock merge requests
		MergeRequest mr1 = new MergeRequest();
		mr1.setIid(1L);
		MergeRequest mr2 = new MergeRequest();
		mr2.setIid(2L);
		List<MergeRequest> mergeRequests = new ArrayList<>();
		mergeRequests.add(mr1);
		mergeRequests.add(mr2);

		// Configure the merge request API to return the mock merge requests
		when(mergeRequestApi.getMergeRequests(anyString())).thenReturn(mergeRequests);

		// Create some mock comments
		Note comment1 = new Note();
		comment1.setAuthor(new Author().withName("user1"));
		Note comment2 = new Note();
		comment2.setAuthor(new Author().withName("user2"));
		Note comment3 = new Note();
		comment3.setAuthor(new Author().withName("user1"));
		List<Note> comments = new ArrayList<>();
		comments.add(comment1);
		comments.add(comment2);
		comments.add(comment3);

		// Create a mock NotesApi object for a specific merge request and configure it to return the comments
		NotesApi mergeRequestNotesApi = mock(NotesApi.class);
		when(gitLabApi.getNotesApi()).thenReturn(mergeRequestNotesApi);
		when(mergeRequestNotesApi.getMergeRequestNotes(anyString(), anyLong())).thenReturn(comments);

		// Create an instance of IndividualStatisticsService
		IndividualStatisticsService service = new IndividualStatisticsService();

		// Test the method
		int commentCount = service.getUserCommentsCountOnMerge(gitLabApi, "repositoryPath", "user1");

		// Verify the result
		assertEquals(4, commentCount);
	}

	@Test
	public void testGetUserCommitCountException() throws GitLabApiException {
		// Create a mock GitLabApi object
		GitLabApi gitLabApi = mock(GitLabApi.class);

		// Create a mock CommitsApi object and configure it to throw a GitLabApiException
		CommitsApi commitsApi = mock(CommitsApi.class);
		when(gitLabApi.getCommitsApi()).thenReturn(commitsApi);
		when(commitsApi.getCommits(anyString())).thenThrow(new GitLabApiException("API error"));

		// Create an instance of IndividualStatisticsService
		IndividualStatisticsService service = new IndividualStatisticsService();

		// Test the method
		int commitCount = service.getUserCommitCount(gitLabApi, "repositoryPath", "author1");

		// Verify the result
		assertEquals(-1, commitCount);
	}

	@Test
	public void testGetUserMergeRequestsCountException() throws GitLabApiException {
		// Create a mock GitLabApi object
		GitLabApi gitLabApi = mock(GitLabApi.class);

		// Create a mock MergeRequestApi object and configure it to throw a GitLabApiException
		MergeRequestApi mergeRequestApi = mock(MergeRequestApi.class);
		when(gitLabApi.getMergeRequestApi()).thenReturn(mergeRequestApi);
		when(mergeRequestApi.getMergeRequests(anyString())).thenThrow(new GitLabApiException("API error"));

		// Create an instance of IndividualStatisticsService
		IndividualStatisticsService service = new IndividualStatisticsService();

		// Test the method
		int mergeRequestCount = service.getUserMergeRequestsCount(gitLabApi, "repositoryPath", "assignee2");

		// Verify the result
		assertEquals(-1, mergeRequestCount);
	}

	@Test
	public void testGetUserIssuesCountException() throws GitLabApiException {
		// Create a mock GitLabApi object
		GitLabApi gitLabApi = mock(GitLabApi.class);

		// Create a mock IssuesApi object and configure it to throw a GitLabApiException
		IssuesApi issuesApi = mock(IssuesApi.class);
		when(gitLabApi.getIssuesApi()).thenReturn(issuesApi);
		when(issuesApi.getIssues(anyString())).thenThrow(new GitLabApiException("API error"));

		// Create an instance of IndividualStatisticsService
		IndividualStatisticsService service = new IndividualStatisticsService();

		// Test the method
		int issueCount = service.getUserIssuesCount(gitLabApi, "repositoryPath", "assignee1");

		// Verify the result
		assertEquals(-1, issueCount);
	}

	@Test
	public void testGetUserCommentsCountOnMergeException() throws GitLabApiException {
		// Create a mock GitLabApi object
		GitLabApi gitLabApi = mock(GitLabApi.class);

		// Create a mock MergeRequestApi object and configure it to throw a GitLabApiException
		MergeRequestApi mergeRequestApi = mock(MergeRequestApi.class);
		when(gitLabApi.getMergeRequestApi()).thenReturn(mergeRequestApi);
		when(mergeRequestApi.getMergeRequests(anyString())).thenThrow(new GitLabApiException("API error"));

		// Create an instance of IndividualStatisticsService
		IndividualStatisticsService service = new IndividualStatisticsService();

		// Test the method
		int commentCount = service.getUserCommentsCountOnMerge(gitLabApi, "repositoryPath", "user1");

		// Verify the result
		assertEquals(-1, commentCount);
	}

}
