package server.statistics;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.*;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Stream;

import lombok.SneakyThrows;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.test.util.ReflectionTestUtils;

import server.statistics.util.StatisticsEntry;

public class StatisticsRunnerTest {
	@InjectMocks
	private StatisticsRunner sr;

	@Mock
	private CommandRunner cr;

	String username, token, gitlabUrl;

	@BeforeEach
	void setUp() {
		MockitoAnnotations.openMocks(this);

		username = "gitbull";
		token = "token";
		gitlabUrl = "https://gitlab.website.whatever.com";

		ReflectionTestUtils.setField(sr, "username", username);
		ReflectionTestUtils.setField(sr, "token", token);
		ReflectionTestUtils.setField(sr, "gitlabUrl", gitlabUrl);
	}

	@Test
	public void testAddRepoSuccess() {
		StatisticsEntry se = new StatisticsEntry("/path/to/repo", 2);
		sr.addRepository(se.getCourseName(), se.getWeeks());
		assertEquals(sr.getRepositories().get(0), se);
	}

	@Test
	public void testAddRepoAutocompleteSlash() {
		StatisticsEntry seWithoutSlash = new StatisticsEntry("path/to/repo", 2);
		StatisticsEntry seWithSlash = new StatisticsEntry("/path/to/repo", 2);

		sr.addRepository(seWithoutSlash.getCourseName(), seWithoutSlash.getWeeks());

		assertEquals(sr.getRepositories().get(0), seWithSlash);
		assertNotEquals(sr.getRepositories().get(0), seWithoutSlash);
	}

	@Test
	public void testAddRepoBoundaries() {
		sr.addRepository("/path/to/repo", 0);
		sr.addRepository("/path/to/repo", -1);

		assertTrue(sr.getRepositories().isEmpty());

		StatisticsEntry se = new StatisticsEntry("/path/to/repo", 1);
		sr.addRepository(se.getCourseName(), se.getWeeks());

		assertEquals(sr.getRepositories().get(0), se);
	}

	@SneakyThrows
	@Test
	public void testWeekCounted() {
		StatisticsEntry se = new StatisticsEntry("/path/to/repo", 2);
		sr.addRepository(se.getCourseName(), se.getWeeks());

		sr.run();

		se.countWeek();
		assertEquals(sr.getRepositories().get(0), se);

		sr.run();

		assertTrue(sr.getRepositories().isEmpty());

		sr.run();

		assertTrue(sr.getRepositories().isEmpty());
	}

	@SneakyThrows
	@Test
	public void testProperCommands() {
		StatisticsEntry se = new StatisticsEntry("/cse2077/2022-2023/14a/14A", 2);
		sr.addRepository(se.getCourseName(), se.getWeeks());

		doNothing().when(cr).runScript(any());

		sr.run();
		List<String> courseDetails = Stream.of(se.getCourseName().split("/")).toList();

		ArrayList<String> ranCommands = sr.analyzeRepository(se.getCourseName(), se.getAtWeek(),
				courseDetails.get(1),
				courseDetails.get(2));

		ArrayList<String> properCommands = new ArrayList<>();
		properCommands.add("MOVETO /repo");
		properCommands.add("git clone https://" + username + ":" + token + "@" + gitlabUrl.substring(8)
				+ se.getCourseName());
		properCommands.add("MOVETO /python/geit");
		properCommands.add("python3 geit.py --target-repo ./../../repo/14A");
		properCommands.add("MOVETO /repo");
		properCommands.add("rm -r 14A");
		properCommands.add("MOVETO /");
		properCommands.add("mkdir results");
		properCommands.add("mv ./python/geit/index_14A.html ./results/cse2077_2022-2023_14a_14A_1.html");

		assertEquals(ranCommands, properCommands);
	}
}
