import com.diffplug.gradle.spotless.SpotlessExtension
import nl.javadude.gradle.plugins.license.DownloadLicensesExtension
import nl.javadude.gradle.plugins.license.LicenseExtension
import org.openapitools.generator.gradle.plugin.tasks.GenerateTask
import org.springframework.boot.gradle.tasks.run.BootRun
import groovy.util.NodeList
import groovy.util.Node

group = "nl.tudelft.ewi.queue"
version = "1.0.0"

val javaVersion = JavaVersion.VERSION_11

val springBootVersion: String = "2.1.11.RELEASE"

val thymeleafVersion = "3.0.9.RELEASE"
val thymeleafLayoutDialectVersion = "2.3.0"

val lombokVersion = "1.18.12"

val queryDslVersion = "4.2.1"
val jacksonVersion = "2.10.3"
val hibernateVersion = "5.4.8.Final"

val aptSourceDir = file("$buildDir/querydsl/src/apt/java")
val genSourceDir = file("$buildDir/labracore-api/src/main/java")

// A definition of all dependencies and repositories where to find them that need to
// be available during compilation of this build script.
buildscript {
	dependencies {
		// Spring Loaded for hot reloading Java classes within IntelliJ
		// specifically for the Spring Boot development capabilities.
		classpath("org.springframework", "springloaded", "1.2.8.RELEASE")
	}
}

// The repositories used to lookup dependencies.
repositories {
	mavenLocal()
		mavenCentral()

		maven {
			url = uri("https://build.shibboleth.net/nexus/content/repositories/releases")
		}
}

// The plugins used by Gradle to generate files, start Spring boot, perform static analysis etc.
plugins {
    // Plugin for the Kotlin-DSL to be on classpath, disabled
    // because it does not need to be applied during build.
    `kotlin-dsl` apply false

    // Standard plugins for Gradle to work properly
    java
    `maven-publish`
    eclipse
    idea
    jacoco

    // Spring plugins for managing dependencies and creating
    // a nice Spring Boot application.
    id("org.springframework.boot").version("2.1.7.RELEASE")
    id("io.spring.dependency-management").version("1.0.9.RELEASE")

    // Plugin to provide task to check the current versions of
    // dependencies and of Gradle to see if updates are available.
    // TODO: Check if this works after upgrade 0.17.0 -> 0.28.0
    id("com.github.ben-manes.versions").version("0.28.0")

    // Spotless plugin for checking style of Java code.
    // TODO: Check if this works after upgrade 3.24.2 -> 3.27.2
    id("com.diffplug.gradle.spotless").version("3.27.2")

    // Plugin for checking license headers within our code and files.
    id("com.github.hierynomus.license").version("0.15.0")
    id("com.github.hierynomus.license-report").version("0.15.0")

    // Plugin for checking security issues in dependencies of this project.
    id("org.owasp.dependencycheck").version("5.3.0") apply false

    // Open API generator for generating the Client code for Labracore.
    id("org.openapi.generator").version("4.2.3")

    // Sass compiler plugin
    id("com.github.salomonbrys.gradle.sass") version "1.2.0"
}

sourceSets {
	main {
		java {
			srcDir(file("src/main/java"))
				srcDir(genSourceDir)
		}

	}

	test {
		java {
			srcDir(file("src/test/java"))
		}
	}
}

val developmentOnly = configurations.create("developmentOnly")
configurations {
	developmentOnly
		runtimeClasspath {
			extendsFrom(developmentOnly)
		}
	compileOnly {
		extendsFrom(configurations.annotationProcessor.get())
	}
}

/////// Plugins configurations ///////
java {
	sourceCompatibility = javaVersion
		targetCompatibility = javaVersion
}

publishing {
	publications {
		create<MavenPublication>("generatePom") {
			from(components.findByName("java"))

				pom {
					withXml {
						val repo = asNode().appendNode("repositories").appendNode("repository")
							repo.appendNode("id", "shibboleth")
							repo.appendNode("url", "https://build.shibboleth.net/nexus/content/repositories/releases")
					}
				}
		}
	}
}

// Configure IDE plugins.
idea {
	module {
		// Required for Spring Loaded plugin. IDEA should use
		// the same output directory as Gradle.
		inheritOutputDirs = false
		outputDir = file("$buildDir/classes/main/")

		// Configure generated source directories
		generatedSourceDirs.add(genSourceDir)

		// Set IntelliJ to download source and javadoc by default
		isDownloadSources = true
		isDownloadJavadoc = true
	}
}

eclipse {
	classpath {
		// TODO: Check these configurations: JavaSE-1.8 is added in containers?
		containers.remove("org.eclipse.jdt.launching.JRE_CONTAINER")
		containers.add("org.eclipse.jdt.launching.JRE_CONTAINER/org.eclipse.jdt.internal.debug.ui.launcher.StandardVMType/JavaSE-1.8")
	}
}

// Configure Jacoco testing plugin.
configure<JacocoPluginExtension> {
	toolVersion = "0.8.5"
}

// Configure license plugins.
configure<DownloadLicensesExtension> {
	includeProjectDependencies = true
}

configure<LicenseExtension> {
	header = file("$rootDir/LICENSE.header")
	skipExistingHeaders = false

	mapping(mapOf(
		"java" to "SLASHSTAR_STYLE"
	))

	excludes(listOf(
		"**/*.json",

		// Ignore copy-pasted library .js and .css's
		"**/tempusdominus-bootstrap*",
		"**/chosen.*",
		"**/bootstrap-multiselect.*"
	))
}

// Configure Spotless plugin for style checking Java code.
configure<SpotlessExtension> {
	java {
		// Use the eclipse formatter format and import order.
		eclipse().configFile(file("eclipse-formatter.xml"))
		importOrderFile(file("$rootDir/importorder.txt"))

		// Check for a license header in the form of LICENSE.header.java.
		licenseHeaderFile(file("$rootDir/LICENSE.header.java"))

		// Default added rules.
		paddedCell()
		removeUnusedImports()
		trimTrailingWhitespace()
		endWithNewline()

		// Exclude generated source directory.
		targetExclude(fileTree(genSourceDir))
	}
}


/////// TASKS ///////
val jacocoTestReport by tasks.getting(JacocoReport::class) {
	group = "Reporting"
	reports {
		xml.isEnabled = true
		csv.isEnabled = true

		html.destination = file("$buildDir/reports/coverage")
	}
}

// Task for generating the client code for connecting with the Labracore API
val generateLabracoreClient by tasks.register("generateLabracoreClient", GenerateTask::class) {
	// TODO: Re-enable task when merging Labracore branch.
	enabled = false

	// Name of the Client code generator.
	generatorName.set("java")

	// Set output directory, specification and configuration.
	outputDir.set("$buildDir/labracore-api")
	inputSpec.set("${rootDir}/oa-spec-labracore.json")
	configFile.set("${rootDir}/oa-config-labracore.json")

	// Mark the output directory for this task to prevent gradle from marking it as stale
	outputs.dir("$buildDir/labracore-api")
}

val processResources by tasks.getting(ProcessResources::class)

val compileJava by tasks.getting(JavaCompile::class) {
	// Set the output directory for APT
	options.annotationProcessorGeneratedSourcesDirectory = aptSourceDir

	// Set the compiler to incremental and to show warnings
	options.isIncremental = true
	options.compilerArgs.addAll(setOf(
		"-Xlint:deprecation",
		"-Xlint:unchecked"
	))

	// Add dependencies for generating sources and copying resources
	dependsOn.addAll(setOf(
		processResources,
		generateLabracoreClient
	))
}

val jar by tasks.getting(Jar::class) {
	// Set the name and version of the produced JAR
	archiveBaseName.set("queue")
	archiveVersion.set("1.0.0")
}

// Configure Spring Boot plugin task for running the application.
val bootRun by tasks.getting(BootRun::class) {
	sourceResources(sourceSets.main.get())
}

sassCompile {
    source = fileTree("src/main/resources/static/sass")
    outputDir = file("src/main/resources/static/css")
}

tasks.named("processResources") {
    dependsOn("sassCompile")
}

tasks.withType<Test>().configureEach {
	useJUnitPlatform()
}

dependencies {
	// Hibernate for database-entity mapping and EntityManagers.
	implementation("org.hibernate", "hibernate-core", hibernateVersion)
	implementation("org.hibernate", "hibernate-entitymanager", hibernateVersion)
	implementation("org.hibernate", "hibernate-java8", hibernateVersion)

	/////// Spring dependencies ///////

	// Generic Spring Boot starter dependencies
	//implementation("org.springframework.boot", "spring-boot-properties-migrator")
	developmentOnly("org.springframework.boot", "spring-boot-devtools", springBootVersion)
	implementation("org.springframework.boot", "spring-boot-starter-actuator", springBootVersion)
	implementation("org.springframework.boot", "spring-boot-starter-data-jpa", springBootVersion)
	implementation("org.springframework.boot", "spring-boot-starter-web", springBootVersion)
	implementation("org.springframework.boot", "spring-boot-starter-websocket", springBootVersion)
	implementation("org.springframework.boot", "spring-boot-starter-webflux", springBootVersion)

	// Spring Boot and jedis dependencies for interacting with redis session stores.
	implementation("redis.clients", "jedis", "2.9.0")
	implementation("org.springframework.boot", "spring-boot-starter-data-redis", springBootVersion)
	implementation("org.springframework.session", "spring-session-core", springBootVersion)
	implementation("org.springframework.session", "spring-session-data-redis", springBootVersion)

	// Spring dependency for sending push notifications.
	implementation("org.springframework", "spring-messaging")

	// Dependencies for enabling Spring security + SAML security in Spring
	implementation("org.springframework.boot", "spring-boot-starter-security", springBootVersion)
	implementation("org.springframework.security.extensions", "spring-security-saml2-core", "1.0.10.RELEASE")

	// Dependencies for enabling Thymeleaf templating language with Spring.
	implementation("org.springframework.boot", "spring-boot-starter-thymeleaf", springBootVersion)
	implementation("org.thymeleaf", "thymeleaf-spring5", "3.0.11.RELEASE")
	implementation("org.thymeleaf.extras", "thymeleaf-extras-springsecurity4", "3.0.4.RELEASE")
	implementation("org.thymeleaf.extras", "thymeleaf-extras-java8time", "3.0.4.RELEASE")
	implementation("nz.net.ultraq.thymeleaf", "thymeleaf-layout-dialect", "2.4.1")

	// A parent for generated POMs to inherit from for ease of configuration of Spring Boot.
	implementation("org.springframework.boot", "spring-boot-starter-parent", springBootVersion, ext = "pom")

	/////// Other dependencies ///////

	implementation("org.javassist", "javassist", "3.25.0-GA")

	// Dependencies for consuming an Open API generator Client.
	implementation("io.swagger", "swagger-annotations", "1.6.0")
	implementation("org.modelmapper", "modelmapper", "2.3.6")
	implementation("org.openapitools", "jackson-databind-nullable", "0.2.1")

	//    implementation("org.apache.httpcomponents", "fluent-hc", "4.5.5")

	// Bouncycastle for implementations of the Java Crypto API (JDK1.5-11)
	implementation("org.bouncycastle", "bcpkix-jdk15on", "1.64")
	implementation("org.bouncycastle", "bcprov-jdk15on", "1.64")

	// WebPush library for sending push notifications
	implementation("nl.martijndwars", "web-push", "5.1.0")

	//Jackson + JSON for parsing and (de-)serializing JSON objects
	implementation("org.json", "json", "20190722")
	implementation("com.fasterxml.jackson.core", "jackson-core", jacksonVersion)
	implementation("com.fasterxml.jackson.core", "jackson-databind", jacksonVersion)
	implementation("com.fasterxml.jackson.dataformat", "jackson-dataformat-csv", jacksonVersion)
	implementation("com.fasterxml.jackson.module", "jackson-modules-java8", jacksonVersion, ext = "pom")

	// Better Streams API than Java
	implementation("org.jooq", "jool", "0.9.12")

	// Apache commons for many helpful utility classes
	implementation("org.apache.commons", "commons-lang3", "3.0")

	// QueryDSL for providing a DSL based on Entity classes to write database queries
	implementation("com.querydsl", "querydsl-jpa", queryDslVersion)

	// Database migration + database driver dependencies
	implementation("org.liquibase", "liquibase-core")
	implementation("com.h2database", "h2")
	implementation("mysql", "mysql-connector-java", "8.0.18")
	implementation("org.postgresql", "postgresql", "9.4.1212")

	// EE XML parsing and deserialization dependencies
	//    implementation("javax.xml.bind", "jaxb-api", "2.3.0")
	//    implementation("org.glassfish.jaxb", "jaxb-runtime", "2.3.0")

	// EE Jakarta
	implementation("com.sun.activation", "jakarta.activation", "1.2.1")
	implementation("com.sun.mail", "jakarta.mail", "1.6.4")

	// Sentry for writing error logs to a server for developer access
	implementation("io.sentry", "sentry-spring", "1.7.27")

	// Flexmark for parsing Markdown into HTML
	implementation("com.vladsch.flexmark", "flexmark-all", "0.50.42")

	// Webjars to be loaded within HTML resources
	implementation("org.webjars", "webjars-locator-core", "0.43")
	implementation("org.webjars", "jquery", "3.4.1")
	implementation("org.webjars.npm", "ramda", "0.25.0")
	implementation("org.webjars", "bootstrap", "4.3.1")
	implementation("org.webjars", "font-awesome", "5.10.1")
	implementation("org.webjars.bower", "momentjs", "2.20.1")
	implementation("org.webjars", "stomp-websocket", "2.3.3-1")
	implementation("org.webjars", "handlebars", "4.0.14")
	implementation("org.webjars", "chartjs", "2.7.0")
	implementation("org.webjars", "bootstrap-select", "1.13.8")

	/////// Test dependencies ///////
	testImplementation("org.springframework.boot", "spring-boot-starter-test", springBootVersion) {
		exclude("junit", "junit")
	}
	testImplementation("org.springframework.security", "spring-security-test", "5.1.9.RELEASE") {
		exclude("junit", "junit")
	}
	testImplementation("org.junit.jupiter:junit-jupiter:5.6.2")
	testImplementation("org.mockito:mockito-junit-jupiter:3.3.3")


	/////// Annotation processing dependencies ///////
	annotationProcessor("javax.annotation", "javax.annotation-api", "1.3.2")

	annotationProcessor("org.springframework.boot", "spring-boot-configuration-processor", springBootVersion)
	annotationProcessor("org.hibernate.javax.persistence", "hibernate-jpa-2.1-api", "1.0.2.Final")
	annotationProcessor("com.querydsl", "querydsl-apt", queryDslVersion, classifier = "jpa")

	compileOnly("org.projectlombok", "lombok", lombokVersion)
	annotationProcessor("org.projectlombok", "lombok", lombokVersion)
}
