/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue;

import java.time.*;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.PostConstruct;

import nl.tudelft.ewi.queue.model.*;
import nl.tudelft.ewi.queue.repository.AssignmentRepository;
import nl.tudelft.ewi.queue.repository.CourseRepository;
import nl.tudelft.ewi.queue.repository.LabRepository;
import nl.tudelft.ewi.queue.repository.RequestRepository;
import nl.tudelft.ewi.queue.repository.RequestTypeRepository;
import nl.tudelft.ewi.queue.repository.RoomRepository;
import nl.tudelft.ewi.queue.repository.UserRepository;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Profile;
import org.springframework.stereotype.Service;

@Service
@Profile("!production")
public class DatabaseLoader {

	@Autowired
	private UserRepository userRepository;

	@Autowired
	private LabRepository labRepository;

	@Autowired
	private CourseRepository courseRepository;

	@Autowired
	private RequestRepository requestRepository;

	@Autowired
	private RequestTypeRepository requestTypeRepository;

	@Autowired
	private RoomRepository roomRepository;

	private static final Logger logger = LoggerFactory.getLogger(DatabaseLoader.class);

	@Autowired
	private AssignmentRepository assignmentRepository;

	@Value("${institute.netidDomain}")
	private String emailSuffix;

	public DatabaseLoader() {
	}

	/**
	 * Create a new user with the given name as username, password and displayname
	 *
	 * @param  name
	 *
	 * @return
	 */
	private User dummyUser(String name, DefaultRole defaultRole) {
		//The {noop} is the encoder qualifier according to: https://docs.spring.io/spring-security/site/docs/5.2.2.BUILD-SNAPSHOT/reference/htmlsingle/#pe-dpe
		return new User(name + "@" + emailSuffix, "{noop}" + name, name, "a@b.c", defaultRole,
				name.hashCode());
	}

	private void saveUserWithCheck(User user) {
		String newName = StringUtils.stripAccents(user.getDisplayName());
		logger.info("inserted: {}", newName);
		user.setDisplayName(newName);
		userRepository.save(user);
	}

	@PostConstruct
	private void initDatabase() {
		// Normal students
		List<User> students = new ArrayList<>();

		for (int i = 1; i <= 100; i++) {
			User student = dummyUser("student" + i, DefaultRole.ROLE_STUDENT);
			students.add(student);
			userRepository.save(student);
		}

		// Special users
		User student12 = userRepository
				.save(new User("studentCedille" + emailSuffix, "studentCedille", "??????tudent12", "a@b" +
						".c", DefaultRole.ROLE_STUDENT, -2));
		saveUserWithCheck(student12);

		User student13 = new User("utf8" + emailSuffix, "", "UTF8 Mu??????i??????",
				"utf8" + emailSuffix, DefaultRole.ROLE_STUDENT, -2);
		saveUserWithCheck(student13);

		User assistant1 = userRepository.save(dummyUser("assistant1", DefaultRole.ROLE_STUDENT));
		User assistant2 = userRepository.save(dummyUser("assistant2", DefaultRole.ROLE_STUDENT));
		User assistant3 = userRepository.save(dummyUser("assistant3", DefaultRole.ROLE_STUDENT));
		User assistant4 = userRepository.save(dummyUser("assistant4", DefaultRole.ROLE_STUDENT));

		User manager1 = userRepository.save(dummyUser("manager1", DefaultRole.ROLE_STUDENT));
		User manager2 = userRepository.save(dummyUser("manager2", DefaultRole.ROLE_STUDENT));
		User manager3 = userRepository.save(dummyUser("manager3", DefaultRole.ROLE_STUDENT));

		User teacher1 = userRepository.save(dummyUser("teacher1", DefaultRole.ROLE_TEACHER));
		User teacher2 = userRepository.save(dummyUser("teacher2", DefaultRole.ROLE_TEACHER));
		User teacher3 = userRepository.save(dummyUser("teacher3", DefaultRole.ROLE_TEACHER));
		User teacher4 = userRepository.save(dummyUser("teacher4", DefaultRole.ROLE_TEACHER));

		User admin1 = userRepository.save(dummyUser("admin", DefaultRole.ROLE_ADMIN));
		User admin2 = userRepository.save(dummyUser("admin2", DefaultRole.ROLE_ADMIN));

		// Course
		Course course1 = new Course("Algorithms & Datastructures", "TI1316");
		Assignment assignment1 = new Assignment(course1, "Assembly programming");
		course1.addAssignment(assignment1);

		courseRepository.save(course1);
		assignmentRepository.save(assignment1);

		Course course2 = new Course("Concepts of Programming Languages", "TI2606");
		Assignment assignment2 = new Assignment(course2, "Interpreter implementation");

		course2.addAssignment(assignment2);

		courseRepository.save(course2);
		assignmentRepository.save(assignment2);

		Course course3 = new Course("OOP", "TI1206");
		course3.addAssignment(new Assignment(course3, "Week 1"));
		course3.addAssignment(new Assignment(course3, "Week 2"));
		course3.addAssignment(new Assignment(course3, "Week 3"));
		course3.addAssignment(new Assignment(course3, "Week 4"));
		course3.addAssignment(new Assignment(course3, "Week 5"));
		course3.addAssignment(new Assignment(course3, "Week 6"));
		course3.addAssignment(new Assignment(course3, "Week 7"));
		course3.addAssignment(new Assignment(course3, "Week 8"));
		course3.addAssignment(new Assignment(course3, "Week 9"));

		courseRepository.save(course3);

		Course course4 = new Course("Computerorganisation", "TI1406");
		course4.addAssignment(new Assignment(course4, "Assembly factorial"));
		course4.addAssignment(new Assignment(course4, "Assembly ping-pong"));

		courseRepository.save(course4);

		for (int i = 0; i < 25; i++) {
			User student1 = students.get(i);
			User student2 = students.get(i + 25);
			User student3 = students.get(i + 50);
			User student4 = students.get(i + 75);
			student1.addRole(new Student(student1, course1, LocalDateTime.now()));
			student2.addRole(new Student(student2, course2, LocalDateTime.now()));
			student3.addRole(new Student(student3, course3, LocalDateTime.now()));
			student4.addRole(new Student(student4, course4, LocalDateTime.now()));
			userRepository.save(student1);
			userRepository.save(student2);
			userRepository.save(student3);
			userRepository.save(student4);
		}

		assistant1.addRole(new Assistant(assistant1, course1));
		assistant2.addRole(new Assistant(assistant2, course1));
		assistant3.addRole(new Assistant(assistant3, course1));
		assistant4.addRole(new Assistant(assistant4, course1));

		assistant1.addRole(new Student(assistant1, course2, LocalDateTime.now()));
		assistant2.addRole(new Assistant(assistant2, course2));
		assistant3.addRole(new Assistant(assistant3, course2));
		assistant4.addRole(new Assistant(assistant4, course2));

		assistant1.addRole(new Student(assistant1, course3, LocalDateTime.now()));
		assistant2.addRole(new Student(assistant2, course3, LocalDateTime.now()));
		assistant3.addRole(new Assistant(assistant3, course3));
		assistant4.addRole(new Assistant(assistant4, course3));

		assistant1.addRole(new Student(assistant1, course4, LocalDateTime.now()));
		assistant2.addRole(new Student(assistant2, course4, LocalDateTime.now()));
		assistant3.addRole(new Student(assistant3, course4, LocalDateTime.now()));
		assistant4.addRole(new Assistant(assistant4, course4));

		manager1.addRole(new Manager(manager1, course1));
		manager2.addRole(new Manager(manager2, course2));
		manager3.addRole(new Manager(manager3, course2));

		teacher1.addRole(new Teacher(teacher1, course1));
		teacher2.addRole(new Teacher(teacher2, course1));
		teacher3.addRole(new Teacher(teacher3, course2));
		teacher4.addRole(new Teacher(teacher4, course2));

		userRepository.save(assistant1);
		userRepository.save(assistant2);
		userRepository.save(assistant3);
		userRepository.save(assistant4);

		userRepository.save(manager1);
		userRepository.save(manager2);
		userRepository.save(manager3);

		userRepository.save(teacher1);
		userRepository.save(teacher2);
		userRepository.save(teacher3);
		userRepository.save(teacher4);

		userRepository.save(admin1);
		userRepository.save(admin2);

		RequestType requestType = new RequestType();
		requestType.setName("Submission");
		requestTypeRepository.save(requestType);
		RequestType questionType = new RequestType();
		questionType.setName("Question");
		requestTypeRepository.save(questionType);

		// Labs
		Lab lab1 = new Lab();
		lab1.setCourse(course1);
		lab1.setSlot(new LabSlot(LocalDateTime.now().plusMinutes(1),
				LocalDateTime.now().plusHours(4).plusMinutes(1)));
		lab1.setSignOffIntervals(false);
		lab1.setIntervalTime(15L);
		lab1.setCapacity(3L);
		lab1.addAssignment(assignment1);
		ArrayList<RequestType> requestTypes = new ArrayList<>();
		requestTypes.add(requestType);
		requestTypes.add(questionType);
		lab1.setAllowedRequestTypes(requestTypes);
		lab1.setTitle("ADS Assembly 1");
		Room room1 = new Room("DW-PC 4 (060 1verd)");
		Room room2 = new Room("DW-PC 3 (010 1verd)");
		Room room3 = new Room("DW-PC 2 (200BG)");
		roomRepository.save(room1);
		roomRepository.save(room2);
		roomRepository.save(room3);

		lab1.addRoom(room1);
		lab1.addRoom(room2);
		lab1.addRoom(room3);
		lab1.setCommunicationMethod(CommunicationMethod.JITSI_MEET);

		roomRepository.save(room1);
		roomRepository.save(room2);
		roomRepository.save(room3);
		labRepository.save(lab1);

		for (int i = 0; i < 25; i++) {
			Request request = new Request(students.get(i), assignment1, room1, requestType, "", lab1);
			request.setJitsiRoom();
			requestRepository.save(request);
		}

		Lab lab2 = new Lab();
		lab2.setCourse(course1);
		lab2.setSlot(new LabSlot(LocalDateTime.now(), LocalDateTime.now().plusHours(4)));
		lab2.setSignOffIntervals(false);
		lab2.setIntervalTime(15L);
		lab2.setCapacity(3L);
		lab2.addAssignment(assignment1);
		lab2.setAllowedRequestTypes(requestTypes);
		Room room4 = new Room("DW-PC 3 (010 1verd)2");
		Room room5 = new Room("DW-PC 2 (200BG)2");
		lab2.addRoom(room4);
		lab2.addRoom(room5);
		lab2.setTitle("ADS Assembly 2");
		lab2.setCommunicationMethod(CommunicationMethod.STUDENT_VISIT_TA);

		assignment1.addLab(lab1);
		assignment1.addLab(lab2);

		roomRepository.save(room4);
		roomRepository.save(room5);
		labRepository.save(lab2);
		assignmentRepository.save(assignment1);
		for (int i = 0; i < 25; i++) {
			Request request = new Request(students.get(i), assignment1, room4, requestType, "", lab2);
			requestRepository.save(request);
		}

		Lab lab3 = new Lab();
		lab3.setTitle("Exam lab");
		lab3.setCourse(course2);
		lab3.setSlot(new LabSlot(LocalDateTime.now().plusMinutes(1),
				LocalDateTime.now().plusHours(4).plusMinutes(1)));
		lab3.setSignOffIntervals(true);
		lab3.setSlotSelectionOpensAt(LocalDateTime.now());
		lab3.setIntervalTime(15L);
		lab3.setCapacity(3L);
		lab3.addAssignment(assignment2);
		lab3.setAllowedRequestTypes(requestTypes);
		Room room6 = new Room("DW-PC 3 (010 1verd)1");
		Room room7 = new Room("DW-PC 2 (200BG)1");
		lab3.addRoom(room6);
		lab3.addRoom(room7);
		lab3.setCommunicationMethod(CommunicationMethod.STUDENT_VISIT_TA);
		lab3.setTitle("CPL: Interpreter");
		assignment2.addLab(lab3);

		lab3.setExamLab(true);
		lab3.setExamLabPercentage(10);
		roomRepository.save(room6);
		roomRepository.save(room7);
		lab3 = labRepository.save(lab3);
		assignmentRepository.save(assignment2);

		for (int i = 0; i < 25; i++) {
			RequestSlot requestSlot = new RequestSlot();
			requestSlot.setOpensAt(LocalDateTime.now());
			requestSlot.setClosesAt(LocalDateTime.now().plusMinutes(15));

			Request request = new Request(students.get(i), assignment2, room6, requestType, "", lab3,
					requestSlot);
			requestSlot.setRequest(request);
			requestRepository.save(request);
		}
	}
}
