/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue;

import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.Security;
import java.security.spec.InvalidKeySpecException;
import java.util.List;
import java.util.Locale;

import javax.annotation.PostConstruct;

import nl.martijndwars.webpush.PushService;
import nl.martijndwars.webpush.Utils;
import nl.tudelft.ewi.queue.dialect.AuthenticatedDialect;
import nl.tudelft.ewi.queue.dialect.FilterDialect;
import nl.tudelft.ewi.queue.dialect.PrettyTimeDialect;
import nl.tudelft.ewi.queue.dialect.RequestDialect;
import nl.tudelft.ewi.queue.resolver.UserArgumentResolver;

import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.modelmapper.ModelMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.boot.autoconfigure.domain.EntityScan;
import org.springframework.boot.web.servlet.ServletContextInitializer;
import org.springframework.cache.annotation.EnableCaching;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.data.jpa.convert.threeten.Jsr310JpaConverters;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.web.method.support.HandlerMethodArgumentResolver;
import org.springframework.web.servlet.HandlerExceptionResolver;
import org.springframework.web.servlet.LocaleResolver;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;
import org.springframework.web.servlet.i18n.FixedLocaleResolver;
import org.thymeleaf.extras.java8time.dialect.Java8TimeDialect;
import org.thymeleaf.extras.springsecurity4.dialect.SpringSecurityDialect;

import io.sentry.spring.SentryExceptionResolver;
import io.sentry.spring.SentryServletContextInitializer;

@EntityScan(basePackageClasses = { QueueApplication.class, Jsr310JpaConverters.class })
@SpringBootApplication
@EnableScheduling
@EnableCaching
public class QueueApplication implements
		WebMvcConfigurer {
	private static final Logger logger = LoggerFactory.getLogger(QueueApplication.class);
	private static final LocaleResolver localeResolver = new FixedLocaleResolver(Locale.ENGLISH);
	private static final Java8TimeDialect java8TimeDialect = new Java8TimeDialect();
	private static final AuthenticatedDialect authenticatedDialect = new AuthenticatedDialect();
	private static final PrettyTimeDialect prettyTimeDialect = new PrettyTimeDialect();
	private static final RequestDialect requestDialect = new RequestDialect();
	private static final FilterDialect filterDialect = new FilterDialect();
	private static final SpringSecurityDialect springSecurityDialect = new SpringSecurityDialect();

	@Autowired
	private ApplicationContext context;

	@Value("${queue.gcmApiKey}")
	private String gcmApiKey;

	@Value("${queue.pushPublicKey}")
	private String pushPublicKey;

	@Value("${queue.pushPrivateKey}")
	private String pushPrivateKey;

	public static void main(String[] args) {
		SpringApplication.run(QueueApplication.class, args);
	}

	/**
	 * Register a HandlerExceptionResolver that sends all exceptions to Sentry and then defers all handling to
	 * the other HandlerExceptionResolvers.
	 *
	 * @return the Sentry Exception Resolver
	 */
	@Bean
	public HandlerExceptionResolver sentryExceptionResolver() {
		return new SentryExceptionResolver();
	}

	/**
	 * Register a ServletContextInitializer that installs the SentryServletRequestListener so that Sentry
	 * events contain HTTP request information.
	 *
	 * @return the SentryServletContextInitializer
	 */
	@Bean
	public ServletContextInitializer sentryServletContextInitializer() {
		return new SentryServletContextInitializer();
	}

	@PostConstruct
	public static void registerSecurityProvider() {
		Security.addProvider(new BouncyCastleProvider());
	}

	@Override
	public void addArgumentResolvers(List<HandlerMethodArgumentResolver> argumentResolvers) {
		argumentResolvers.add(context.getBean(UserArgumentResolver.class));
	}

	@Bean
	public static LocaleResolver localeResolver() {
		return localeResolver;
	}

	@Bean
	public ModelMapper modelMapper() {
		return new ModelMapper();
	}

	@Bean
	public PushService pushService() {
		PushService pushService = new PushService(gcmApiKey);
		if (pushPublicKey == null || pushPrivateKey == null
				|| pushPublicKey.equals("VGhpcyBpcyBhIHB1YmxpYyBrZXk=")
				|| pushPrivateKey.equals("VGhpcyBpcyBhIHByaXZhdGUga2V5")) {
			System.err.println("***** You did not define the push keys yet! Push service unavailable! *****");
			return new PushService();
		}

		try {
			pushService.setPublicKey(Utils.loadPublicKey(pushPublicKey));
			pushService.setPrivateKey(Utils.loadPrivateKey(pushPrivateKey));
		} catch (IllegalArgumentException | NoSuchProviderException | NoSuchAlgorithmException
				| InvalidKeySpecException | NoSuchMethodError e) {
			// TODO FIXME Catching NoSuchMethodError here because maven has trouble loading Bouncy Castle
			logger.error("Could not load keys for push API");
		}
		return pushService;
	}

	@Bean
	public static Java8TimeDialect java8TimeDialect() {
		return java8TimeDialect;
	}

	@Bean
	public static AuthenticatedDialect authenticatedDialect() {
		return authenticatedDialect;
	}

	@Bean
	public static PrettyTimeDialect prettyTimeDialect() {
		return prettyTimeDialect;
	}

	@Bean
	public static RequestDialect requestDialect() {
		return requestDialect;
	}

	@Bean
	public static FilterDialect filterDialect() {
		return filterDialect;
	}

	@Bean
	public static SpringSecurityDialect securityDialect() {
		return springSecurityDialect;
	}
}
