/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Timer;

import nl.tudelft.ewi.queue.service.SAMLUserDetailsServiceImpl;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.velocity.app.VelocityEngine;
import org.opensaml.saml2.metadata.provider.HTTPMetadataProvider;
import org.opensaml.saml2.metadata.provider.MetadataProvider;
import org.opensaml.saml2.metadata.provider.MetadataProviderException;
import org.opensaml.xml.parse.ParserPool;
import org.opensaml.xml.parse.StaticBasicParserPool;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Profile;
import org.springframework.core.io.ResourceLoader;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.saml.*;
import org.springframework.security.saml.context.SAMLContextProviderImpl;
import org.springframework.security.saml.context.SAMLContextProviderLB;
import org.springframework.security.saml.key.JKSKeyManager;
import org.springframework.security.saml.key.KeyManager;
import org.springframework.security.saml.log.SAMLDefaultLogger;
import org.springframework.security.saml.metadata.*;
import org.springframework.security.saml.parser.ParserPoolHolder;
import org.springframework.security.saml.processor.*;
import org.springframework.security.saml.storage.EmptyStorageFactory;
import org.springframework.security.saml.util.VelocityFactory;
import org.springframework.security.saml.websso.*;
import org.springframework.security.web.DefaultSecurityFilterChain;
import org.springframework.security.web.FilterChainProxy;
import org.springframework.security.web.access.channel.ChannelProcessingFilter;
import org.springframework.security.web.authentication.SavedRequestAwareAuthenticationSuccessHandler;
import org.springframework.security.web.authentication.SimpleUrlAuthenticationFailureHandler;
import org.springframework.security.web.authentication.logout.LogoutHandler;
import org.springframework.security.web.authentication.logout.SecurityContextLogoutHandler;
import org.springframework.security.web.authentication.logout.SimpleUrlLogoutSuccessHandler;
import org.springframework.security.web.authentication.www.BasicAuthenticationFilter;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;

@Profile("production")
@Configuration
@EnableWebSecurity
public class SamlWebSecurityConfig extends WebSecurityConfigurerAdapter {

	@Autowired
	private ResourceLoader resourceLoader;

	@Autowired
	private SAMLUserDetailsServiceImpl samlUserDetailsServiceImpl;

	/**
	 * In the java keystore, a public-private key pair to identify this service provider (Queue) to an
	 * identity provider and to sign/encrypt messages exchanged with the IDP.
	 *
	 * This is the path to the keystore.
	 */
	@Value("${saml.keystore.path}")
	private String keystorePath;

	/**
	 * The name of the keypair to access from the keystore.
	 */
	@Value("${saml.keystore.alias}")
	private String keystoreAlias;

	/**
	 * Password for accessing the keystore and the specific keypair. Note: currently, the password for
	 * accessing the private key and the public key have to be exactly the same.
	 */
	@Value("${saml.keystore.password}")
	private String keystorePassword;

	/**
	 * The entity ID of the Queue Service Provider. Normally, this is the URL to the metadata.xml file of a
	 * service (so queue.tudelft.nl/saml/metadata for instance).
	 */
	@Value("${saml.entityId}")
	private String entityId;

	/**
	 * The URL to construct the SAML POST endpoint from. This URL should contain a protocol, server, port and
	 * context path. When testing locally, this could be http://localhost:8081 for instance. When live, this
	 * could be https://queue.tudelft.nl.
	 */
	@Value("${saml.entityBaseURL}")
	private String entityBaseURL;

	/**
	 * The protocol that the IDP is expected to use to contact this SP upon confirmation of the requested
	 * identity.
	 */
	@Value("${saml.contextProvider.scheme}")
	private String contextProviderScheme;

	/**
	 * The server name the IDP is expected to use to contact this SP.
	 */
	@Value("${saml.contextProvider.serverName}")
	private String contextProviderServerName;

	/**
	 * The port the IDP is expected to use to contact this SP.
	 */
	@Value("${saml.contextProvider.port}")
	private int contextProviderPort;

	/**
	 * The URL at which the metadata.xml file can be found for the configured identity provider.
	 */
	@Value("${saml.metadataUrl}")
	private String metadataProductionUrl;

	/**
	 * If set to true, Spring will check if the signature used in the metadata is valid. (Preferably this
	 * always happens, just to be sure that the metadata is valid)
	 */
	@Value("${saml.metadataTrustCheck}")
	private boolean metadataProductionTrustCheck;

	/**
	 * If set to true, Spring will reject the metadata file if it is not digitally signed by the IDP.
	 */
	@Value("${saml.metadataRequirementSignature}")
	private boolean metadataProductionRequireSignature;

	/**
	 * If set to true, Spring will track the HTTP session and check the SentInResponseTo header.
	 */
	@Value("${saml.checkSentInResponseToHeader}")
	private boolean checkSentInResponseToHeader = true;

	protected HttpSecurity samlizedConfig(HttpSecurity http) throws Exception {
		//@formatter:off
        http
            .httpBasic()
                .authenticationEntryPoint(samlEntryPoint())
                .and()
            .csrf()
                .ignoringAntMatchers("/saml/**")
                .and()
            .authorizeRequests()
                .antMatchers("/saml/**")
                .permitAll()
                .and()
            .addFilterBefore(metadataGeneratorFilter(), ChannelProcessingFilter.class)
            .addFilterAfter(samlFilter(), BasicAuthenticationFilter.class)
            .exceptionHandling().accessDeniedPage("/error.html");
        //@formatter:on

		return http;
	}

	/**
	 * Defines the web based security configuration.
	 *
	 * @param  http      It allows configuring web based security for specific http requests.
	 * @throws Exception
	 */
	@Override
	protected void configure(HttpSecurity http) throws Exception {
		http = samlizedConfig(http);

		http.authorizeRequests()
				.antMatchers("/").permitAll()
				.antMatchers("/manifest.json").permitAll()
				.antMatchers("/favicon.ico").permitAll()
				.antMatchers("/sw.js").permitAll()
				.antMatchers("/css/**").permitAll()
				.antMatchers("/img/**").permitAll()
				.antMatchers("/js/**").permitAll()
				.antMatchers("/webjars/**").permitAll()
				.antMatchers("/stomp/**").permitAll()
				.antMatchers("/saml/**").permitAll()
				.antMatchers("/lab/submit*").permitAll()
				.antMatchers("/privacy").permitAll()
				.anyRequest()
				.authenticated();

		http
				.logout()
				.logoutSuccessUrl("/");
	}

	@Bean
	public static SAMLBootstrap SAMLBootstrap() {
		return new SAMLBootstrap();
	}

	// Initialization of the velocity engine
	@Bean
	public VelocityEngine velocityEngine() {
		return VelocityFactory.getEngine();
	}

	// XML parser pool needed for OpenSAML parsing
	@Bean(initMethod = "initialize")
	public StaticBasicParserPool parserPool() {
		return new StaticBasicParserPool();
	}

	@Bean(name = "parserPoolHolder")
	public ParserPoolHolder parserPoolHolder() {
		return new ParserPoolHolder();
	}

	// Bindings, encoders and decoders used for creating and parsing messages
	@Bean
	public MultiThreadedHttpConnectionManager multiThreadedHttpConnectionManager() {
		return new MultiThreadedHttpConnectionManager();
	}

	@Bean
	public HttpClient httpClient() {
		return new HttpClient(multiThreadedHttpConnectionManager());
	}

	// SAML Authentication Provider responsible for validating of received SAML messages
	@Bean
	public SAMLAuthenticationProvider samlAuthenticationProvider() {
		SAMLAuthenticationProvider samlAuthenticationProvider = new SAMLAuthenticationProvider();
		samlAuthenticationProvider.setUserDetails(samlUserDetailsServiceImpl);
		samlAuthenticationProvider.setForcePrincipalAsString(false);

		return samlAuthenticationProvider;
	}

	/**
	 * Provider of default SAML Context.
	 *
	 * Normally, we would return an instance of SAMLContextProviderImpl. For queue, we return an instance of
	 * SAMLContextProviderLB. This provider deals with reverse proxies, where SSL is terminated before the
	 * request hits the application. See also https://goo.gl/GXi3Q1.
	 *
	 * @return
	 */
	@Bean
	public SAMLContextProviderImpl contextProvider() {
		SAMLContextProviderLB samlContextProvider = new SAMLContextProviderLB();
		samlContextProvider.setScheme(contextProviderScheme);
		samlContextProvider.setServerName(contextProviderServerName);
		samlContextProvider.setServerPort(contextProviderPort);
		samlContextProvider.setIncludeServerPortInRequestURL(false);
		samlContextProvider.setContextPath("/");

		if (!checkSentInResponseToHeader) {
			samlContextProvider.setStorageFactory(new EmptyStorageFactory());
		}

		return samlContextProvider;
	}

	// Logger for SAML messages and events
	@Bean
	public SAMLDefaultLogger samlLogger() {
		return new SAMLDefaultLogger();
	}

	// SAML 2.0 WebSSO Assertion Consumer
	@Bean
	public WebSSOProfileConsumer webSSOprofileConsumer() {
		return new WebSSOProfileConsumerImpl();
	}

	// SAML 2.0 Holder-of-Key WebSSO Assertion Consumer
	@Bean
	public WebSSOProfileConsumerHoKImpl hokWebSSOprofileConsumer() {
		return new WebSSOProfileConsumerHoKImpl();
	}

	// SAML 2.0 Web SSO profile
	@Bean
	public WebSSOProfile webSSOprofile() {
		return new WebSSOProfileImpl();
		// WebSSOProfileImpl defines maxAuthenticationAge as 7200 seconds. Increase if necessary
	}

	// SAML 2.0 Holder-of-Key Web SSO profile
	@Bean
	public WebSSOProfileConsumerHoKImpl hokWebSSOProfile() {
		return new WebSSOProfileConsumerHoKImpl();
	}

	// SAML 2.0 ECP profile
	@Bean
	public WebSSOProfileECPImpl ecpprofile() {
		return new WebSSOProfileECPImpl();
	}

	@Bean
	public SingleLogoutProfile logoutprofile() {
		return new SingleLogoutProfileImpl();
	}

	/**
	 * @return The key manager for SAML. This keymanager refers to the java keystore that can be accessed with
	 *         a configured pass.
	 */
	@Bean
	public KeyManager keyManager() {
		return new JKSKeyManager(
				// File pointing to the keystore
				resourceLoader.getResource(keystorePath),
				// Password to access the keystore (null = no password)
				keystorePassword,
				// Passwords used to access private keys
				ImmutableMap.of(keystoreAlias, keystorePassword),
				// Default key
				keystoreAlias);
	}

	@Bean
	public WebSSOProfileOptions defaultWebSSOProfileOptions() {
		WebSSOProfileOptions webSSOProfileOptions = new WebSSOProfileOptions();
		webSSOProfileOptions.setIncludeScoping(false);

		// Force IdP to re-authenticate user if issued token is too old to prevent
		// "Authentication statement is too old to be used with value" exception
		// See: http://stackoverflow.com/questions/30528636/saml-login-errors
		webSSOProfileOptions.setForceAuthN(true);

		return webSSOProfileOptions;
	}

	// Entry point to initialize authentication, default values taken from
	// properties file
	@Bean
	public SAMLEntryPoint samlEntryPoint() {
		SAMLEntryPoint samlEntryPoint = new SAMLEntryPoint();
		samlEntryPoint.setDefaultProfileOptions(defaultWebSSOProfileOptions());
		return samlEntryPoint;
	}

	// Setup advanced info about metadata
	@Bean
	public ExtendedMetadata extendedMetadata() {
		ExtendedMetadata extendedMetadata = new ExtendedMetadata();
		extendedMetadata.setIdpDiscoveryEnabled(false);
		extendedMetadata.setSignMetadata(false);

		return extendedMetadata;
	}

	@Bean
	@Qualifier("SSO-metadata")
	public ExtendedMetadataDelegate ssoMetaDataProvider()
			throws MetadataProviderException {
		Timer backgroundTaskTimer = new Timer(true);

		HTTPMetadataProvider httpMetadataProvider = new HTTPMetadataProvider(backgroundTaskTimer,
				httpClient(), metadataProductionUrl);
		httpMetadataProvider.setParserPool(parserPool());

		ExtendedMetadataDelegate extendedMetadataDelegate = new ExtendedMetadataDelegate(httpMetadataProvider,
				extendedMetadata());
		extendedMetadataDelegate.setMetadataTrustCheck(metadataProductionTrustCheck);
		extendedMetadataDelegate.setMetadataRequireSignature(metadataProductionRequireSignature);
		return extendedMetadataDelegate;
	}

	// IDP Metadata configuration - paths to metadata of IDPs in circle of trust
	// is here
	// Do no forget to call initialize method on providers
	@Bean
	@Qualifier("metadata")
	public CachingMetadataManager metadata() throws MetadataProviderException {
		List<MetadataProvider> providers = new ArrayList<>();
		providers.add(ssoMetaDataProvider());

		return new CachingMetadataManager(providers);
	}

	// Filter automatically generates default SP metadata
	@Bean
	public MetadataGenerator metadataGenerator() {
		MetadataGenerator metadataGenerator = new MetadataGenerator();
		metadataGenerator.setEntityId(entityId);
		metadataGenerator.setEntityBaseURL(entityBaseURL);
		metadataGenerator.setExtendedMetadata(extendedMetadata());
		metadataGenerator.setKeyManager(keyManager());

		return metadataGenerator;
	}

	// The filter is waiting for connections on URL suffixed with filterSuffix
	// and presents SP metadata there
	@Bean
	public MetadataDisplayFilter metadataDisplayFilter() {
		return new MetadataDisplayFilter();
	}

	// Handler deciding where to redirect user after successful login
	@Bean
	public SavedRequestAwareAuthenticationSuccessHandler successRedirectHandler() {
		SavedRequestAwareAuthenticationSuccessHandler successRedirectHandler = new SavedRequestAwareAuthenticationSuccessHandler();
		successRedirectHandler.setDefaultTargetUrl("/");

		return successRedirectHandler;
	}

	// Handler deciding where to redirect user after failed login
	@Bean
	public SimpleUrlAuthenticationFailureHandler authenticationFailureHandler() {
		SimpleUrlAuthenticationFailureHandler failureHandler = new SimpleUrlAuthenticationFailureHandler();
		failureHandler.setUseForward(true);
		failureHandler.setDefaultFailureUrl("/error");
		return failureHandler;
	}

	@Bean
	public SAMLWebSSOHoKProcessingFilter samlWebSSOHoKProcessingFilter() throws Exception {
		SAMLWebSSOHoKProcessingFilter samlWebSSOHoKProcessingFilter = new SAMLWebSSOHoKProcessingFilter();
		samlWebSSOHoKProcessingFilter.setAuthenticationSuccessHandler(successRedirectHandler());
		samlWebSSOHoKProcessingFilter.setAuthenticationManager(authenticationManager());
		samlWebSSOHoKProcessingFilter.setAuthenticationFailureHandler(authenticationFailureHandler());
		return samlWebSSOHoKProcessingFilter;
	}

	// Processing filter for WebSSO profile messages
	@Bean
	public SAMLProcessingFilter samlWebSSOProcessingFilter() throws Exception {
		SAMLProcessingFilter samlWebSSOProcessingFilter = new SAMLProcessingFilter();
		samlWebSSOProcessingFilter.setAuthenticationManager(authenticationManager());
		samlWebSSOProcessingFilter.setAuthenticationSuccessHandler(successRedirectHandler());
		samlWebSSOProcessingFilter.setAuthenticationFailureHandler(authenticationFailureHandler());
		return samlWebSSOProcessingFilter;
	}

	@Bean
	public MetadataGeneratorFilter metadataGeneratorFilter() {
		return new MetadataGeneratorFilter(metadataGenerator());
	}

	// Handler for successful logout
	@Bean
	public SimpleUrlLogoutSuccessHandler successLogoutHandler() {
		SimpleUrlLogoutSuccessHandler successLogoutHandler = new SimpleUrlLogoutSuccessHandler();
		successLogoutHandler.setDefaultTargetUrl("/");
		return successLogoutHandler;
	}

	// Logout handler terminating local session
	@Bean
	public SecurityContextLogoutHandler logoutHandler() {
		SecurityContextLogoutHandler logoutHandler = new SecurityContextLogoutHandler();
		logoutHandler.setInvalidateHttpSession(true);
		logoutHandler.setClearAuthentication(true);
		return logoutHandler;
	}

	// Filter processing incoming logout messages
	// First argument determines URL user will be redirected to after successful
	// global logout
	@Bean
	public SAMLLogoutProcessingFilter samlLogoutProcessingFilter() {
		return new SAMLLogoutProcessingFilter(successLogoutHandler(), logoutHandler());
	}

	// Overrides default logout processing filter with the one processing SAML
	// messages
	@Bean
	public SAMLLogoutFilter samlLogoutFilter() {
		return new SAMLLogoutFilter(successLogoutHandler(),
				new LogoutHandler[] { logoutHandler() },
				new LogoutHandler[] { logoutHandler() });
	}

	// IDP Discovery service
	@Bean
	public SAMLDiscovery samlIDPDiscovery() {
		SAMLDiscovery samlDiscovery = new SAMLDiscovery();
		samlDiscovery.setIdpSelectionPath("/saml/idpSelection");

		return samlDiscovery;
	}

	// Bindings
	private ArtifactResolutionProfile artifactResolutionProfile() {
		final ArtifactResolutionProfileImpl artifactResolutionProfile = new ArtifactResolutionProfileImpl(
				httpClient());
		artifactResolutionProfile.setProcessor(new SAMLProcessorImpl(soapBinding()));
		return artifactResolutionProfile;
	}

	@Bean
	public HTTPArtifactBinding artifactBinding(ParserPool parserPool, VelocityEngine velocityEngine) {
		return new HTTPArtifactBinding(parserPool, velocityEngine, artifactResolutionProfile());
	}

	@Bean
	public HTTPSOAP11Binding soapBinding() {
		return new HTTPSOAP11Binding(parserPool());
	}

	@Bean
	public HTTPPostBinding httpPostBinding() {
		return new HTTPPostBinding(parserPool(), velocityEngine());
	}

	@Bean
	public HTTPRedirectDeflateBinding httpRedirectDeflateBinding() {
		return new HTTPRedirectDeflateBinding(parserPool());
	}

	@Bean
	public HTTPSOAP11Binding httpSOAP11Binding() {
		return new HTTPSOAP11Binding(parserPool());
	}

	@Bean
	public HTTPPAOS11Binding httpPAOS11Binding() {
		return new HTTPPAOS11Binding(parserPool());
	}

	// Processor
	@Bean
	public SAMLProcessorImpl processor() {
		Collection<SAMLBinding> bindings = new ArrayList<>();
		bindings.add(httpRedirectDeflateBinding());
		bindings.add(httpPostBinding());
		bindings.add(artifactBinding(parserPool(), velocityEngine()));
		bindings.add(httpSOAP11Binding());
		bindings.add(httpPAOS11Binding());

		return new SAMLProcessorImpl(bindings);
	}

	/**
	 * Define the security filter chain in order to support SSO Auth by using SAML 2.0
	 *
	 * @return           Filter chain proxy
	 * @throws Exception
	 */
	@Bean
	public FilterChainProxy samlFilter() throws Exception {
		return new FilterChainProxy(ImmutableList.of(
				new DefaultSecurityFilterChain(
						new AntPathRequestMatcher("/saml/login/**"), samlEntryPoint()),
				new DefaultSecurityFilterChain(
						new AntPathRequestMatcher("/saml/logout/**"), samlLogoutFilter()),
				new DefaultSecurityFilterChain(
						new AntPathRequestMatcher("/saml/metadata/**"), metadataDisplayFilter()),
				new DefaultSecurityFilterChain(
						new AntPathRequestMatcher("/saml/SSO/**"), samlWebSSOProcessingFilter()),
				new DefaultSecurityFilterChain(
						new AntPathRequestMatcher("/saml/SSOHoK/**"), samlWebSSOHoKProcessingFilter()),
				new DefaultSecurityFilterChain(
						new AntPathRequestMatcher("/saml/SingleLogout/**"), samlLogoutProcessingFilter()),
				new DefaultSecurityFilterChain(
						new AntPathRequestMatcher("/saml/discovery/**"), samlIDPDiscovery())));
	}

	/**
	 * Returns the authentication manager currently used by Spring. It represents a bean definition with the
	 * aim allow wiring from other classes performing the Inversion of Control (IoC).
	 *
	 * @throws Exception
	 */
	@Bean
	@Override
	public AuthenticationManager authenticationManagerBean() throws Exception {
		return super.authenticationManagerBean();
	}

	// Register authentication manager for SAML provider
	@Override
	protected void configure(AuthenticationManagerBuilder auth) {
		auth.authenticationProvider(samlAuthenticationProvider());
	}
}
