/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.controller;

import java.io.IOException;
import java.nio.file.Paths;

import javax.transaction.Transactional;
import javax.validation.Valid;

import nl.tudelft.ewi.queue.csv.EmptyCsvException;
import nl.tudelft.ewi.queue.csv.InvalidCsvException;
import nl.tudelft.ewi.queue.dto.create.RoomCreateDTO;
import nl.tudelft.ewi.queue.dto.patch.RoomPatchDTO;
import nl.tudelft.ewi.queue.model.Room;
import nl.tudelft.ewi.queue.repository.FirstYearMentorGroupRepository;
import nl.tudelft.ewi.queue.repository.RoomRepository;
import nl.tudelft.ewi.queue.service.AdminService;

import org.apache.commons.io.FilenameUtils;
import org.modelmapper.ModelMapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

/**
 * Controller for actions only permitted to be done by admins.
 *
 * This entire controller is pre-authorized using the defauls role of admin for a user. If a user has this
 * role, they can access every method in the AdminController.
 *
 * Admins have the right to create mentor group settings and change the available rooms. The page templates
 * associated to these actions are located in the templates/admin directory. The methods (URLs) to get to
 * these pages are all prefixed with {@code /admin}
 */
@Controller
@PreAuthorize("@permissionService.isAdmin(principal)")
@RequestMapping("/admin")
@Validated
public class AdminController {
	private static final String MAPS_DIR = "static/maps";

	@Autowired
	private FirstYearMentorGroupRepository firstYearMentorGroupRepository;

	@Autowired
	private ModelMapper mapper;

	@Autowired
	private RoomRepository roomRepository;

	@Autowired
	private AdminService adminService;

	/**
	 * Adds a model attribute to every template resolution for recognizing the type of page we are in. For
	 * this controller all pages will be in the "admin"-area.
	 *
	 * @return The generic page name: "admin".
	 */
	@ModelAttribute("page")
	public static String page() {
		return "admin";
	}

	/**
	 * @return A redirect to the index page (currently /admin/mentor).
	 */
	@GetMapping()
	public String index() {
		return "redirect:/admin/mentor";
	}

	/**
	 * Gets the page for viewing mentor groups.
	 *
	 * @param  model The model to fill out for template resolution.
	 * @return       The template to load.
	 */
	@GetMapping("/mentor")
	public String mentorGroups(Model model) {
		model.addAttribute("groups", firstYearMentorGroupRepository.findAllByActiveIsTrue());

		return "admin/mentor/groups";
	}

	/**
	 * Processes a POST request for setting the mentor groups for Queue through an input CSV.
	 *
	 * @param  model              The model to fill out for template resolution.
	 * @param  csv                The CSV file to import as a MultipartFile object.
	 * @param  redirectAttributes Attributes that can be filled out on a redirect, for the next method to
	 *                            resolve.
	 * @return                    A redirect to /admin/mentor in the case of success or the original
	 *                            admin/mentor/groups page in case of an error.
	 * @throws IOException        when something went wrong with reading the file outside of a CSV parsing
	 *                            exception.
	 */
	@PostMapping("/mentor/upload")
	public String uploadMentorCsv(Model model,
			@RequestParam("file") MultipartFile csv,
			RedirectAttributes redirectAttributes)
			throws IOException {
		try {
			int created = adminService.importCsv(csv);
			String successMessage = String.format("Imported %d first year students.", created);
			redirectAttributes.addFlashAttribute("message", successMessage);

			return "redirect:/admin/mentor";
		} catch (EmptyCsvException | InvalidCsvException e) {
			model.addAttribute("groups", firstYearMentorGroupRepository.findAllByActiveIsTrue());
			model.addAttribute("csvError", e.getMessage());
		}
		return "admin/mentor/groups";
	}

	/**
	 * Gets the page for viewing the current list of rooms and for navigating to a page to edit a room.
	 *
	 * @param  model The model to fill out for template resolution.
	 * @return       The template to load.
	 */
	@GetMapping("/rooms")
	public String rooms(Model model) {
		model.addAttribute("rooms", roomRepository.findAllByOrderByNameAsc());
		return "admin/rooms";
	}

	/**
	 * Processes a POST request for adding a room to the global list of rooms. This method takes a DTO
	 * containing the information for creating a new room and uses it to create this room.
	 *
	 * @param  dto                The DTO containing the information to create a new room.
	 * @param  redirectAttributes Attributes that can be filled out on a redirect, for the next method to
	 *                            resolve.
	 * @return                    A redirect to the room overview.
	 */
	@Transactional
	@PostMapping("/rooms")
	public String createRoom(@Valid RoomCreateDTO dto, RedirectAttributes redirectAttributes) {
		roomRepository.save(mapper.map(dto, Room.class));

		redirectAttributes.addFlashAttribute("message", "Room has been created.");
		return "redirect:/admin/rooms";
	}

	/**
	 * Gets the page for editing a room with the id given through the path of the /admin/room/edit/{id}
	 * request.
	 *
	 * @param  model The model to fill out for template resolution.
	 * @param  id    The id of the room to edit.
	 * @return       The template to load.
	 */
	@GetMapping("/room/edit/{id}")
	public String editRoom(Model model, @PathVariable("id") Long id) {
		Room room = roomRepository.findByIdOrThrow(id);
		model.addAttribute("room", room);

		return "admin/rooms/edit";
	}

	/**
	 * Processes a POST request to change the information available about a room. This method uses the given
	 * RoomPatchDTO to patch an existing room in the database.
	 *
	 * If the update contains a map file, the map file is added to the directory of all map files, and the
	 * name of the file is saved for the edited room.
	 *
	 * @param  id                 The id of the room to edit.
	 * @param  map                The map file that is to be associated with the room.
	 * @param  dto                The DTO containing the information about the patch.
	 * @param  redirectAttributes Attributes that can be filled out on a redirect, for the next method to
	 *                            resolve.
	 * @return                    A redirect to the room overview.
	 * @throws IOException        when something goes wrong saving the map file.
	 */
	@Transactional
	@PostMapping("/room/edit/{id}")
	public String saveRoom(@PathVariable("id") Long id,
			@RequestParam("map") MultipartFile map,
			RoomPatchDTO dto, RedirectAttributes redirectAttributes)
			throws IOException {
		Room room = roomRepository.findByIdOrThrow(id);
		dto.apply(room);

		if (map != null && !map.getOriginalFilename().isEmpty()) {
			adminService.uploadFile(map);
			room.setMapFilePath(
					Paths.get(MAPS_DIR, FilenameUtils.getName(map.getOriginalFilename())).toString());
		}

		redirectAttributes.addFlashAttribute("message", "Room has been saved.");
		return "redirect:/admin/rooms";
	}

	/**
	 * Processes a request to delete an existing room. The existing room is identified with the given id.
	 *
	 * @param  id                 The id of the room to delete.
	 * @param  redirectAttributes Attributes that can be filled out on a redirect, for the next method to
	 *                            resolve.
	 * @return                    A redirect to the room overview.
	 */
	@PostMapping("/room/delete")
	public String deleteRoom(@RequestParam(value = "id") Long id,
			RedirectAttributes redirectAttributes) {
		Room room = roomRepository.findByIdOrThrow(id);
		roomRepository.delete(room);

		redirectAttributes.addFlashAttribute("message", "Room has been deleted.");
		return "redirect:/admin/rooms";
	}
}
