/*
 * Queue - A Queueing system that can be used to handle labs in higher education
 * Copyright (C) 2016-2020  Delft University of Technology
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package nl.tudelft.ewi.queue.controller;

import java.io.IOException;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.persistence.EntityNotFoundException;
import javax.servlet.http.HttpServletRequest;
import javax.transaction.Transactional;
import javax.validation.Valid;

import nl.tudelft.ewi.queue.annotation.AuthenticatedUser;
import nl.tudelft.ewi.queue.csv.EmptyCsvException;
import nl.tudelft.ewi.queue.csv.InvalidCsvException;
import nl.tudelft.ewi.queue.forms.AssignmentListWrapperForm;
import nl.tudelft.ewi.queue.forms.ParticipantForm;
import nl.tudelft.ewi.queue.model.*;
import nl.tudelft.ewi.queue.repository.*;
import nl.tudelft.ewi.queue.service.CourseService;
import nl.tudelft.ewi.queue.service.GroupService;
import nl.tudelft.ewi.queue.service.LabService;
import nl.tudelft.ewi.queue.service.RequestService;
import nl.tudelft.ewi.queue.validator.ParticipantValidator;
import nl.tudelft.ewi.queue.viewmodel.AssignmentViewModel;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.web.PageableDefault;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.validation.BindingResult;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

import com.querydsl.core.types.dsl.BooleanExpression;

@Controller
@Validated
public class CourseController {

	@Autowired
	private RequestService requestService;

	private static final Logger logger = LoggerFactory.getLogger(CourseController.class);

	@Autowired
	private CourseRepository courseRepository;

	@Autowired
	private RoleRepository roleRepository;

	@Autowired
	private UserRepository userRepository;

	@Autowired
	private CourseService courseService;

	@Autowired
	private GroupService groupService;

	@Autowired
	private LabService labService;

	@Autowired
	private AssignmentRepository assignmentRepository;

	@Autowired
	private ParticipantValidator participantValidator;

	//@Autowired
	//private FirstYearStudentRepository firstYearStudentRepository;

	@ModelAttribute("page")
	public static String page() {
		return "courses";
	}

	@InitBinder("participantForm")
	protected void initBinder(WebDataBinder binder) {
		binder.addValidators(participantValidator);
	}

	@RequestMapping(value = "/courses", method = RequestMethod.GET)
	public String list(@AuthenticatedUser User user, Model model, @PageableDefault(sort = {
			"id" }, direction = Sort.Direction.DESC) Pageable pageable) {
		BooleanExpression archivedAndTeachingCourses = QCourse.course.isArchived.eq(false)
				.or(QCourse.course.in(user.getTeaches()));

		Page<Course> courses = courseRepository.findAll(archivedAndTeachingCourses, pageable);

		model.addAttribute("courses", courses);

		return "course/index";
	}

	@RequestMapping(value = "/course/{id}", method = RequestMethod.GET)
	@PreAuthorize("@permissionService.canViewCourse(principal, #id)")
	public String view(@PathVariable("id") Long id, Model model) {
		Course course = courseRepository.findByIdOrThrow(id);

		model.addAttribute("course", course);

		return "course/view/info";
	}

	@RequestMapping(value = "/course/{id}/leave", method = RequestMethod.GET)
	@PreAuthorize("@permissionService.canLeaveCourse(principal, #id)")
	public String leave(@AuthenticatedUser User user, @PathVariable("id") Long id, Model model) {
		Course course = courseRepository.findByIdOrThrow(id);

		model.addAttribute("course", course);

		return "course/view/leave";
	}

	@RequestMapping(value = "/course/{id}/leave", method = RequestMethod.POST)
	@PreAuthorize("@permissionService.canLeaveCourse(principal, #id)")
	public String leave(@AuthenticatedUser User user, @PathVariable("id") Long id) {
		Course course = courseRepository.findByIdOrThrow(id);

		Optional<Role> role = user.getRole(course);

		role.ifPresent(r -> {
			user.getRoles().remove(r);

			userRepository.save(user);
			roleRepository.delete(r);
		});

		return "redirect:/courses";
	}

	@RequestMapping(value = "/course/{id}/participants", method = RequestMethod.GET)
	@PreAuthorize("@permissionService.canManageParticipants(principal, #id)")
	public String viewParticipants(@PathVariable("id") Long id,
			@RequestParam(value = "staff", required = false) Optional<Boolean> staff,
			@RequestParam(value = "filter", required = false) Optional<String> filter,
			@RequestParam(value = "clean", required = false) Optional<Boolean> clean,
			Model model,
			@PageableDefault(sort = "user.displayName") Pageable pageable) {
		Course course = courseRepository.findByIdOrThrow(id);
		if (staff.isPresent() && staff.get()) {
			List<User> assistants = course.getAssistants();
			model.addAttribute("feedbackCount", courseService.feedbackCount(assistants));
		} else {
			BooleanExpression filterPredicate = filter
					.map(name -> QRole.role.user.username.containsIgnoreCase(name).or(
							QRole.role.user.displayName.containsIgnoreCase(name)))
					.orElse(null);
			BooleanExpression predicate = QRole.role.course.eq(course)
					.and(QRole.role.type.equalsIgnoreCase("student"))
					.and(filterPredicate);

			model.addAttribute("students", roleRepository.findAll(predicate, pageable));
		}

		if (clean.isPresent() && clean.get()) {
			List<User> users = course.getTeachers();
			users.addAll(course.getAssistants());
			users.addAll(course.getManagers());
			int updateCounter = 0;
			for (User user : users) {
				if (user.getSubscription() != null) {
					updateCounter++;
					user.setSubscription(null);
				}
			}
			userRepository.saveAll(users);
			model.addAttribute("cleaned", updateCounter);
		}

		model.addAttribute("staff", staff.orElse(false));
		model.addAttribute("course", courseRepository.findByIdOrThrow(id));

		model.addAttribute("paginationRange", 10);

		return "course/view/participants";
	}

	@RequestMapping(value = "/course/{id}/participants/create", method = RequestMethod.GET)
	@PreAuthorize("@permissionService.canManageParticipants(principal, #id)")
	public String createParticipant(@PathVariable("id") Long id, Model model) {
		setupParticipantForm(id, model);

		return "course/create/participant";
	}

	@RequestMapping(value = "/course/{id}/participants/create", method = RequestMethod.POST)
	@PreAuthorize("@permissionService.canAddParticipant(principal, #id, #participantForm.role)")
	public String storeParticipant(@PathVariable("id") Long id,
			@Valid ParticipantForm participantForm,
			BindingResult bindingResult,
			Model model) {
		if (bindingResult.hasErrors()) {
			model.addAttribute("course", courseRepository.findByIdOrThrow(id));

			return "course/create/participant";
		}

		String username = User.guaranteeValidNetId(participantForm.getUsername());

		Course course = courseRepository.findByIdOrThrow(id);
		course.addRole(getRole(participantForm.getRole(), getUser(username), course));

		courseRepository.save(course);

		return "redirect:/course/" + id + "/participants";
	}

	@RequestMapping(value = "/course/{courseId}/participants/{id}/remove", method = RequestMethod.GET)
	@PreAuthorize("@permissionService.canUpdateParticipant(principal, #courseId, #id)")
	public String deleteParticipant(@PathVariable("courseId") Long courseId, @PathVariable("id") Long id,
			Model model) {
		Role role = getRole(id);

		model.addAttribute("course", courseRepository.findByIdOrThrow(courseId));
		model.addAttribute("role", role);

		return "course/remove/participant";
	}

	@RequestMapping(value = "/course/{courseId}/participants/{id}/remove", method = RequestMethod.POST)
	@PreAuthorize("@permissionService.canUpdateParticipant(principal, #courseId, #id)")
	@Transactional
	public String removeParticipant(@PathVariable("courseId") Long courseId, @PathVariable("id") Long id) {
		Role role = getRole(id);

		roleRepository.delete(role);

		return "redirect:/course/" + courseId + "/participants";
	}

	@RequestMapping(value = "/course/{courseId}/participants/import", method = RequestMethod.POST)
	@PreAuthorize("@permissionService.canManageTeachers(principal, #courseId)")
	public String importParticipants(@PathVariable("courseId") long courseId,
			Model model,
			@RequestParam("file") MultipartFile csv)
			throws IOException {
		Course course = courseRepository.findById(courseId).orElseThrow();
		try {
			courseService.addCourseParticipants(csv, course);
			return "redirect:/course/" + courseId + "/participants";

		} catch (EmptyCsvException | InvalidCsvException e) {
			setupParticipantForm(courseId, model);
			model.addAttribute("csvError", e.getMessage());
		}
		return "course/create/participant";
	}

	private ParticipantForm setupParticipantForm(@PathVariable("courseId") long courseId, Model model) {
		ParticipantForm participantForm = new ParticipantForm();
		participantForm.setCourseId(courseId);

		model.addAttribute("course", courseRepository.findByIdOrThrow(courseId));
		model.addAttribute("participantForm", participantForm);
		return participantForm;
	}

	@RequestMapping(value = "/course/{id}/assignments", method = RequestMethod.GET)
	@PreAuthorize("@permissionService.canManageAssignments(principal, #id)")
	public String viewAssignments(@PathVariable("id") Long id,
			Model model) {
		Course course = courseRepository.findByIdOrThrow(id);

		setupAssignmentForm(model, course);

		return "course/view/assignments";
	}

	@PostMapping(value = "/course/{id}/assignments", params = { "addAssignment" })
	@PreAuthorize("@permissionService.canManageAssignments(principal, #id)")
	public String addAssignment(@PathVariable("id") Long id,
			@ModelAttribute("addAssignment") AssignmentListWrapperForm assignmentList,
			Model model) {
		Course course = courseRepository.findByIdOrThrow(id);

		if (assignmentList == null) {
			assignmentList = new AssignmentListWrapperForm();
		}

		assignmentList.add(new AssignmentViewModel());

		model.addAttribute("course", course);
		model.addAttribute("assignments", assignmentList);
		model.addAttribute("assistants", course.getPrivileged());
		return "course/view/assignments";
	}

	@RequestMapping(value = "/course/{id}/assignments", method = RequestMethod.POST, params = {
			"removeAssignment" })
	@PreAuthorize("@permissionService.canManageAssignments(principal, #id)")
	public String removeAssignment(@PathVariable("id") Long id,
			@RequestParam("removeAssignment") int assignmentID,
			Model model,
			RedirectAttributes redirectAttributes) {
		Course course = courseRepository.findByIdOrThrow(id);
		Long aID = (long) assignmentID;
		Assignment assignment = assignmentRepository.findById(aID).orElseThrow();
		assert assignment.getCourse().getId().equals(course.getId());
		assignmentRepository.deleteById(aID);

		redirectAttributes.addFlashAttribute("message", "Assignment removed.");
		model.addAttribute("course", course);

		return "redirect:/course/" + id + "/assignments";
	}

	@RequestMapping(value = "/course/{id}/assignments", method = RequestMethod.POST, params = {
			"storeAssignments" })
	@PreAuthorize("@permissionService.canManageAssignments(principal, #id)")
	public String storeAssignments(@PathVariable("id") Long id,
			@ModelAttribute("storeAssignments") AssignmentListWrapperForm assignmentList,
			Model model,
			RedirectAttributes redirectAttributes) {
		Course course = courseRepository.findByIdOrThrow(id);

		for (AssignmentViewModel viewAssign : assignmentList.getAssignmentList()) {
			if (!viewAssign.getAssignmentID().isEmpty()) {
				Long assignmentId = Long.parseLong(viewAssign.getAssignmentID());
				Assignment assignment = assignmentRepository.findById(assignmentId).orElseThrow();
				assert assignment.getCourse().getId().equals(course.getId());
				assignment.setName(viewAssign.getName());
				assignment.setAssistants(viewAssign.getAssistants());
				assignmentRepository.save(assignment);
			} else {
				Assignment assignment = viewAssign.convertToNew();
				course.addAssignment(assignment);
			}
		}

		courseRepository.save(course);

		redirectAttributes.addFlashAttribute("message", "Assignments updated.");

		return "redirect:/course/" + id + "/assignments";
	}

	@RequestMapping(value = "/course/{id}/assignments", method = RequestMethod.POST, params = {
			"uploadEnqueue" })
	@PreAuthorize("@permissionService.canManageAssignments(principal, #id)")
	public String uploadEnqueue(@PathVariable("id") Long id,
			@RequestParam("csv") MultipartFile csv,
			@RequestParam("uploadEnqueue") Long assignmentId,
			Model model)
			throws IOException {
		Course course = courseRepository.findByIdOrThrow(id);
		try {

			groupService.importCsv(course, csv);

			List<Group> groups = course.getGroups();

			logger.error("Going to enqueue this many groups: " + groups.size());
			Assignment assignment = assignmentRepository.findById(assignmentId).orElseThrow();

			for (Lab lab : assignment.getLabs()) {
				for (Request req : lab.getPending()) {
					requestService.revoke(req);
				}
			}

			for (Group group : groups) {
				labService.randomEnqueue(assignment, group);
			}

			return "redirect:/course/" + id + "/assignments";
		} catch (EmptyCsvException e) {

			setupAssignmentForm(model, course);
			model.addAttribute("csvError", e.getMessage());
		} catch (InvalidCsvException e) {
			setupAssignmentForm(model, course);
			model.addAttribute("csvError", e.getMessage());
		}
		return "course/view/assignments";
	}

	private void setupAssignmentForm(Model model, Course course) {
		AssignmentListWrapperForm assignmentList = new AssignmentListWrapperForm();
		for (Assignment assignment : course.getAssignments())
			assignmentList.add(AssignmentViewModel.convert(assignment));
		model.addAttribute("course", course);
		model.addAttribute("assignments", assignmentList);
		model.addAttribute("assistants", course.getPrivileged());
	}

	@RequestMapping(value = "/course/{id}/labs", method = RequestMethod.GET)
	@PreAuthorize("@permissionService.canViewCourse(principal, #id)")
	public String viewLabs(@PathVariable("id") Long id, Model model) {
		Course course = courseRepository.findByIdOrThrow(id);

		List<Lab> labs = course.getLabs().stream().sorted(
				Comparator.comparing(l -> l.getSlot().getOpensAt())).collect(Collectors.toList());

		model.addAttribute("course", course);
		model.addAttribute("labs", labs);
		model.addAttribute("occupation", numberOfAvailableSlotsStrings(course));
		if (course.getHasGroups() && course.getGroups().size() == 0) {
			model.addAttribute("alert", "No groups have yet been imported.");
		}

		return "course/view/labs";
	}

	private List<String> numberOfAvailableSlotsStrings(Course course) {
		List<String> retList = new ArrayList<>();
		for (Lab lab : course.getLabs()) {
			if (!lab.isSignOffIntervals()) {
				retList.add("");
			} else {
				long allSlots = labService.getIntervalsForLab(lab).size() * lab.getCapacity();
				long slotsTaken = lab.getRequests().stream().filter(request -> !(request.isRejected()
						|| request.isRevoked()))
						.count();
				String retString = "(" + slotsTaken + "/" + allSlots + " slots taken)";
				retList.add(retString);
			}
		}

		return retList;
	}

	@RequestMapping(value = "/course/{id}/enroll", method = RequestMethod.GET)
	@PreAuthorize("@permissionService.canEnrollCourse(principal, #id)")
	public String enroll(@PathVariable("id") Long id, Model model) {
		Course course = courseRepository.findByIdOrThrow(id);

		model.addAttribute("course", course);

		return "course/enroll";
	}

	@RequestMapping(value = "/course/{id}/enroll", method = RequestMethod.POST)
	@PreAuthorize("@permissionService.canEnrollCourse(principal, #id)")
	public String enroll(@AuthenticatedUser User user, @PathVariable("id") Long id,
			RedirectAttributes redirectAttributes) {
		Course course = courseRepository.findByIdOrThrow(id);

		User student = userRepository.findById(user.getId()).orElseThrow();

		courseService.enroll(student, course);

		redirectAttributes.addFlashAttribute("message", "You are now enrolled.");

		return "redirect:/course/" + course.getId();
	}

	@RequestMapping(value = "/course/{id}/settings", method = RequestMethod.GET)
	@PreAuthorize("@permissionService.canEditCourse(principal, #id)")
	public String settings(@AuthenticatedUser User user, @PathVariable("id") Long id, Model model) {
		Course course = courseRepository.findByIdOrThrow(id);

		model.addAttribute("course", course);

		return "course/view/settings";
	}

	@RequestMapping(value = "/course/{id}/settings", method = RequestMethod.POST)
	@PreAuthorize("@permissionService.canEditCourse(principal, #id)")
	public String update(@AuthenticatedUser User user, @PathVariable("id") Long id, @Valid Course course,
			BindingResult bindingResult) {
		if (bindingResult.hasErrors()) {
			return "course/view/settings";
		}

		courseRepository.save(course);

		return "redirect:/course/" + course.getId();
	}

	@RequestMapping(value = "/course/{id}/remove", method = RequestMethod.GET)
	@PreAuthorize("@permissionService.canRemoveCourse(principal, #id)")
	public String remove(@AuthenticatedUser User user, @PathVariable("id") Long id,
			Model model) {
		Course course = courseRepository.findByIdOrThrow(id);

		model.addAttribute("course", course);

		return "course/view/delete";
	}

	@RequestMapping(value = "/course/{id}/remove", method = RequestMethod.POST)
	@PreAuthorize("@permissionService.canRemoveCourse(principal, #id)")
	public String remove(@AuthenticatedUser User user, @PathVariable("id") Long id) {
		Course course = courseRepository.findByIdOrThrow(id);

		courseRepository.delete(course);

		return "redirect:/courses";
	}

	@RequestMapping(value = "/course/create", method = RequestMethod.GET)
	@PreAuthorize("hasRole('TEACHER')")
	public String create(@AuthenticatedUser User user, Model model) {
		model.addAttribute("course", new Course());

		return "course/create";
	}

	@RequestMapping(value = "/course/create", method = RequestMethod.POST)
	@PreAuthorize("hasRole('TEACHER')")
	public String store(@AuthenticatedUser User user,
			@Valid Course course,
			BindingResult bindingResult,
			@RequestParam("teacherUsername") Optional<String> teacherUsername,
			RedirectAttributes redirectAttributes) {
		if (bindingResult.hasErrors()) {
			return "course/create";
		}
		courseRepository.save(course);

		if (user.isTeacher() && !user.teaches(course)) {
			addTeacherToCourse(user, course);
		} else if (teacherUsername.isPresent() && !teacherUsername.get().isEmpty()) {
			redirectAttributes = addTeacherNetIdToCourse(teacherUsername, course, redirectAttributes);
		}

		return "redirect:/course/" + course.getId();
	}

	@RequestMapping(value = "/course/{id}/status", method = RequestMethod.GET)
	@PreAuthorize("@permissionService.canManageAssignments(principal, #id)")
	public String status(@AuthenticatedUser User user,
			@PathVariable("id") Long id,
			Model model) {
		Course course = courseRepository.findByIdOrThrow(id);
		model.addAttribute("course", course);
		model.addAttribute("labCounter", courseService.requestCountByAssistant(course));
		return "course/view/status";
	}

	/**
	 * Handle {@link AccessDeniedException} when this is thrown in the LabController class.
	 *
	 * @param  user    The user for whom this is exception is thrown.
	 * @param  request The http request which was performed.
	 * @param  e       The exception being thrown.
	 * @return         Returns a redirect to the course's enroll page if the user tries to access a lab.
	 *                 Otherwise if the user tries to create/edit a lab, we still give him the error page.
	 */
	@ExceptionHandler(AccessDeniedException.class)
	public String notInCourse(@AuthenticatedUser User user, HttpServletRequest request,
			Exception e) {
		Pattern testPattern = Pattern.compile("/course/([0-9]+)/*");
		Matcher matcher = testPattern.matcher(request.getRequestURI());
		if (matcher.matches() && matcher.groupCount() == 1) {
			return "redirect:/course/" + matcher.group(1) + "/enroll";
		}
		logger.error("Access Denied: (User: " + user.getDisplayName() + ", page: " + request.getRequestURI()
				+ "): " + e);
		e.printStackTrace();
		return "error/thouShaltNotPass";
	}

	protected static Role getRole(String name, User user, Course course) {
		switch (name) {
			case "student":
				return new Student(user, course, LocalDateTime.now());
			case "assistant":
				return new Assistant(user, course);
			case "manager":
				return new Manager(user, course);
			case "teacher":
				return new Teacher(user, course);
			default:
				throw new IllegalArgumentException("Unknown role name");
		}
	}

	protected Role getRole(Long id) {
		Role role = roleRepository.findById(id).orElseThrow();

		if (null == role) {
			throw new EntityNotFoundException("Role was not found");
		}

		return role;
	}

	protected User getUser(String username) {
		User user = userRepository.findByUsername(username);

		if (null == user) {
			throw new EntityNotFoundException("User was not found");
		}

		return user;
	}

	private RedirectAttributes addTeacherNetIdToCourse(Optional<String> teacherUsername,
			Course course,
			RedirectAttributes redirectAttributes) {
		String teacherNetId = User.guaranteeValidNetId(teacherUsername.get());
		User teacher = userRepository.findByUsername(teacherNetId);
		if (teacher != null) {
			addTeacherToCourse(teacher, course);
		} else {
			redirectAttributes.addFlashAttribute("message",
					"Teacher not added. Could not find teacher with NetID: " + teacherNetId);
		}
		return redirectAttributes;
	}

	private void addTeacherToCourse(User teacher, Course course) {
		Role teacherRole = new Teacher(teacher, course);
		course.addRole(teacherRole);
		roleRepository.save(teacherRole);
	}
}
